var exec = require('child_process').exec,
	Promise = require('bluebird'),
	PromiseObject = require('promise-object')(Promise),
	util = require('util'),
	fs = require('fs'),
	Path = require('path'),
	FTP = require('ftp'),
	prompt = require('prompt'),
	SFTP = require('sftp'),
	async = require('async');

/*
- Connection.disconnect()
- Connection.connect()
- Connection.cd(‘path')
- Connection.ls // will return raw data, and object with proper data array
- Connection.mkdir(‘path’) // will create a full path, even if sub folders don’t exist
- Connection.rmdir(‘folder’, {recursive:true}) // need to set recursive explicit to make sure you know what it does
- Connection.rmfile(‘path’)
- Connection.upload(’source’, ‘dest’) // if the dest folder doest exist it should get created
- Connection.get(‘path’) // download a file
*/

var Proxy = PromiseObject.create({

	_protocol : "FTP",
	_auth: false,
	_rootPath: "",
	_createRootPath: false,
	_host: false,
	_connection: false,
	_isFTP: false,
	_isSFTP: false,

	initialize: function ($self, $config) {
		this._DEBUG = $config.DEBUG;

		if (!$config.host) {
			return console.log("Host is required".red);
		}

		this._host = $config.host;
        this._protocol = $config.protocol || "FTP";
        this._auth = $config.auth;
        this._rootPath = $config.rootPath;
        this._createRootPath = $config.createRootPath;

        // set protocol shorthand
        if (this._protocol.toUpperCase() === "FTP") {
        	this._isFTP = true;
        } else if (this._protocol.toUpperCase() === "SFTP") {
        	this._isSFTP = true;
        }

        this._setupConnection().then(function () {
        	console.log('Connection Successfull'.green);
        	if (typeof $config.ready === "function") {
        		$config.ready();
        	}
        }, function () {
        	$self.disconnect();
        })


    },

    _setupConnection: function ($deferred, $self) {
    	if (!this._auth) {
        	this._passwordPrompt().then($self._createConnection).then($self._travelToRoot).done(function () {
                    $deferred.resolve();
        		}, function (err) {
        			if ($self[err]) return $self[err]();
        		});
        } else {
        	this._createConnection().then($self._travelToRoot)
        		.done(function () {
        			$deferred.resolve();
	        	}, function (err) {
	        		if (err && $self[err]) $self[err]();
	        		$deferred.reject();
	        	});
        }
    },

    _passwordPrompt: function ($deferred, $self) {
    	prompt.get(['username', {name:'password',hidden:true}], function (err, result) {
			if (err) throw err;

			$self._auth = {
				username: result.username,
				password: result.password
			}
			
			$deferred.resolve();
		});
    },

    _travelToRoot: function ($deferred, $self) {
    	
    	if (!this._rootPath) return $deferred.resolve();

	    if (this._createRootPath) {
	    	this.cdOrCreate(this._rootPath).done(function () {               
	    		$deferred.resolve();
	    	},function (err) {
	    		$deferred.reject(err);
	    	});
    	} else {
    		$self.cd(this._rootPath).done(function () {
    			$deferred.resolve();
    		},function (err) {
    			$deferred.reject(err);
    			console.log("please make sure the root folder exists, or use the createRootPath flag".yellow);    			
    		})
    	}
    },

    _createConnection: function ($deferred, $self) {

    	// setup connection with the correct protocol    	
    	if (this._isSFTP) {

    		var connection = new SFTP({host: this._host,username: this._auth.username,password: this._auth.password,password_prompt: "password:", cmdDone:'sftp> sftp>'},function () {});

    		connection.on('connect', function(err) {
		    	if (err) return $deferred.reject(err);

		    	$self._connection = connection;		    	
                // setTimeout(function () {
                    console.log('delayed connection'.green);
                    $deferred.resolve();    
                // },1000);
		    	
			});

    		connection.on('data',function (data) {
                // console.log(data);
    			if (data.match('Permission denied')) {
    				$deferred.reject('permissionDenied');
    			}
    		})

    	} else if (this._isFTP) {

    		var connection = new FTP();

			connection.on('ready', function() {
				$self._connection = connection;
				$deferred.resolve();
			});

			connection.on('error', function (error) {
				if (error && error.code === 530) {
					$deferred.reject("permissionDenied");
				}
			})

			connection.connect({ host: this._host, user: this._auth.username, password: this._auth.password});
    	}
    },

    cdOrCreate: function ($deferred, $self, path) {

    	$self.cd(path).then(function() {
    		$deferred.resolve();
    	},function () {
    		$self.mkdir(path, true).done(function () {
                $self.cd(path).done(function () {
                    $deferred.resolve();
                }, function () {
                    $deferred.reject();        
                })    			
    		}, function () {
    			$deferred.reject();
    		})    		
    	})
    },

    cd: function ($deferred, $self, path) {

    	if (this._isSFTP) {

    		$self._connection.cd(Path.join(path), function (err) {
    			if (err && err.toString().match("No such file or directory")) {    				
    				$deferred.reject("noFile");
    			} else {
    				$deferred.resolve();
    			}
    		});
    	} else if (this._isFTP) {
			$self._connection.cwd(Path.join(path), function (err) {
    			if (err && err.toString().match("No such file or directory")) {    				
    				$deferred.reject("noFile");
    			} else {
    				$deferred.resolve();
    			}
    		})
    	}
    },

    ls: function ($deferred, $self, path) {

    	if (this._isSFTP) {


    		$self._connection.readdir(path,function (err, items) {
    			var resolveItems = [];
                
				// $self._connection.$lsCache = {};
 
				items.forEach(function (item) {

					if (item.path.charAt(0) != ".") {
						resolveItems.push({
							type: item.permissions.charAt(0),
							path: item.path,
							rights: {
								user: item.permissions.substr(1,3).replace(/-/g,""),
								group: item.permissions.substr(4,3).replace(/-/g,""),
								other: item.permissions.substr(7,3).replace(/-/g,"")
							}
						})
					}
				})

				$deferred.resolve(resolveItems);
    		});

    	} else if (this._isFTP) {

    		$self._connection.list(path, function (error, items) {
    			var resolveItems = [];
				
				if (items) {
					items.forEach(function (item) {
						resolveItems.push({
							type: item.type,
							path: item.name,
							rights: item.rights,						
						})
					})
				}

				$deferred.resolve(resolveItems);
			});

    	}
    },

    mkdirs: function ($deferred, $self, dirs, recursive) {
        async.forEachLimit(dirs, 1, function (dir, callback) {

            if (dir && dir.dest) dir = dir.dest;

            $self.mkdir(dir, recursive).done(function () {
                callback();
            }, function (err) {
                $deferred.reject(err)
            })
        }, function () {
            $deferred.resolve();
        });
    },

    mkdir: function ($deferred, $self, path, recursive) {

        // should support array of paths

    	if (this._isSFTP) {
    		if (recursive) {
    			$self._connection.mkdirp(path, function (err) {
    				if (err) return $deferred.reject(err);
    				$deferred.resolve();
    			})
    		} else {
    			$self._connection.mkdir(path, function (err) {
    				if (err) return $deferred.reject(err);
    				$deferred.resolve();
    			})
    		}    		
    	} else if (this._isFTP) {
    		$self._connection.mkdir(path, recursive, function (err) {
                if (err) return $deferred.reject(err);
    			$deferred.resolve();
    		})
    	}
    },

    rmdirs: function () {

    },

    // only allow folder in your current location
    rmdir: function ($deferred, $self, path, recursive) {

        // should support an array of paths
        if (recursive) {
            $self._lswalk(path).done(function (items) {
                
                $self._rmFiles(items.files).done(function () {
                    $self._rmDirs(items.dirs).done(function() {
                         $self._connection.rmdir(path, function (err) {                            
                            if (err && err.code !== 550) console.log('recursive error'.red, err);
                            $deferred.resolve();
                        });
                    })
                })
            })
        } else {
            $self._connection.rmdir(path, function (err) {
                if (err) console.log('rmdir error'.red,err);
                $deferred.resolve();
            });          
        }
    },

    _rmFiles: function ($deferred, $self, files) {

        if (!files || files.length === 0) return $deferred.resolve();

        async.forEachLimit(files.reverse(),1,function (item,callback) {
            $self.rmfile(item).done(function () {
                callback();
            }, function () {
                callback();
            })
        }, function () {
            $deferred.resolve();
        })
    },

    _rmDirs: function ($deferred, $self, dirs) {

        if (!dirs || dirs.length === 0) return $deferred.resolve();

        async.forEachLimit(dirs.reverse(),1,function (item,callback) {
            $self.rmdir(item).done(function () {
                callback();
            }, function () {
                callback();
            })
        }, function () {
            $deferred.resolve();
        })
    },

    rmfile: function($deferred, $self, path) {
        
    	if ($self._isSFTP) {
            $self._connection.unlink(path, function (err) {
                if (err) console.log(err);
                console.log('done deleting'.yellow,path.toString().grey);
                $deferred.resolve();
            });
    	} else if ($self._isFTP) {
            $self._connection.delete(path, function (err) {
                if (err) console.log(err);
                console.log('done deleting'.yellow,path.toString().grey);
                $deferred.resolve();
            });
    	}
    },

    upload: function ($deferred, $self, files) {
    	if ($self._isFTP) {
            async.forEachLimit(files,1,function (file,callback) {
                $self._connection.put(file.source, file.dest, function (err) {
                    console.log('done writing'.green,file.source.toString().grey, '->',file.dest.grey );
                    if (err) return $deferred.reject(err);
                    callback();
                });
            }, function () {
                $deferred.resolve();
            })
    	} else if ($self._isSFTP) {
            async.forEachLimit(files,1,function (file,callback) {
                $self._connection.writeFile(file.dest, fs.readFileSync(file.source), function () {
                    console.log('done writing'.green,file.source.toString().grey, '->',file.dest.grey);
                    callback();
                });
            }, function () {
                $deferred.resolve();
            })
    	}
    },

    disconnect: function ($deferred, $self) {
    	if (this._isSFTP) {
    		this._connection.disconnect();
    	} else if (this._isFTP) {
    		this._connection.end();
    	}
    },


    // ls on ftp needs to include current path 
	_lswalk: function ($deferred, $self, dir) {
		var response = {files: [], dirs: []};

		function walk (dir, callback) {
			var dirs = [];
			$self.ls(dir).done(function (items) {
				
				if (items) {
					items.forEach(function (item) {
						var pushThis = ($self._isFTP) ? Path.join(dir, item.path) : item.path
						if (item.type === 'd') {
							response.dirs.push(pushThis);
							dirs.push(pushThis);
						} else if (item.type === '-') {
							response.files.push(pushThis);
						}
					});
				}


				if (!dirs.length) {
					callback();
				} else {
					async.forEachSeries(
						dirs,
						function (current, callback) {
                            setTimeout(function () {
                                walk(Path.join(current), function () {
                                    callback();
                                });    
                            },100);							
						},
						callback
					);
				}
			});
		}

		walk(dir, function () {			
			$deferred.resolve(response);
		});
	},

	permissionDenied: function ($self) {
		console.log('Username or Password was wrong, please try again'.red);
		$self._auth = false;
		$self._setupConnection();
		return;
	},

	noFile: function ($self) {
		console.log("no such file or directory".red);
		return;
	}

});

module.exports = Proxy;