var exec = require('child_process').exec,
	glob = require('glob'),
	when = require('whenplus'),
	PromiseObject = require('promise-object'),
	fs = require('fs'),
	mkdirp = require('mkdirp'),
	path = require('path'),
	clone = require('clone'),
	Builder = require('builder'),
	malChangeset = require('node-changeset');

var MarcomBuilder = PromiseObject.create({
	initialize: function ($self, $config, callback) {
		this._imagePath = $config.imagePath || './images/';
		this._videoPath = $config.videoPath || './videos/';
		this._scripts = $config.scripts;
		this._tagList = $config.tagList;
		this._vendors = this._prepareVendors($config.vendors);


		this._changeset = new malChangeset({revision:$config.overrideRevision});
		// this._changeset = $config.overrideRevision;



		this._changeset.getData().then(function () {
			$self._builder = new Builder($self._changeset.active.revision);

			mkdirp('output/'+$self._changeset.active.branch+'_r' + $self._changeset.active.revision, function () {
				when.mapUnfulfilled($self._vendors, $self._processVendor).allLimit(1).done(callback, function (error) {console.log(error.stack);});
			});
		});
	},

	_getGit: function ($deferred) {

		exec('git rev-list HEAD --count',function (err,stdout,stderr) {
			if (err) throw new Error('Git/Hg Repo not found, use "--r 1" to set the changeset manually');
			$deferred.resolve(parseInt(stdout)-1);
		});
	},

	_getMercurial: function ($deferred) {

		exec('hg tip',function (err,stdout,stderr) {
			if (err) return $deferred.reject();
			var changeset = stdout.match(/changeset\:\s{3}(\d+)\:/);
			$deferred.resolve(changeset ? changeset[1] : -1);
		});
	},

	_getChangeset: function ($deferred,$self) {

		if (this._changeset) return $deferred.resolve(this._changeset);

		this._getMercurial().then(function (changeset) {
			$deferred.resolve(changeset);
		},function () {
			$self._getGit().then(function (changeset) {
				$deferred.resolve(changeset);
			});
		});
	},

	matchAny: function (list, match) {
		var matched = false;

		list.forEach(function (item) {
			if (match.indexOf(item) !== -1) matched = true;
		});

		return matched;
	},

	matchAll: function (list,match) {
		var checklist = [];
		list.forEach(function (item) {
			if (match.indexOf(item) !== -1) checklist.push(item);
		});

		return checklist.length === list.length;
	},

	_prepareVendors: function ($self, vendors) {
		return vendors.map(function (vendor) {
			vendor.sites = [];

			for (var geo in vendor.geos) {
				for (var site in vendor.geos[geo]) {


					if (vendor.geos[geo][site].name && !vendor.geos[geo][site].size) {
						throw new Error('You need to supply a size when using name property');
					}

					var filepath = 'sites/' + geo + '/' + site,
						name = (vendor.geos[geo][site].name) ? vendor.geos[geo][site].name : site.match(/^((?:[a-z0-9]+\.)??([a-z0-9]*?)\.(?:[a-z0-9]{2}\.)?[a-z0-9]{2,4})$/)[2],
						// scripts = clone($self._scripts),
						media = glob.sync(filepath + '/**/*.+(webm|mp4)'),
						images = glob.sync($self._imagePath + '*'),
						videos = glob.sync($self._videoPath + '*'),
						vendorFiles = glob.sync(path.join(__dirname,'../includes/'+vendor.type+'/**/*')),
						size = (vendor.geos[geo][site].size) ? vendor.geos[geo][site].size : vendor.geos[geo][site],
						identifiers = [size, String(size), geo, name, geo + '_' + name,vendor.type];

						if (vendor.geos[geo][site].tags) identifiers = identifiers.concat(vendor.geos[geo][site].tags);

						if (vendor.tags) identifiers = identifiers.concat(vendor.tags);

						if ($self._tagList && !$self.matchAny($self._tagList,identifiers)) continue;


					var scripts = $self._scripts.map(function (input) {
						var temp = input.files.filter(function (script) {
							var identifier = script.match(/-(.+)\.[a-z]{2}$/i);

							if (!identifier) {
								return true;
							} else {
								identifier = identifier[1];
								var list = identifier.split('-');
								return $self.matchAll(list,identifiers);
							}
						});

						var fileCompression = (typeof input.compression != 'undefined' ? input.compression : true);

						return {name: input.name,files: temp, compression: fileCompression};

					});

					// prepare site scripts
					glob.sync(filepath + '/**/site.js').forEach(function (script) {
						scripts.push({name: 'site', files: [script]});
					});

					// filter image assets based on width, geo, and name identifiers
					images = images.filter(function (image) {
						var identifier = image.match(/-(.+)\.[a-z]{3}$/i);

						if (!identifier) {
							return true;
						} else {
							identifier = identifier[1];
							var list = identifier.split('-');
							return $self.matchAll(list,identifiers);
						}
					});
					// filter image assets based on width, geo, and name identifiers
					videos = videos.filter(function (video) {

						var identifier = video.match(/-(.+)\.[a-z0-9]{3,4}$/i);


						if (!identifier) {
							return true;
						} else {
							identifier = identifier[1];
							var list = identifier.split('-');
							return $self.matchAll(list,identifiers);
						}
					});

					var copy = [];

					// add rmv
					copy.push($self._filePostPreprocessor({name: 'rmv', file: vendor.rmv, source: fs.readFileSync(vendor.rmv, 'utf8')}));


					vendor.sites.push({
						filepath: filepath,
						geo: geo,
						name: name,
						scripts: scripts,
						copy: copy.concat(videos, images, media, vendorFiles),
					});
				}
			}

			return vendor;
		});
	},

	_processVendor: function ($deferred, $self, vendor) {

		if (vendor.sites.length === 0) {
			return $deferred.resolve();
		}

		mkdirp('output/'+this._changeset.active.branch+'_r' + this._changeset.active.revision + '/' + vendor.type, function () {

			if (vendor.scaffolding === 'flat') {
				$self._scaffoldingFlat(vendor).done($deferred.resolve, $deferred.reject);
			} else if (vendor.scaffolding === 'per_site') {
				when.map(vendor.sites, [$self._scaffoldingPerSite, vendor]).done($deferred.resolve, $deferred.reject);
			} else {
				throw new Error('Invalid scaffolding: ' + vendor.scaffolding + ' is not a valid type');
			}
		});
	},

	_scaffoldingFlat: function ($deferred, $self, vendor) {
		var dir = path.join('output/'+this._changeset.active.branch+'_r' + this._changeset.active.revision, vendor.type),
			buildScripts = null,
			copy = [];

		vendor.sites = vendor.sites.map(function (site) {
			site.copy.forEach(function (file) {
				if (copy.indexOf(file) === -1) {
					copy.push(file);
				}
			});

			site.scripts = site.scripts.filter(function (script) {
				if (!buildScripts && script.name === 'build') {
					buildScripts = script;
				}
				return script.name !== 'build';
			});

			return site;
		});

		$self._builder.process([buildScripts], vendor.compression, vendor.exclusions).done(function (scripts) {
			var files = [].concat(copy);

			// handle build.js
			buildScripts = $self._filePostPreprocessor(scripts[0]);
			files.push({file: buildScripts.name + '.js', source: buildScripts.source});

			// handle all other site.js files
			when.map(vendor.sites, [$self._processSite, vendor]).done(function (sites) {
				sites.forEach(function (site) {
					site.scripts.forEach(function (script) {
						files.push({file: site.name + '_' + site.geo + '_' + script.name + '.js', source: script.source});
					});
					files.push({file: site.name + '_' + site.geo + '_index.html', source: $self._generateIndexFile(site, vendor, ['build'])});
				});

				when.map(files, [$self._copyFile, dir]).done($deferred.resolve, $deferred.reject);
			});
		}, $deferred.reject);
	},

	_scaffoldingPerSite: function ($deferred, $self, site, vendor) {
		var dir = path.join('output/'+this._changeset.active.branch+'_r' + this._changeset.active.revision, vendor.type, site.geo + '_' + site.name);

		mkdirp(dir, function () {
			$self._processSite(site, vendor).done(function (site) {
				var files = [].concat(site.copy);

				site.scripts.forEach(function (script) {
					if (vendor.type === 'youtube' && script.name === 'site') {
						files.push({file: 'youtube_site.js', source: script.source});
					} else {
						files.push({file: (script.name !== 'build' ? site.name + '_' + site.geo + '_' : '') + script.name + '.js', source: script.source});
					}
				});

				files.push({file: site.name + '_' + site.geo + '_index.html', source: $self._generateIndexFile(site, vendor)});

				when.map(files, [$self._copyFile, dir]).done($deferred.resolve, $deferred.reject);
			});
		});
	},

	_generateIndexFile: function (site, vendor, additional) {
		var html = '';

		if (vendor.type == 'youtube') {
			return '<!DOCTYPE html><html><head><meta charset="UTF-8"><title>Demo</title><script src="/879366/Enabler_01_49.js"></script><script src="https://www.gstatic.com/external_hosted/polymer/custom.elements.min.js"></script><script src="https://www.gstatic.com/ads/ci/ytclosebutton/1/ytclosebutton_min.js"></script></script><link rel="stylesheet" type="text/css" href="DCRM_HTML5_Inpage_YTCloseButton_970x250.css"><script src="DCRM_HTML5_Inpage_YTCloseButton_970x250.js" type="text/javascript"></script></head><body><div id="ad-container_dc"><div id="mal-ad-container"><div id="mal-ad-container-last"></div></div><ci-ytclosebutton lang="en" theme="gray" shadow="true" id="ytClose_dc" /></div></body></html>';
		} else {
			site.scripts.forEach(function (script) {
				html += '<script type="text/javascript" src="' + (script.name !== 'build' ? site.name + '_' + site.geo + '_' : '') + script.name + '.js' + '"></script>';
			});

			if (additional) {
				additional.forEach(function (script) {
					html += '<script type="text/javascript" src="' + script + '.js' + '"></script>';
				});
			}

			html += '<script type="text/javascript" src="' + path.basename(vendor.rmv) + '"></script>';

			return '<!DOCTYPE html><html><head><title>Demo</title><meta http-equiv="X-UA-Compatible" content="IE=9" /><script type="text/javascript">var MALPaths = {images: "./", videos: "./", scripts: "./"};</script></script></head><body><div id=\'mal-ad-container\'></div>' + html + '</body></html>';
		}
	},

	_processSite: function ($deferred, $self, site, vendor) {
		var self = this;

		console.log((vendor.type + ' building: ').grey + site.geo + '_' + site.name);

		var nonPackedScripts = [],
			packedScripts = [],
			resolved = false;

		site.scripts.forEach(function(script) {
			if(script.compression === false) nonPackedScripts.push(script);
			else packedScripts.push(script);
		});

		this._builder.process(packedScripts, vendor.compression, vendor.exclusions).done(function (scripts) {
		 	packedScripts = scripts.map($self._filePostPreprocessor);
		 	if(!nonPackedScripts.length || resolved) $deferred.resolve(site);
		 	else resolved = true;
		});

		if(nonPackedScripts.length) {
			this._builder.process(nonPackedScripts, false, vendor.exclusions).done(function (scripts) {
			 	nonPackedScripts = scripts.map($self._filePostPreprocessor);
				if(resolved) $deferred.resolve(site);
				else resolved = true;
			});
		}
	},

	_filePostPreprocessor: function (file) {

		if (file.name === 'site') {
			file.source = file.source.replace('*/', '*/\n\nMALSiteChangeset = \'' + this._changeset.active.revision + '\';');
		} else if (file.name === 'build') {
			file.source = file.source.replace('*/', '*/\n\nMALChangeset = \'' + this._changeset.active.revision + '\';');
		}

		return file;
	},

	_copyFile: function ($deferred, $self, source, destination) {
		if (typeof source !== 'string') {
			fs.writeFile(path.join(destination, path.basename(source.file)), source.source, function () {
				$deferred.resolve();
			});
			return;
		}

		var readStream = fs.createReadStream(source);
		readStream.on('error', $deferred.reject);

		var writeStream = fs.createWriteStream(path.join(destination, path.basename(source)));
		writeStream.on('error', $deferred.reject);
		writeStream.on('close', $deferred.resolve);

		readStream.pipe(writeStream);
	}
});

module.exports = function(grunt) {
	grunt.registerTask('marcom-build', 'Builds marcom project for multiple sites', function() {
		var callback = this.async(),
			config = grunt.config('marcom');
			config.overrideRevision = grunt.option('r');

			if (grunt.option('tags')) {
				config.tagList = grunt.option('tags').split(',');
			}

		new MarcomBuilder(config, callback);
	});
};