/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * RawTween class. Provides a basic method for number tweening.
 *
 * # Usage
 *
 *  example of tweening some numbers
 *
 *     var tweens = new MAL.RawTween();
 *
 *     tweens.addTween({
 *     delay: 1,
 *      duration: 1,
 *		start: {a: 0, b: 50},
 *		finish: {a: 100, b: -1200},
 *		easing: [0, 0, 1, 1],
 *		callbacks: {
 *		start: function (tween, current) {console.log('start')},
 *		update: function (tween, current) {console.log('update', current.a, current.b)},
 *		complete: function (tween, current) {console.log('complete')}
 *		}
 *     });
 *
 *     tweens.start();
 *
 * @class MAL.RawTween
 */
MAL.define('RawTween', MAL.Util.create({
	/**
	 * initializes MAL.RawTween
	 *
	 *     var rawTweens = new MAL.RawTween({fps: 60});
	 *
	 * @method constructor
	 * @param {Object} $config
	 */

	/**
	 * @cfg {Number} fps
	 * fps for all tweens
	 */
	initialize: function ($config) {
		$config = $config || {};
		this.fps = $config.fps || 60;
		this.timeout = null;
		this.tweens = [];
	},

	/**
	 * Starts the raw tween loop
	 *
	 *     rawTweens.start()
	 */
	start: function () {
		this.startTime = new Date().getTime();
		this._loop();
	},

	/**
	 * animation loop
	 *
	 * @private
	 */
	_loop: function () {
		clearTimeout(this.timeout);

		var time = new Date().getTime();
		for (var i = 0; i < this.tweens.length; i++)
			this._update(this.tweens[i], time);

		this.timeout = setTimeout(this._loop.bind(this), 1000 / this.fps);
	},

	/**
	 * update tween with time
	 *
	 * @private
	 */
	_update: function (tween, time) {
		if (tween.startTime >= time) return;

		if (!tween.started) {
			tween.started = true;
			if (tween.callbacks.start) tween.callbacks.start(tween);
		}

		var finished = time > (tween.startTime + tween.duration),
			percent = (time - tween.startTime) / tween.duration;

		if (MAL.isArray(tween.easing)) {
			percent = MAL.cubicBezierWithPercent(percent, tween.easing[0], tween.easing[1], tween.easing[2], tween.easing[3]);
		} else {
			percent = tween.easing(percent);
		}

		if (percent > 1) percent = 1;

		tween.percent = percent;

		for (var i in tween.values.finish) {
			if (finished) {
				tween.values.current[i] = tween.values.finish[i];
			} else {
				tween.values.current[i] = tween.values.start[i] + (tween.values.delta[i] * percent);
			}
		}

		if (tween.callbacks.update) tween.callbacks.update(tween, tween.values.current);
		if (finished && tween.callbacks.complete) tween.callbacks.complete(tween, tween.values.current);
		if (finished) this.tweens.splice(this.tweens.indexOf(tween), 1);
	},

	/**
	 * Add a raw tween
	 *
	 *     tweens.addTween({
	 *			delay: 1,
	 *			duration: 1,
	 *			start: {a: 0, b: 50},
	 *			finish: {a: 100, b: -1200},
	 *			easing: [0, 0, 1, 1],
	 *			callbacks: {
	 *			start: function (tween, current) {console.log('start')},
	 *			update: function (tween, current) {console.log('update', current.a, current.b)},
	 *			complete: function (tween, current) {console.log('complete')}
	 *		}
	 *     });
	 *
	 * @param {Object} tween
	 */
	addTween: function ($tween) {
		var tween = {};
		tween.started = false;
		tween.finished = false;
		tween.obj = $tween.obj;
		tween.delay = $tween.delay * 1000 || 0;
		tween.duration = $tween.duration * 1000 || 1000;
		tween.startTime = new Date().getTime() + tween.delay;
		tween.easing = $tween.easing || [0.25, 0.1, 0.25, 1];
		tween.values = {
			start: $tween.start || {},
			delta: {},
			current: {},
			finish: $tween.finish
		};
		tween.callbacks = {
			start: $tween.callbacks.start || null,
			update: $tween.callbacks.update || null,
			complete: $tween.callbacks.complete || null
		};

		// if start value is not supplied set it to zero
		for (var i in tween.values.finish) {
			if (!tween.values.start[i]) tween.values.start[i] = 0;
			tween.values.delta[i] = tween.values.finish[i] - tween.values.start[i];
		}

		this.tweens.push(tween);
	}
}));