/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Stylesheets class. Provides an easy way to interface with stylesheets in a cross browser way, and also tracks images. **Anything related to transformations or animated filters should be done through {@link MAL.tween.CSSLayer} or {@link MAL.tween.JSLayer}**.
 *
 * # Usage
 *
 * **NOTE: default functionality of the stylesheet is to prefix all ID's and classes with mal-ad-, in order to override this convenience you have to use {@link #addRawRules}**
 *
 * create a stylesheet object that manages all ad related styles
 *
 *     var stylesheet = new MAL.Stylesheet({
 *      callback: function(){
 *          // stylesheet ready
 *      }
 *     });
 *
 * add some basic rules
 *
 *     stylesheet.addRules({
 *      '#Box': {
 *       width: 50,
 *       height: 50,
 *       left: 0,
 *       top: 0,
 *       position: 'absolute'
 *      },
 *      '#Box2': {
 *       width: 50,
 *       height: 50,
 *       left: 50,
 *       top: 50,
 *       position: 'absolute'
 *      }
 *     });
 *
 * add some rules with defaults
 *
 *     stylesheet.addRulesWithDefaults(
 *      // defaults
 *      {
 *       left: 0,
 *       top: 0,
 *       position: 'absolute'
 *      },
 *
 *      // rules
 *      {
 *       // because of the defaults this rule will actually also include {left:0,top:0,position:'absolute'}
 *       '#Box': {},
 *
 *       // because of the defaults this rule will actually also include {left:0,top:0}
 *       '#Box2': {
 *        left: 50,
 *        top: 50
 *       }
 *      }
 *     );
 *
 * write css rules to the stylesheets
 *
 *     stylesheet.writeStyles()
 *
 * get images that have been used in the rules
 *
 *     var images = stylesheet.getImages();
 *
 * @class MAL.Stylesheet
 * @extends MAL.Object
 */
MAL.define('Stylesheet', MAL.Object.extend({
	/**
	 * initializes MAL.Stylesheet
	 *
	 *     var stylesheet = new MAL.Stylesheet({
	 *      callback: function(){
	 *          // stylesheet ready
	 *      }
	 *     });
	 *
	 * @method constructor
	 * @param {Object} config
	 */

	/**
	* @cfg {String} callback
	* callback called when DOM is ready
	*/
	initialize: function ($config) {
		this._super();

		// check if we already have existing styleSheets on the page
		this._element = window.document.createElement('style');

		// append our stylesheet to the head
		window.document.getElementsByTagName('head')[0].appendChild(this._element);

		// basic properties
		this._ready = false;
		this._styles = [];
		this._images = new MAL.Set();

		// tracks instances
		this._views = new MAL.Set();

		// tracks view classes
		this._viewSignatures = new MAL.Set();

		this._viewsToWrite = [];

		if ($config.callback) this._checkCSS($config.callback);
	},

	/**
	 * make sure that the stylesheet object is ready to be written to
	 *
	 * @private
	 */
	_checkCSS: function (callback) {
		var style = this._getStylesheet(),
			self = this;

		// check if the stylesheet is processed and ready
		try {
			if (style && style.rules && !MAL.isUndefined(style.rules.length)) this._ready = true;
			if (style && style.cssRules && !MAL.isUndefined(style.cssRules.length)) this._ready = true;
			if (MAL.Environment.browser === 'safari' && style && style.cssRules === null) this._ready = true;
		} catch (e) {
			// TODO: figure out a way to avoid using this... for some reason firefox thinks its hacking
			//console.log(e.message);
		}

		// write html if we are really ready
		if (this._ready) {
			if (callback) window.setTimeout(callback.bind(this), 0);
			return;
		}

		window.setTimeout(function () {
			self._checkCSS(callback);
		}, 0);
	},

	/**
	 * obtain the actual stylesheet object that matches the element we made
	 *
	 * @private
	 */
	_getStylesheet: function () {
		var stylesheet = false;

		for (var i = 0; i < window.document.styleSheets.length; i++) {
			if ((window.document.styleSheets[i].ownerNode || window.document.styleSheets[i].owningElement) === this._element)	{
				stylesheet = window.document.styleSheets[i];
				break;
			}
		}

		return stylesheet;
	},

	/**
	 * writes rules to documents style sheet
	 *
	 * @param {Object} rules
	 * @param {Object} animations
	 */
	writeRules: function (rules, animations) {
		// obtain the real stylesheet
		var stylesheet = this._getStylesheet(),
			stylesheetText = '';

		// HACK: stylesheetText is a hack so we can use the chrome development tools (when this is fixed we can remove this)
		for (var selector in rules) {
			var propertiesString = '',
				rule = rules[selector];

			for (var property in rule) if (rule[property])
				propertiesString += rule[property].name + ':' + rule[property].value + ';';

			if (MAL.Environment.browser === 'chrome') {
				stylesheetText += selector + ' {' + propertiesString + '}\n';
			} else if (MAL.Environment.browser === 'safari' && !MAL.isUndefined(stylesheet.addRule)) {
				stylesheet.addRule(selector, propertiesString);
			} else if (!MAL.isUndefined(stylesheet.insertRule)) {
				stylesheet.insertRule(selector + ' {' + propertiesString + '}', stylesheet.cssRules.length);
			} else {
				// in IE we can't use comma seperated selectors... so we need to just keep adding rules
				var selectors = selector.split(',');

				for (var i = 0; i < selectors.length; i++) {
					stylesheet.addRule(selectors[i], propertiesString, stylesheet.rules.length);
				}
			}
		}

		if (MAL.Environment.getCSSPropertyName('animation-name')) {
			for (var animation in animations) {
				if (MAL.Environment.browser === 'chrome') {
					stylesheetText += animation + ' {' + animations[animation] + '}';
				} else if (MAL.Environment.browser === 'safari' && !MAL.isUndefined(stylesheet.addRule)) {
					stylesheet.addRule(animation, animations[animation]);
				} else if (!MAL.isUndefined(stylesheet.insertRule)) {
					stylesheet.insertRule(animation + ' {' + animations[animation] + '}', stylesheet.cssRules.length);
				}
			}
		}

		if (stylesheetText) this._element.innerHTML += stylesheetText;
	},

	/**
	 * adds views to stylesheet
	 *
	 * @param {Array...} views
	 */
	addViews: function () {
		var views = MAL.argumentsToArray(arguments);
		views.forEach(this.addView);
	},

	/**
	 * adds view to stylesheet
	 *
	 * @param  {MAL.View/MAL.ViewController} view
	 */
	addView: function (view) {
		if (view instanceof MAL.ViewController) {
			view = view.view;
		}

		if (this._views.has(view)) return;
		this._views.add(view);
		this._viewsToWrite.push(view);
	},

	/**
	 * adds a set of rules to the rules array
	 *
	 *     stylesheet.addRules({
	 *      '#Box': {
	 *       width: 50,
	 *       height: 50,
	 *       left: 0,
	 *       top: 0,
	 *       position: 'absolute'
	 *      },
	 *      '#Box2': {
	 *       width: 50,
	 *       height: 50,
	 *       left: 50,
	 *       top: 50,
	 *       position: 'absolute'
	 *      }
	 *     });
	 *
	 * @param {Object} rules
	 */
	addRules: function (rules) {
		this.addStyle(new MAL.Style(rules));
	},

	/**
	 * adds a set of rules with default rules to the rules array
	 *
	 *     stylesheet.addRulesWithDefaults(
	 *      // defaults
	 *      {
	 *       left: 0,
	 *       top: 0,
	 *       position: 'absolute'
	 *      },
	 *
	 *      // rules
	 *      {
	 *       // because of the defaults this rule will actually also include {left:0,top:0,position:'absolute'}
	 *       '#Box': {},
	 *
	 *       // because of the defaults this rule will actually also include {left:0,top:0}
	 *       '#Box2': {
	 *        left: 50,
	 *        top: 50
	 *       }
	 *      }
	 *     );
	 *
	 * @param {Object} default rules with default (first argument is default)
	 * @param {Object} rules everything else is a rule
	 */
	addRulesWithDefaults: function (defaults, rules) {
		this.addStyle(new MAL.Style(rules, {defaults: defaults}));
	},

	/**
	 * adds a set of raw rules to the rules object (**does not prefix classes or IDs with mal-ad-**)
	 *
	 *     stylesheet.addRawRules({
	 *      '#Box': {
	 *        width: 50,
	 *        height: 50,
	 *        left: 0,
	 *        top: 0,
	 *        position: 'absolute'
	 *      },
	 *      '#Box2': {
	 *        width: 50,
	 *        height: 50,
	 *        left: 50,
	 *        top: 50,
	 *        position: 'absolute'
	 *      }
	 *     });
	 *
	 * @param {Object} rules
	 */
	addRawRules: function (rules) {
		this.addStyle(new MAL.Style(rules, {raw: true}));
	},

	/**
	 * add a style object to the write cue
	 *
	 *     stylesheet.addStyle(
	 *          new MAL.Style({
	 *              '.button': {
	 *                  width: 200,
	 *                  height: 200
	 *               }
	 *          })
	 *     );
	 *
	 * @param {Object} rules
	 */
	addStyle: function () {
		var styles = MAL.argumentsToArray(arguments);
		styles.forEach(function (style) {
			this._styles.push(style);
		}, this);
	},

	/**
	 * writes styles to stylesheet
	 */
	writeStyles: function () {
		var views = this._viewsToWrite,
			view,
			rules = {},
			animations = {};

		while ((view = views.shift())) {
			var viewStyles = view.getStyles();

			viewStyles.forEach(function (viewStyle) {
				var viewSignature = viewStyle.view.objectClassPath.join('.'),
					additionalRules = {};

				if (!this._viewSignatures.has(viewSignature)) {
					this._viewSignatures.add(viewSignature);
					this._images.add(viewStyle.getImages());

					additionalRules = MAL.mergeOptions(additionalRules, viewStyle.rules);
					animations = MAL.mergeOptions(animations, viewStyle.animations);
				}

				additionalRules = MAL.mergeOptions(additionalRules, viewStyle.instanceRules);

				for (var rule in additionalRules) {
					rules = MAL.mergeOptions(rules, additionalRules);
					break;
				}
			}, this);
		}

		this.writeRules(rules, animations);

		var style;

		while((style = this._styles.shift())) {
			this._images.add(style.getImages());
			this.writeRules(style.rules, style.animations);
		}
	},


	/**
	 * returns the images that have been used in the stylesheet
	 *
	 *     var images = stylesheet.getImages();
	 *
	 * @return {Array} images
	 */
	getImages: function () {
		return this._images.clone();
	}
}));