/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * TimeoutManager class. Provides a way to ensure that timeouts with the same timeout run grouped. **This should not be used for timeouts that need to possibly be cleared.**
 *
 * # Usage
 *
 *     // create a timeout manager (usually you only want one of these ever running)
 *     var timeouts = new MAL.TimeoutManager();
 *
 *     // add a timeout
 *     timeouts.addTimeout(function () {
 *
 *     }, 1000);
 *
 * @class MAL.TimeoutManager
 */
MAL.define('TimeoutManager', MAL.Util.create({
	/**
	 * initializes MAL.TimeoutManager
	 *
	 *     // create a timeout manager (usually you only want one of these ever running)
	 *     var timeouts = new MAL.TimeoutManager();
	 *
	 * @method constructor
	 * @param {Object} config
	 */

	/**
	* @cfg {Number} leniency
	* event loop delay in milliseconds
	*/
	initialize: function ($config) {
		// bind all event based methods
		MAL.bindAll(this, '_loop');

		// optional config object
		$config = $config || {};

		this.queue = {};
		this._loop();
		this.leniency = $config.leniency || 0;
	},

	/**
	 * internal loop
	 * @private
	 */
	_loop: function () {
		for (var item in this.queue) {
			var tasks = this.queue[item];

			window.setTimeout((function (tasks) {
				return function () {
					for (var task = 0; task < tasks.length; task++) tasks[task]();
				};
			}(tasks)), item);

			delete this.queue[item];
		}

		window.setTimeout(this._loop, this.leniency);
	},

	/**
	 * add a timeout similar to window.setTimeout but managed
	 *
	 *     // wait 1000 ms then fire callback
	 *     timeouts.addTimeout(function () {
	 *
	 *     }, 1000);
	 *
	 *     // you can also bind a scope
	 *     timeouts.addTimeout((function () {
	 *
	 *     }).bind(this), 1000);
	 *
	 * @param {Function} callback
	 * @param {Number} delay
	 */
	addTimeout: function (callback, delay) {
		if (!this.queue[delay]) this.queue[delay] = [];
		this.queue[delay].push(callback);
	}
}));