/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * ViewController class is made to be extended upon and should never be directly initialized
 *
 * ## example
 *
 *    var viewController = MAL.ViewController.extend({
 *      intialize: function(){
 *         this._super();
 *      }
 *    });
 *
 * @class MAL.ViewController
 * @extends MAL.Object
 */
MAL.define('ViewController', MAL.Object.extend({
	/**
	 * initializes MAL.ViewController
	 *
	 * @method constructor
	 * @param {Object} config
	 */
	initialize: function ($config) {
		$config = $config || {};

		this._super($config);
		this._autoInitializeView($config.layerType);

		this._stateListeners = {};
	},

	_autoInitializeView: function (layerType) {
		var View = MAL.get('views.' + this.objectClassPath.slice(1).join('.'));

		if (View) {
			this.view = new View({controller: this, layerType: layerType});
		}
	},

	/**
	 * transitioning state var
	 * @type {Boolean}
	 */
	transitioning: false,

	/**
	 * enabled state var
	 * @type {Boolean}
	 */
	enabled: true,

	/**
	 * provides a buffer that ensures that *transitioning=false* and *enabled=true*
	 *
	 * @param  {Function} callback
	 * @return {Function} callback
	 */
	interactionCheck: function (callback) {
		var self = this;

		return function (event) {
			if(event && event.stopPropagation){
				event.stopPropagation();
			}
			if (!self.transitioning && self.enabled) {
				callback.apply(self, arguments);
			}
		};
	},

	/**
	 * enable/disable all state logging
	 */
	logStates: false,

	/**
	 * array of states to not log
	 */
	logStatesIgnore: [],

	/**
	 * add a state listener to the view controller
	 *
	 *     this.addStateListener('poking', onPokeToggle);
	 *
	 * @param {String} name
	 * @param {Function} listener
	 */
	addStateListener: function (name, listener) {
		if (!MAL.isFunction(listener)) throw new Error ('the listener you tried to add for "' + (this.objectClassPathToken + ':' + type) + '" is not a function');

		if (MAL.isUndefined(this._stateListeners[name])) {
			this._stateListeners[name] = [];
		}

		this._stateListeners[name].push(listener);
	},

	/**
	 * remove a state listener from the view controller.
	 *
	 *     // you must pass in the original function (bind makes this a little less ideal)
	 *     this.removeStateListener('poking', onPokeToggle);
	 *
	 * @param {String} name
	 * @param {Function} listener
	 */
	removeStateListener: function (name, listener) {
		if (this._stateListeners[name] instanceof Array){
			var listeners = this._stateListeners[name];

			for (var i=0, l=listeners.length; i < l; i++){
				if (listeners[i] === listener){
					listeners.splice(i, 1);
					break;
				}
			}
		}
	},

	/**
	 * @private
	 */
	_callStateListeners: function(name, active, _internalCall) {
		var listeners = this._stateListeners[name];

		if (this.logging && this.logStates && this.logStatesIgnore.indexOf(name) === -1) {
			if (MAL.Environment.browser === 'chrome') {
				console.log('%c' + this.getFileAndLineNumberAtStackIndex(_internalCall ? 4 : 3) + '%c state %c' + name + '%c ' + (active ? 'added to' : 'removed from') + ' %c' + this.objectClassPathToken, 'font-weight: bold; color: #1795de;', 'color: #000', 'font-weight: bold', '', 'font-weight: bold');
			} else {
				MAL.log('StateChange ' + this.objectClassPathToken + (active ? ' added ' : ' removed ') + name);
			}
		}

		if (MAL.isArray(listeners)) {
			listeners.forEach(function (listener) {
				listener(active);
			});
		}
	},

	/**
	 * toggles a view controllers state
	 *
	 *     this.toggleState('poking');
	 *
	 * @param {String} name
	 */
	toggleState: function (name) {
		if (this.hasState(name)) {
			this.removeState(name, true);
		} else {
			this.addState(name, true);
		}
	},

	/**
	 * adds the state to the view controller
	 *
	 *     this.addState('poking');
	 *
	 * @param {String} name
	 */
	addState: function (name, _internalCall) {
		if (!this.hasState(name)) {
			this.view.addClass('state-' + name);
			this._callStateListeners(name, true, _internalCall);
		}
	},

	/**
	 * removes the state from the view controller
	 *
	 *     this.removeState('poking');
	 *
	 * @param {String} name
	 */
	removeState: function (name, _internalCall) {
		if (this.hasState(name)) {
			this.view.removeClass('state-' + name);
			this._callStateListeners(name, false, _internalCall);
		}
	},

	/**
	 * tells you if the view controller has the state
	 *
	 *     this.hasState('poking');
	 *
	 * @param {String} name
	 * @return {Boolean} active
	 */
	hasState: function (name) {
		if (name.match(/^mal-/)) {
			if (MAL.toArray(document.querySelectorAll('.' + name + ' .mal-view.mal-' + this.view.name)).indexOf(this.view.element) !== -1) {
				return true;
			} else {
				return false;
			}
		} else {
			return this.view.hasClass('state-' + name);
		}
	}
}));