/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * basics for any layer (should only be extended upon)
 *
 * @class MAL.tween.Layer
 * @extends MAL.Object
 * @private
 */
MAL.define('tween.Layer', MAL.Object.extend({
	/**
	 * initializes MAL.tween.Layer
	 *
	 * @method constructor
	 * @param {Object} config
	 */
	initalize: function ($config) {
		this._super();
	},

	/**
	 * set the layers data
	 *
	 * @param  {Object} data
	 */
	setData: function (data) {
		if (!this._data) this._data = {};

		for (var property in data) {
			this._data[property] = data[property];
		}

		return this;
	},

	/**
	 * get the layers data
	 *
	 * @param  {String} name
	 * @return {String/Object/Number} data
	 */
	getData: function (name) {
		if (!this._data) this._data = {};

		if (name) {
			return this._data[name];
		} else {
			return this._data;
		}
	},

	/**
	 * add an object of events to the layers element
	 *
	 *     layer.addEvents({
	 *      click: function (e) {},
	 *      mouseover: function (e) {}
	 *     });
	 *
	 * @param {Object} events
	 */
	addEvents: function (events) {
		for (var type in events)
			this.addEvent(type, events[type]);
	},

	/**
	 * add a event to the layers element
	 *
	 *     layer.addEvent('click', function (e) {
	 *     }};
	 *
	 * @param {String} type event name
	 * @param {Function} callback
	 */
	addEvent: function (type, callback) {
		MAL.addEvent(this.element, type, callback);
	},

	/**
	 * returns a RegExp that correctly matches for a className
	 *
	 * @private
	 * @param {String} className
	 */
	_classRegExp: function (className){
		return new RegExp("(^|\\s)" + className + "(\\s|$)");
	},

	/**
	 * checks if the layer has the requested class
	 *
	 *     if (layer.hasClass('Selected')) {
	 *      // layer has the class
	 *     }
	 *
	 * @param {String} className
	 * @param {String} prefix optional prefix override
	 */
	hasClass: function (className, prefix) {
		prefix = MAL.isUndefined(prefix) ? 'mal-layer-' : prefix;
		return this.element.classList ? this.element.classList.contains(prefix + className) : this._classRegExp(prefix + className).test(this.element.className);
	},

	/**
	 * add a set of classes to the layer
	 *
	 *     layer.addClass(['Box', 'Selected']);
	 *
	 * or add one
	 *
	 *     layer.addClass('Selected');
	 *
	 * @param {String/Array} classNames
	 * @param {String} prefix optional prefix override
	 */
	addClass: function (classNames, prefix) {
		classNames = MAL.isArray(classNames) ? classNames : classNames.split(' ');
		prefix = MAL.isUndefined(prefix) ? 'mal-layer-' : prefix;

		var i, l;

		// use native method if supported
		if (this.element.classList) {
			for (i = 0; i < classNames.length; i++) this.element.classList.add(prefix + classNames[i]);
			return;
		}

		// use manual method because its not native
		var classArray = this.element.className.split(' '),
			changed = false;

		for (i = 0, l = classNames.length; i < l; i++) {
			if (this.hasClass(classNames[i], arguments[1])) continue;
			classArray.push(prefix + classNames[i]);
			changed = true;
		}

		if (changed) this.element.className = MAL.trim(classArray.join(' '));

		return this;
	},

	/**
	 * remove set of classes from the layer
	 *
	 *     layer.removeClass(['Box', 'Selected']);
	 *
	 * or remove one
	 *
	 *     layer.removeClass('Selected');
	 *
	 * @param {String/Array} classNames
	 * @param {String} prefix optional prefix override
	 */
	removeClass: function (classNames, prefix) {
		classNames = MAL.isArray(classNames) ? classNames : classNames.split(' ');
		prefix = MAL.isUndefined(prefix) ? 'mal-layer-' : prefix;

		var i, l;

		// use native method if supported
		if (this.element.classList) {
			for (i = 0; i < classNames.length; i++) this.element.classList.remove(prefix + classNames[i]);
			return this;
		}

		// use manual method because its not native
		var classArray = this.element.className.split(' '),
			changed = false;

		for (i = 0, l = classNames.length; i < l; i++ ) {
			if (!this.hasClass(classNames[i], arguments[1])) continue;
			classArray.splice(classArray.indexOf(prefix + classNames[i]), 1);
			changed = true;
		}

		if (changed) this.element.className = MAL.trim(classArray.join(' '));

		return this;
	},

	/**
	 * overridden left setter
	 *
	 * @private
	 */
	_cssSetLeft: function (value) {
		this._initialProperties.left = value;

		return {
			name: 'left',
			value: value + this._suffixes.left
		};
	},

	/**
	 * overridden top setter
	 *
	 * @private
	 */
	_cssSetTop: function (value) {
		this._initialProperties.top = value;

		return {
			name: 'top',
			value: value + this._suffixes.top
		};
	},

	/**
	 * pull the value from the elements computed/current style
	 *
	 * @param {String} property name of css property
	 * @param {Boolean} raw return string value
	 */
	cssGetComputedProperty: function (property, raw) {
		var string;

		property = MAL.Environment.getCSSPropertyName(property, true);

		if (!property) return false;

		if (window.document.defaultView && window.document.defaultView.getComputedStyle) {
			string = window.document.defaultView.getComputedStyle(this.element, '').getPropertyValue(property);
			return raw ? string : MAL.parseNumber(string);
		} else if (this.element.currentStyle) {
			string = this.element.currentStyle[property];
			return raw ? string : MAL.parseNumber(string);
		}
	},

	/**
	 * turns string values into floats if they match the criteria
	 *
	 * @private
	 * @param  {String} value
	 * @return {String/Number}
	 */
	_parsePropertyValue: function (value) {
		var floatValue = parseFloat(value);

		if (MAL.isString(value) && value.replace(/(?:px|deg)$/, '') === String(floatValue)) value = floatValue;
		return value;
	}
}));