/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Stage class. Provides an animation environment for layers ({@link MAL.tween.CSSLayer}/{@link MAL.tween.JSLayer})
 *
 * **Usually you want to only have ONE stage**
 *
 * # Creating a stage
 *
 * create stage
 *     var stage = new MAL.tween.Stage();
 *
 * alias a getter
 *     var $ = stage.getLayers;
 *
 * add a tween to a layer
 *     $('Box').addTween({delay: .2, duration: .5, finish: {x: 100}});
 *
 * start stage animations
 *     stage.start();
 *
 * # Creating a stage with both JS and CSS layers (make sure you need to do this, this is usually a last resort)
 *
 * so sometimes you may want to force some elements to be CSS driven animations and some to be JS driven
 *
 * create the special stage that forces all tweens to be in JS unless overridden
 *     var stage = new MAL.tween.Stage({dualLayer: true, layerType: 'JS'});
 *
 * you can override a layer type like this
 *     stage.addCSSLayers('Box'); // this layer will animate with CSS
 *
 * so this is an example of two layers both being driven by CSS and JS
 *     $('Box', 'Box2').addTween({delay: .2, duration: .5, finish: {x: 100}});
 *
 * Box will be animated with CSSLayer and Box2 will be animated with JSLayer
 *
 * @class MAL.tween.Stage
 */

 /**
 * dispatched when a layer has been added to the stage
 *
 * @event layer-add
 * @param {Event} this
 * @param {Object} message {layer:}
 */

MAL.define('tween.Stage', MAL.Object.extend({
	/**
	 * initialize
	 *
	 * @method constructor
	 * @param {Object} config
	 */

	/**
	 * @cfg {String} layerType
	 * type of layer to use (CSS or JS). **be sure that the layer type is loaded before doing this**
	 */

	/**
	 * @cfg {String} dualLayer
	 * type you should set this to true if you will be using the layerType over ride methods
	 * - {@link #addJSLayer}
	 * - {@link #addCSSLayer}
	 */
	initialize: function ($config) {
		this._super();

		// handle options
		$config = $config || {};

		// object of layers on the stage
		this._layerCache = {};
		this._layers = [];
		this._animating = new MAL.Set();
		this._startAnimations = new MAL.Set();
		this._dualLayer = !!$config.dualLayer;
		this._positionAnchorPointElement = $config.positionAnchorPointElement;

		// set started default
		this.started = false;
		this._startTime = null;

		//
		this._layerType = $config.layerType || (MAL.Environment.bugs.transform3d ? 'JS' : false) || (MAL.Environment.getCSSPropertyName('transition') ? 'CSS' : 'JS');

		this._timeout = null;

		// timeout manager
		this.timeouts = new MAL.TimeoutManager({leniency: 0});

		if (this._dualLayer || this._layerType === 'JS') this._loop();
	},

	/**
	 * update loop
	 *
	 * @private
	 */
	_loop: function () {
		var time = new Date().getTime();

		this._animating.forEach(function (layer) {
			if (layer instanceof MAL.tween.JSLayer || layer instanceof MAL.tween.CSSJSLayer) layer._update(time);
		});

		if (MAL.Environment.requestAnimationFrame) {
			window[MAL.Environment.requestAnimationFrame](this._loop);
		} else {
			window.clearTimeout(this._timeout);
			this._timeout = window.setTimeout(this._loop, 0);
		}
	},

	/**
	 * simple CSS layer flagging
	 *
	 *     stage.addCSSLayers('Box', 'Cube');
	 *
	 * or
	 *
	 *     stage.addJSLayers(['Box', 'Cube']);
	 */
	addCSSLayers: function (layers) {
		layers = MAL.argumentsToArray(arguments);

		layers.forEach(function (layer) {
			this.addCSSLayer(layer);
		}, this);
	},

	/**
	 * simple JS layer flagging
	 *
	 *     stage.addJSLayers('Box', 'Cube');
	 *
	 * or
	 *
	 *     stage.addJSLayers(['Box', 'Cube']);
	 */
	addJSLayers: function (layers) {
		layers = MAL.argumentsToArray(arguments);

		layers.forEach(function (layer) {
			this.addJSLayer(layer);
		}, this);
	},

	/**
	 * simple CSSJS layer flagging
	 *
	 *     stage.addCSSJSLayers('Box', 'Cube');
	 *
	 * or
	 *
	 *     stage.addCSSJSLayers(['Box', 'Cube']);
	 */
	addCSSJSLayers: function (layers) {
		layers = MAL.argumentsToArray(arguments);

		layers.forEach(function (layer) {
			this.addCSSJSLayer(layer);
		}, this);
	},

	/**
	 * add forced CSS layer to the dualLayer stage
	 *
	 *     var Box = stage.addCSSLayer('Box');
	 *
	 * you can also add a layer with properties
	 *
	 *     var Box = stage.addCSSLayer('Box', {x: 50, y: 50, scale: 2});
	 *
	 * @param {String/HTMLElement} reference ID of the element (element id or instance)
	 * @param {Object} properties properties the layer should have
	 * @return {MAL.tween.CSSLayer}
	 */
	addCSSJSLayer: function (reference, properties) {
		if (!this._dualLayer) MAL.error('Stage: dualLayer support needs to be enabled');
		if (!MAL.tween.CSSLayer) MAL.error('MAL.tween.CSSLayer is missing');
		return this.addLayer(reference, properties, 'CSSJS');
	},

	/**
	 * add forced CSS layer to the dualLayer stage
	 *
	 *     var Box = stage.addCSSLayer('Box');
	 *
	 * you can also add a layer with properties
	 *
	 *     var Box = stage.addCSSLayer('Box', {x: 50, y: 50, scale: 2});
	 *
	 * @param {String/HTMLElement} reference ID of the element (element id or instance)
	 * @param {Object} properties properties the layer should have
	 * @return {MAL.tween.CSSLayer}
	 */
	addCSSLayer: function (reference, properties) {
		if (!this._dualLayer) MAL.error('Stage: dualLayer support needs to be enabled');
		if (!MAL.tween.CSSLayer) MAL.error('MAL.tween.CSSLayer is missing');
		return this.addLayer(reference, properties, 'CSS');
	},

	/**
	 * add forced JS layer to the dualLayer stage
	 *
	 *     var Box = stage.addJSLayer('Box');
	 *
	 * you can also add a layer with properties
	 *
	 *     var Box = stage.addJSLayer('Box', {x: 50, y: 50, scale: 2});
	 *
	 * @param {String/HTMLElement} reference ID of the element (element id or instance)
	 * @param {Object} properties properties the layer should have
	 * @return {MAL.tween.JSLayer}
	 */
	addJSLayer: function (reference, properties) {
		if (!this._dualLayer) MAL.error('Stage: dualLayer support needs to be enabled');
		if (!MAL.tween.JSLayer) MAL.error('MAL.tween.JSLayer is missing');
		return this.addLayer(reference, properties, 'JS');
	},

	/**
	 * add layer to the stage
	 *
	 *     var Box = stage.addLayer('Box');
	 *
	 * you can also add a layer with properties
	 *
	 *     var Box = stage.addLayer('Box', {x: 50, y: 50, scale: 2});
	 *
	 * @param {String/HTMLElement} reference ID of the element (element id or instance)
	 * @param {Object} properties properties the layer should have
	 * @return {MAL.tween.JSLayer/MAL.tween.CSSLayer}
	 */
	addLayer: function (reference, properties, _layerType) {
		// restrict scope
		var element;

		// detect if element exists
		if (typeof reference === 'object' && reference.nodeType > 0) {
			element = reference;
		} else {
			MAL.error('InvalidElement:', reference);
		}

		// check if layer is in stage
		if (element.MALTweenLayer) MAL.error('LayerExists: layer "' + reference + '" is already in stage');

		// use the correct layer object
		var layerType = _layerType || this._layerType;

		var layer = new MAL.tween[layerType + 'Layer']({stage: this, element: element, begin: properties, positionAnchorPointElement: this._positionAnchorPointElement});

		// add listeners
		layer.addEventListeners({
			'animation-start': this._onAnimationStart,
			'animation-end': this._onAnimationEnd,
			'animation-add': this._onAnimationAdd
		});

		// directly link layer to element
		element.MALTweenLayer = layer;

		// dispatch layer add event
		this.dispatchEvent('layer-add', {layer: element.MALTweenLayer});

		// push the layer onto the stack
		this._layers.push(layer);

		// add helper reference
		if (typeof reference === 'string') this._layerCache[reference] = element.MALTweenLayer;

		// return layer
		return element.MALTweenLayer;
	},

	/**
	 * add multiple layers to stage without start properties
	 *
	 *     var layers = stage.addLayers('Box', 'Rectangle', 'Circle');
	 *
	 * obtain one of the layers you added and animate on it
	 *
	 *     layers.Box.addTween({duration: 1, finish: {x: 50, y: 50}});
	 *
	 * @param {Array} layers set of layers
	 * @return {Object} layers
	 */
	addLayers: function (layers) {
		// convert arguments into an array
		layers = MAL.argumentsToArray(arguments);

		var references = {};

		// traverse layer array
		layers.forEach(function (layer) {
			// add layer to stack
			references[layer] = this.addLayer.apply(this, [layer]);
		}, this);

		return references;
	},

	/**
	 * get a layer reference from the element ID
	 *
     *     var Box = stage.getLayer('Box');
	 *
	 * @param {String/Object} reference ID of the element (element id or instance)
	 * @return {MAL.tween.JSLayer/MAL.tween.CSSLayer}
	 */
	getLayer: function (reference) {
		// restrict scope
		var element;

		if (typeof reference === 'object' && MAL.isUndefined(reference.nodeType) && MAL.tween.Layer) {
			return reference;
		} else if (typeof reference === 'object' && reference.nodeType > 0) {
			// detect if element exists
			element = reference;
		}

		if (!element.MALTweenLayer) this.addLayer(reference);//MAL.error('MissingLayer: layer "' + reference + '" is not on the stage');

		return element.MALTweenLayer;
	},

	/**
	 * creates a proxy layers object that can perform multiple methods on a set of layers
	 *
     *     var numbers = stage.getLayers('One', 'Two', 'Three');
     *
     * or
     *
     *     var numbersArray = ['One', 'Two', 'Three'],
     *     numbers = stage.getLayers(numbersArray);
     *
     * then we can set a property on all the layers
     *
     *     numbers.setProperty('opacity', 0);
     *
     * you can also use the each method to do something with the layers
     *
     *     numbers.each(function (layer) {
	 *      console.log(layer.element);
     *     });
     *
     * or you can just pull an element from the proxy object like this
     *
     *     var first = numbers[0];
	 *
	 * @param {Array} reference ID of the element (element id or instance)
	 * @return {Object} proxy layer object
	 */
	getLayers: function (layers) {
		// convert arguments into an array
		layers = MAL.argumentsToArray(arguments);

		// if we only have one layer there is no huge advantage of using the proxy
		if (layers.length === 1) return this.getLayer(layers[0], true);

		// update query array to hold the actual Layer objects
		layers.forEach(function (value, key, array) {
			array[key] = this.getLayer(value, true);
		}, this);

		//new ExampleProxy({self: window, layers: [function () {}]});
		return new MAL.tween[this._layerType + 'LayerProxy']({layers: layers});
	},

	/**
	 * layer animation starts
	 *
	 * @private
	 */
	_onAnimationStart: function (o, e) {
		// add layer to animating pool
		this._animating.add(o);
	},

	/**
	 * layer animation ends
	 *
	 * @private
	 */
	_onAnimationEnd: function (o, e) {
		if (o._tweens.length) return;

		// remove layer to animating pool
		this._animating.remove(o);
	},

	/**
	 * layer aniamtion adds
	 *
	 * @private
	 */
	_onAnimationAdd: function (o, e) {
		// add layer to start animations pool if the stage hasn't started yet
		if (!this.started) this._startAnimations.add(o);
	},

	/**
	 * start stage animation
	 *
	 *     stage.start();
	 *
	 */
	start: function () {
		this.started = true;
		this._startTime = new Date().getTime();
		this._layers.forEach(function (layer) {
			if (layer.layerType === 'CSS') layer._setCSSTextFromBuffer();
		});

		this._startAnimations.forEach(function (layer) {
			layer._startNextAnimation();
		});
	}
}));