/**
 * @author Martijn Korteweg <martijn@mediaartslab.com>
 * @docauthor Martijn Korteweg <martijn@mediaartslab.com>
 *
 *
 * video should control all video element interactions. The Advanced video class added several options to the base video class
 *
 * - Bandwidth Calculation
 * - Animated Volume fade
 * - Que Points
 *
 * #Usage
 *
 * Initialize your video object first
 *
 * 		var video = new MAL.video.Advanced({
 *	  			controls: false,
 * 		 	minimumTime: 1,
 * 		  	volume: 0,
 * 		   	autobuffer: true
 *      	});
 *
 * Then you can set your source object
 *
 * 		video.Advanved.setSource({
 * 			path: 'http://somwehre',
 * 		 	file: 'videoname'
 * 		  	type: 'mp4', // override only
 * 		   	filePathProxy: RMV.filePathProxy
 *       	});
 *
 * #Bandwidth
 * To calculate the bandwidth from the video file you can used calculateBandwidth when the object is initilized
 *
 * 		video.Advanced.calculateBandwidth({
 * 			mp4: 4.2,
 * 			webm: 4.1
 * 		});
 *
 * #Fade Volume
 * You can fade the volume of the video over time
 *
 * 		video.Advanced.fadeVolume({
 * 			delay: 0,
 * 			volume: 1,
 * 			duration: 2,
 * 			ease: [0,0,1,1]
 * 		});
 *
 * #Que Points
 * Add que points to any moment in time of the video, all Quepoints are based on seconds in the video
 *
 * 		video.Advanced.addCuePoint(3,'bigboom');
 *
 * 		video.Advanced.addEventListener('bigboom',function () {
 * 			//do something
 * 		});
 * 		
 * @class MAL.video.Advanced
 * @extends MAL.video.Base
 *
 */
MAL.define('video.Advanced', MAL.video.Base.extend({
	/**
	 * initializes MAL.video.Advanced
	 *
	 * @method constructor	 
	 *
	 *
	 */
	
	/**
	 * @cfg {Number} cuePointInterval
	 * By default quepoints get fired on the timeupdate event, this option can override that to a timeout.
	 */
	initialize: function ($config) {
		this._options = $config || {};

		this._super(this._options);

		// the quepoints can be set to a specific check interval to tweak hit timing
		this._cuePointInterval = this._options.cuePointInterval;

		// quepoints array
		this._cuePoints = [];

		this._volumeOptions = {};

		// have we allready given a bandwidth reading
		this._dispatchedBandwidth = false;

		// video size set in Mb
		this._videoSize = 0;
	},

	/**
	 * Changing volume with a fade
	 * @param  {Object} $config
	 * @param {Number} $config.delay delay before fade happens
	 * @param {Number} $config.volume the target volume to fade towards
	 * @param {Number} $config.duration duration of the fade
	 * @param {Array} $config.ease Cubic Bezier Ease array
	 *
	 *		video.Advanced.fadeVolume({
	 *	 		delay: 0,
	 *	   		volume: 1,
	 *	     	duration: 2,
	 *	      	ease: [0,0,1,1]
	 *		});
	 * 
	 */
	fadeVolume: function($config) {

		$config = $config || {};

		// delay of volume change
		this._volumeOptions.delay = $config.delay || 0;

		// new volume
		this._volumeOptions.volume = $config.volume;

		// duration of the volume change
		this._volumeOptions.duration = $config.time || $config.duration || 0.25;

		// easing of the volume change
		this._volumeOptions.ease = $config.ease || [0,0,1,1];

		// if there is a startwidth it will be changed to that volume instantly
		this.element.volume = this._volumeOptions.currVolume = $config.startWidth || this.element.volume;

		// set the muted to false
		this.element.muted = false;

		// start time of the animation
		this._volumeOptions.startTime = new Date().getTime();

		// kickoff the fade loop
		this._fadeLoop();
	},

	/**
	 * 
	 * Loop for fading animation
	 * @private
	 *
	 */
	_fadeLoop: function() {

		var now = new Date().getTime(),
			cT = ((now-this._volumeOptions.startTime)/1000)-this._volumeOptions.delay;

		// wait for delay
		if (cT <= 0) {
			setTimeout(this._fadeLoop,(1000/60));
			return;
		}

		var	b = this._volumeOptions.currVolume,
			c = this._volumeOptions.volume - b,
			d = this._volumeOptions.duration,
			a = ((1 / d) * cT),
			ease = this._volumeOptions.ease,
			secpos = MAL.cubicBezierWithPercent(a, ease[0], ease[1], ease[2], ease[3]),
			newVolume = ((c * secpos) + b);

		if (cT > this._volumeOptions.duration) {
			newVolume = this._volumeOptions.volume;
		} else {
			setTimeout(this._fadeLoop,(1000/60));
		}

		var volume = Math.round((newVolume)*1000)/1000;
		this.setVolume(volume);
	},

	/**
	 * Stop video from loading by overriding the source
	 *
	 * 		video.Advanced.setBlankSource();
	 */
	setBlankSource: function() {
		this._loaded = true;
		this.canPlayThrough = false;
		this.malCanPlayThrough = false;
		this.element.src = 'data:video/' + this.type + ';base64,' + this._blanks[this.type];
		this.element.parentNode.removeChild(this.element);
		this.element = {};
	},

	/**
	 * Adding quepoint to the video
	 *
	 * Quepoints are based on events, when you add a quepoint there will be an event on the this
	 * instance with the same name as your quepoint.
	 *
	 * @param {Number} time At what time does this need to get fired
	 * @param {String} name Event name that gets fired
	 *
	 * 		video.Advanced.addCuePoint(3,'bigboom');
	 *
	 * 		video.Advanced.addEventListener('bigboom',function () {
	 * 			//do something
	 * 		});
	 */
	addCuePoint: function(time, name) {

		if (this._cuePoints.length === 0) {
			if (!MAL.isUndefined(this._cuePointInterval)) {
				setInterval(this._fireCuePoints, this._cuePointInterval);
			} else {
				this.element.addEventListener('timeupdate', this._fireCuePoints, true);
			}
			// when a seek has happend on the video, the quepoints will be reset.
			this.element.addEventListener('seeked', this._resetCuePoints, true);
		}

		// push the quepoint to the array
		this._cuePoints.push({
			time: time,
			name: name,
			fired: false
		});
	},

	/**
	 * Reset quepoint to make sure it can fire again
	 * @param  {String} name name of the quepoint
	 * @private
	 */
	_resetCuePoint: function (name) {
		for (var i = 0, l = this._cuePoints.length; i < l; i++) {
			if (this._cuePoints[i].name === name) {
				this._cuePoints[i].fired = false;
				break;
			}
		}
	},

	/**
	 * Reset all quepoints
	 *
	 * Bugzid: 4903, added setTimeout for safari 5.1 to catch up on seeked event.
	 * @private
	 */
	_resetCuePoints: function (e) {
		setTimeout(function () {
			this._cuePoints.forEach(function(value, key, ref) {
				if (this.element.currentTime < value.time)
					value.fired = false;
			}.bind(this));
		}.bind(this),10);
	},

	/**
	 * Fire the event for the quepoint
	 * @private
	 */
	_fireCuePoints: function () {
		// check if the video is playing
		if (!this.playing) return;

		// currentTime of the video
		var time = this.element.currentTime;

		// set current scope
		var _scope = this;

		// run through all quepoints to fire them
		this._cuePoints.forEach(function(value, key, ref) {
			if (value.fired || time < value.time) return;
			_scope.dispatchEvent(value.name);
			value.fired = true;
		});

	},

	/**
	 * start bandwidth calculation
	 *
	 * @param {Object} sizeObject object of Mb sizes from all videos served
	 * @param {Number} sizeObject.mp4
	 * @param {Number} sizeObject.webm
	 * 
	 */
	calculateBandwidth: function(sizeObject) {
		if (typeof sizeObject[this.type] === 'undefined')
			return;

		// initial bandwidth set
		this._bandwidth = {
			average:0,
			readings:0,
			history:[]
		};

		// set the video size
		this._videoSize = sizeObject[this.type];

		// listen to the progress event on this object
		this.addEventListener('progress',this._calculateBandwidthLoop.bind(this), false);
	},

	/**
	 * Loop bandwidth
	 * @private
	 * @param  {Object} o Event
	 * @param  {Object} e Passed variabled from the calculations
	 */
	_calculateBandwidthLoop: function (o,e) {

		// lets not calculate when cached
		if (MAL.isUndefined(e.cached) || e.cached === 100 || e.timeTaken === 0) return;

		// if there is a video size we should calculate the current bandwidth
		if (this._videoSize > 0) {
			this._bandwidth.readings++;

			var bandwidth = (((this._videoSize * 1024) * ((e.percent-e.cached)/100) * 8) / e.timeTaken);

			this._bandwidth.average += (bandwidth-this._bandwidth.average)/this._bandwidth.readings;
			this._bandwidth.history.push(bandwidth);
		}

		// when load is done, lets show the average bandwidth.
		if (e.percent === 100 && this._dispatchedBandwidth === false) {
			this._dispatchedBandwidth = true;
			this.dispatchEvent('averagebandwidth', this._bandwidth);
		}

		// dispatch current bandwidth
		this.dispatchEvent('bandwidth', this._bandwidth);
	}

}));