/**
 * @author Martijn Korteweg <martijn@mediaartslab.com>
 * @docauthor Martijn Korteweg <martijn@mediaartslab.com>
 *
 *
 * video should control all video element interactions.
 *
 * #usage
 *
 * Initialize your video object first
 *
 * 		var video = new MAL.video.Base({
 *	  			controls: false,
 * 		 	minimumTime: 1,
 * 		  	volume: 0,
 * 		   	autobuffer: true
 *      	});
 *
 * Then you can set your source object
 *
 * 		video.setSource({
 * 			path: 'http://somwehre',
 * 		 	file: 'videoname'
 * 		  	type: 'mp4', // override only
 * 		   	filePathProxy: RMV.filePathProxy
 *       	});
 *
 * @class MAL.video.Base
 * @extends MAL.Object
 */
MAL.define('video.Base', MAL.Object.extend({
	/**
	 * initializes MAL.video.Base
	 *
	 * @method constructor
	 * @param {Object} $config
	 */

	/**
	 * @cfg {Boolean} autobuffer
	 * autobuffer setting to preload video when the source is set
	 */

	/**
	 * @cfg {Boolean} controls
	 * turn on controls on the video element
	 */

	/**
	 * @cfg {Number} minimumTime
	 * minimum time before the progress event gets fired
	 */

	/**
	 * @cfg {Boolean} loop
	 * Make the video loop
	 */

	/**
	 * @cfg {Number} volume
	 * starting volume
	 */

	/**
	 * @event start
	 * Fired when the video starts to play
	 */

	/**
	 * @event cached
	 * Fired on the first progress event reading
	 * @param {Number} percent the percentage of the video that is loaded on cached
	 */

	/**
	 * @event ended
	 * Fired when video is ended
	 */

	/**
	 * @event canplay
	 * Passthrough from the video element
	 */

	/**
	 * @event loadeddata
	 * Passthrough from the video element
	 */

	/**
	 * @event error
	 * Passthrough from the video element
	 */

	/**
	 * @event stalled
	 * Passthrough from the video element
	 */

	/**
	 * @event abort
	 * Passthrough from the video element
	 */

	/**
	 * @event waiting
	 * Passthrough from the video element
	 */

	/**
	 * @event loadstart
	 * Passthrough from the video element
	 */

	/**
	 * @event emptied
	 * Passthrough from the video element
	 */

	/**
	 * @event play
	 * Passthrough from the video element
	 */

	/**
	 * @event pause
	 * Passthrough from the video element
	 */

	/**
	 * @event playing
	 * Passthrough from the video element
	 */

	/**
	 * @event seeking
	 * Passthrough from the video element
	 */

	/**
	 * @event seeked
	 * Passthrough from the video element
	 */

	/**
	 * @event ratechange
	 * Passthrough from the video element
	 */

	/**
	 * @event durationchange
	 * Passthrough from the video element
	 */

	/**
	 * @event volumechange
	 * Passthrough from the video element
	 */

	/**
	 * @event timeupdate
	 * Passthrough from the video element
	 */

	initialize: function ($config) {
		this._super();

		// setup options from $options passed in
		this._options = $config || {};

		// create the video element with buffer set from options
		this.element = document.createElement('video');

		this._setAutoBuffer(this._options.autobuffer);

		this.element.loop = this._options.loop;

		this.element.controls = this._options.controls || false;

		this.minimumTime = this._options.minimumTime || 1;

		// progress loops reset
		this._loops = 0;

		// start progress loop
		setTimeout(this._loop, 200);


		var events = [
			'_onCanPlayThrough',
			'_onLoadedMetaData',
			'_onEnded'
		];

		events.forEach(function(value, key, ref) {
			this.element.addEventListener(value.slice(3).toLowerCase(), this[value].bind(this), true);
		}.bind(this));

		var PassThroughEvents = [
			'_onCanPlay',
			'_onLoadedData',
			'_onError',
			'_onStalled',
			'_onSuspend',
			'_onAbort',
			'_onWaiting',
			'_onLoadStart',
			'_onEmptied',
			'_onPlay',
			'_onPause',
			'_onPlaying',
			'_onSeeking',
			'_onSeeked',
			'_onRateChange',
			'_onDurationChange',
			'_onVolumeChange',
			'_onTimeUpdate'
		];

		// making sure that we can listen to all element events on the video object
		PassThroughEvents.forEach(function(value, key, ref) {
			this.element.addEventListener(value.slice(3).toLowerCase(), function(e,o) {

				if (!this._forceload) this.dispatchEvent(e.type, false, e);
			}.bind(this), true);
		}.bind(this));

		// the internal object with all data related to the video load process
		this._loadDetails = {
			timeTaken: 0,
			percent: 0,
			cached: 0,
			percentNeeded: 100,
			percentLoaded: 0,
			timeNeeded: null, // time needed till 100%
			timeNeededTotal: null, // time till 100% from start
			bufferPercent: 0,
			bufferTime: null, // time needed till play
			averageBufferTime: 0 // average needed till play
		};

		// current playing state
		this.playing = false;

		// current can play through state
		this._canPlayThrough = false;

		// duration of the video
		this._duration = 0;

		// if the video has played
		this._played = false;

		// current volume of the video
		this._volume = null;

		// if our calculations think it can be played
		this.malCanPlayThrough = false;

		// if the source has been set
		this._hasSource = false;

		if (!MAL.isUndefined(this._options.volume)) {
			this._volume = this._options.volume;
			this.element.addEventListener('loadeddata',this._setLoadedVolume.bind(this),false);
		}

		this.types = {
			mp4: 'video/mp4; codecs="avc1.42E01E, mp4a.40.2"',
			webm: 'video/webm; codecs="vp8, vorbis"',
			ogg: 'video/ogg; codecs="theora, vorbis"'
		};

		// check what type of video can be played
		for (var i in this.types) {
			if (this.element.canPlayType(this.types[i]) === 'probably') {
				this.type = i; break;
			}
		}

		// Browser Bug: Windows Firefox 23+ has an issue with hardware acceleration (http://support.mozilla.org/en-US/questions/963666)
		if (MAL.Environment.bugs.forceWebm) this.type = 'webm';

		// force load variables
		this._lastPercentChangeTime = null;
		this._stalled = false;
		this._forceload = false;

		// check if the element has suspended the load
		this.element.addEventListener('suspend', function () {
			if (!this.malCanPlayThrough && !this._stalled && this._loadDetails.percent > 0) {
				this._stalled = true;
				this._forceload = true;
				this._forceLoadChangeCurrentTime();
			}
		}.bind(this), false);

	},

	/**
	 * Setting the source of the video
	 *
	 * 	video.setSource({
	 *	 		path: 'http://somwehre',
	 *	  		file: 'videoname',
	 *	   		type: 'mp4',
	 *	     	filePathProxy: RMV.filePathProxy
	 *	    });
	 *
	 * @param {Object} $config source settings
	 * @param {String} $config.path Path to the file
	 * @param {String} $config.file name of the file
	 * @param {String} $config.type override file type
	 * @param {Function} $config.filePathProxy the proxy function that runs through the RMV object
	 *
	 */

	setSource: function($config) {

		var opts = $config || {};

		if (MAL.isUndefined(opts.path) || MAL.isUndefined(opts.file)) MAL.error('source is required');
		if (!MAL.isUndefined(opts.duration)) this.setDuration(opts.duration);


		this.type = opts.type || this.type;
		this._dispatchedCanPlayThrough = false;

		var source = opts.path + opts.file + '.' + this.type;
		if (MAL.isFunction(opts.filePathProxy))
			source = opts.filePathProxy(opts.path,opts.file + '.' + this.type);

		this.element.sourceElement = document.createElement('source');
		this.element.sourceElement.src = source;
		this.element.sourceElement.type = this.types[this.type];

		this.element.appendChild(this.element.sourceElement);
		this._startTime = new Date().getTime();

		this._loaded = false;
		this._hasSource = true;
	},

	/**
	 * Start the buffering of the video when autobuffer was turned off initially
	 *
	 *		video.Base.startBuffer();
	 *
	 */
	startBuffer: function() {
		this._setAutoBuffer(true);
	},

	/**
	 * sets buffer state
	 *
	 * @private
	 */
	_setAutoBuffer: function (state) {
		this.element.autobuffer = state;
		this.element.preload = state ? 'auto' : 'none';
		this._autobuffer = state;
	},

	/**
	 * Getter for muted state
	 * @return {Boolean} true or false based on volume
	 *
	 *		video.Base.isMuted();
	 *
	 */
	isMuted: function () {
		return this.volume === 0 ? true : false;
	},

	/**
	 * Playing the video
	 * @param  {Number} volume volume between 0 and 1 of the volume the video should be playing with
	 * @param  {Number} delay  Number of seconds you want the video to be delayed before playing.
	 *
	 *		video.Base.play(1,0);
	 *
	 *
	 */
	play: function(volume, delay) {
		var self = this;

		if (this.volume !== volume && volume !== undefined) this.setVolume(volume);

		if (this._played || this._forceload)
			try{this.element.currentTime = 0;}catch(e){}


		this._played = true;

		if (!delay) {
			this.element.play();
			this.dispatchEvent('start');
			this.playing = true;
			this._forceload = false;
		} else {
			setTimeout(function(){
				self.element.play();
				self.dispatchEvent('start');
				self.playing = true;
				self._forceload = false;
			}, delay);
		}
	},

	/**
	 * Stopping the video
	 *
	 * 		video.Base.stop();
	 *
	 */
	stop: function() {
		this.playing = false;
		this.element.pause();
	},

	/**
	 * Set the volume of the video
	 * @param {Number} volume Number between 0 and 1 of the volume
	 *
	 * 		video.Base.setVolume(1);
	 *
	 */
	setVolume: function(volume) {
		this.element.volume = (volume) ? volume : 0;
		this.volume = volume;
		this.element.muted = (volume) ? false : true;
	},

	/**
	 * Change the current time of hte video
	 * @param {Number} time
	 *
	 *		video.Base.setCurrentTime(10); // set the video to 10 seconds
	 *
	 */
	setCurrentTime: function(time) {
		this.element.currentTime = time;
	},

	/**
	 * Setting specific duration of the video
	 * @param {Number} duration
	 *
	 * 		video.Base.setDuration(12); // set the duration of the video to 12
	 *
	 */
	setDuration: function (duration) {
		this._duration = duration;
	},

	/**
	 * Get the current videos duration
	 * @return {Number} Seconds of the duration
	 *
	 *		video.Base.duration(); // returned current duration
	 *
	 */
	duration: function () {
		return MAL.isNumber(this.element.duration) ? this.element.duration : this._duration;
	},

	/**
	 * Getter for _played
	 * @return {Boolean} If the video has played or not
	 *
	 *		video.Base.hasPlayed();
	 *
	 */
	hasPlayed: function() {
		return this._played;
	},

	/**
	 * interval for progress events
	 * @private
	 */
	_loop: function () {
		// progress event calculation/dispatch
		if (this._loaded) {
			return;
		} else if (this._autobuffer === true && this._hasSource) {
			this._dispatchProgress();
		}

		setTimeout(this._loop, 200);
	},

	/**
	 * Checking if the loading progress has stalled, and if we should start forceload
	 * @param  {Number} percent current percentage the video is loaded
	 * @private
	 */
	_checkStalled: function (percent) {
		var now = Date.now();
		if (!this._stalled && (
								// cached
								this._loadDetails.cached > 0 && this._loadDetails.cached === this._loadDetails.percent && now - this._lastPercentChangeTime > 500 ||
								// normal load
								this._lastPercentChangeTime !== null && now - this._lastPercentChangeTime > 3000
							)) {
			this._stalled = true;
			this._forceload = true;
			this._forceLoadChangeCurrentTime();
		} else if (this._lastPercentChangeTime !== null && now - this._lastPercentChangeTime < 1000) {
			this._stalled = false;
		}

		if (this._loadDetails.percent !== percent && !(this._loadDetails.cached > 0 && this._loadDetails.cached === this._loadDetails.percent)) this._lastPercentChangeTime = now;
	},

	/**
	 * Setting current time to for forceload
	 * @private
	 */
	_forceLoadChangeCurrentTime: function () {
		var time = this.duration() * (this._loadDetails.percent / 100);
		this.log('Video: Force Loading (time: ' + time + ')');
		this.element.currentTime = time;
	},

	/**
	 * manual progress event that does additional calculations
	 *
	 * @private
	 */
	_dispatchProgress: function () {

		// set some current values
		var currentTime = new Date().getTime(),
			percent = this._loadDetails.percent;

		// check the time taken to load
		this._loadDetails.timeTaken = (currentTime - this._startTime) / 1000;
		// percent calculation
		if (this.element && this.element.buffered && this.element.buffered.length) {
			percent = (this.element.buffered.end(0) / this.duration()) * 100;
		}

		// check if IE is 90% cached to force to 100
		if ((MAL.Environment.browser === 'msie' && percent > 90) || (MAL.Environment.browser === 'chrome' && percent > 98) || (MAL.Environment.os === 'ipad' && MAL.Environment.osVersion >= 8 && percent > 98)) percent = 100;

		// check if we are 100% cached, then dispatch a cached event
		if (percent === 100 && this._loadDetails.percent === 0)
			this.dispatchEvent('cached', {percent:percent});

		if (!this._played) this._checkStalled(percent);

		// set current percent to load details
		this._loadDetails.percent = percent;

		// when the video is fully loaded
		if (this._loadDetails.percent >= 100) {
			this.canPlayThrough = this._loaded = true;
			this._loadDetails.bufferPercent = 100;
			this._loadDetails.bufferTime = 0;
			this._loadDetails.averageBufferTime = 0;
			this._loadDetails.timeNeeded = 0;
			this.dispatchEvent('progress', this._loadDetails);
			if (this._forceload) this.element.currentTime = 0;

			if (this.malCanPlayThrough === false) {
				this.malCanPlayThrough = true;
				this._malCanPlayThrough();
			}

			return;
		}

		// we start the load calculations beyond this point, this gets set to true in _onLoadedMetaData
		if (!this._loadstart) return;

		// initial loop after the data loaded event
		if (this._loops === 0) {
			this._loadDetails.cached = this._loadDetails.percent;
			this.dispatchEvent('cached', {percent:percent});
			this._startTime = currentTime;
		}

		// increment the loops after initial chcek has been done
		this._loops++;


		// get percent loaded from the current percent minus the cachedPercent, and percent needed till full load
		var	percentLoaded = (this._loadDetails.percent - this._loadDetails.cached)/100,
			percentNeeded = (100-this._loadDetails.percent) / 100;

 		this._loadDetails.percentLoaded = (percentLoaded*100);
 		this._loadDetails.percentNeeded = (percentNeeded*100);


		// time needed till 100% loaded
		if (percentLoaded > 0 ) this._loadDetails.timeNeeded = (this._loadDetails.timeTaken/percentLoaded) * percentNeeded;

		// if timeNeeded is Infinity we set it to 0
		if (isNaN(this._loadDetails.timeNeeded)) this._loadDetails.timeNeeded = 0;

		// total time Needed till 100% from timeNeeded plus how long it allready took
		this._loadDetails.timeNeededTotal = this._loadDetails.timeNeeded + this._loadDetails.timeTaken;

		// calculate time needed till we can play, based on time needed to 100% and susctract the play time from that
		this._loadDetails.bufferTime = (this._loadDetails.timeNeeded - this.duration());

		// when the bufferTime Needed to play is lower then 0 it should be ready and set reset it to 0
		if (this._loadDetails.bufferTime < 0) this._loadDetails.bufferTime = 0;

		// calculate buffer percent based on time needed and time taken
		if (this._loadDetails.bufferPercent !== 100 && (this._loadDetails.timeTaken != this._loadDetails.timeNeededTotal)) {
			this._loadDetails.bufferPercent = (this._loadDetails.timeTaken/this._loadDetails.timeNeededTotal) * 100;
		}

		// calculate average buffer time needed
		var averageAdjustment = (this._loadDetails.timeNeeded-this._loadDetails.averageBufferTime)/this._loops;
		this._loadDetails.averageBufferTime += averageAdjustment;

		// dispatch the progress event after minimum load time is over
		if (this._loadDetails.timeTaken > this.minimumTime) {

			this.dispatchEvent('progress', this._loadDetails);
			if (!this._forceload && this._loadDetails.bufferPercent >= 100 && this.malCanPlayThrough === false) {
				this.malCanPlayThrough = true;
				this._malCanPlayThrough();
			}
		}

		//console.log(this._loadDetails);


	},

	/**
	 * When video can play through based on browser and our own determination, we fire the malcanplaythrough event
	 * @private
	 */
	_malCanPlayThrough: function() {

		if (this._dispatchedCanPlayThrough === true) return;

		if (this.malCanPlayThrough === true && this.canPlayThrough === true) {
			this._dispatchedCanPlayThrough = true;
			this.dispatchEvent('malcanplaythrough', false);
		}
	},

	/**
	 * When video is ready based on the browser, the canplaythrough event will be fired
	 * @param  {Event} e the original event from the video
	 * @private
	 */
	_onCanPlayThrough: function (e) {
		this.canPlayThrough = true;
		this._malCanPlayThrough();
		this._dispatchProgress();
		this.dispatchEvent('canplaythrough', false, e);
	},

	/**
	 * Setting volume when metadata is loaded, this is the best time to set volume
	 * @param {Event} e
	 * @private
	 */
	_setLoadedVolume: function (e) {
		this.setVolume(this._volume);
		this.log('setting volume to ',this._volume,' on metadataloaded');
	},

	/**
	 * Start the load process when metadata has been loaded.
	 * @param  {Event} e original event from the element
	 * @private
	 */
	_onLoadedMetaData: function (e) {
		this._loadstart = true;
		this.dispatchEvent('loadedmetadata', false, e);
	},

	/**
	 * When the video has ended
	 * @param  {Event} e original event from the video element
	 * @private
	 */
	_onEnded: function (e) {
		this.dispatchEvent('ended', false, e);
		this.playing = false;
	}
}));