/* global Backbone, jQuery, _, MakeFormatBuilder */
var MakeFormatBuilder = MakeFormatBuilder || {};

(function(Backbone, $, _, builder) {
	'use strict';

	/**
	 * Defines the format parameters to register with the TinyMCE Formatter.
	 *
	 * @since 1.4.1.
	 */
	builder.definitions.list = {
		selector: 'ul',
		classes: 'ttfmake-list'
	};

	/**
	 * Define the selector for detecting this format in existing content.
	 *
	 * @since 1.4.1.
	 */
	builder.nodes.list = 'ul.ttfmake-list';

	/**
	 * Defines the listbox item in the 'Choose a format' dropdown.
	 *
	 * @since 1.4.1.
	 *
	 * @returns object
	 */
	builder.choices.list = function() {
		var parent = builder.getParentNode('ul'),
			choice, isUL;

		// Determine if the current node or a parent is an unordered list.
		isUL = ( $(parent).is('ul') );

		choice = {
			value: 'list',
			text: 'List',
			disabled: (false === isUL)
		};

		return choice;
	};

	/**
	 * The Button format model.
	 *
	 * @since 1.4.1.
	 */
	builder.formats = builder.formats || {};
	builder.formats.list = builder.FormatModel.extend({
		/**
		 * Default format option values.
		 *
		 * @since 1.4.1.
		 */
		defaults: {
			update: false,
			id: 0,
			icon: '',
			colorIcon: '#808080'
		},

		/**
		 * Populate the options with any existing values.
		 *
		 * @since 1.4.1.
		 */
		initialize: function() {
			var node = builder.getParentNode(builder.nodes.list);

			// Create a new element ID.
			this.set('id', this.createID());

			// Check to see if we're updating an existing format.
			if (true === this.get('update')) {
				this.parseAttributes(node);
			}
		},

		/**
		 * Defines the fields in the options form.
		 *
		 * @since 1.4.1.
		 *
		 * @returns array
		 */
		getOptionFields: function() {
			var items = [
				builder.getIconButton( 'icon', 'Icon' ),
				builder.getColorButton( 'colorIcon', 'Icon Color' )
			];

			return this.wrapOptionFields(items);
		},

		/**
		 * Parse an existing format node and extract its format options.
		 *
		 * @since 1.4.1.
		 *
		 * @param node
		 */
		parseAttributes: function(node) {
			var self = this,
				$node = $(node),
				iconClasses;

			// Get an existing ID.
			if ($node.attr('id')) this.set('id', $node.attr('id'));

			// Parse the icon.
			iconClasses = $node.find('li').first().attr('class').split(/\s+/);
			if (iconClasses) {
				$.each(iconClasses, function(index, iconClass) {
					if (iconClass.match(/^fa-/) && ! iconClass.match(/fa-li/)) {
						self.set('icon', iconClass);
						return false;
					}
				});
			}

			// Icon color.
			if ($node.attr('data-icon-color')) this.set('colorIcon', $node.attr('data-icon-color'));
		},

		/**
		 * Insert the format markup into the editor.
		 *
		 * @since 1.4.1.
		 */
		insert: function() {
			var $node, iconClasses;

			// If no icon is set, remove any existing format and then bail.
			if ('' == this.escape('icon')) {
				this.remove();
				return;
			}

			// If not updating an existing format, apply to the current selection using the Formatter.
			if (true !== this.get('update')) {
				builder.editor.formatter.apply('list');
			}

			// Make sure the right node is selected.
			$node = $(builder.getParentNode(builder.nodes.list));

			// Set the element ID, if it doesn't have one yet.
			if (! $node.attr('id')) {
				$node.attr('id', this.escape('id'));
			}

			// Add attributes.
			$node.attr('data-icon-color', this.escape('colorIcon'));

			// Set up the list items.
			iconClasses = $node.find('li').first().attr('class');
			if (iconClasses) {
				iconClasses = iconClasses.split(/\s+/);
				$.each(iconClasses, function(index, iconClass) {
					// Remove any existing FontAwesome classes.
					if (iconClass.match(/^fa-/)) {
						$node.find('li').removeClass(iconClass);
					}
				});
			}
			// Add the current icon class to each list item.
			$node.find('li').addClass(this.escape('icon'));
		},

		/**
		 * Remove the existing format node.
		 *
		 * @since 1.4.1.
		 */
		remove: function() {
			var $node = $(builder.getParentNode(builder.nodes.list)),
				listID = $node.attr('id'),
				iconClasses = $node.find('li').first().attr('class').split(/\s+/);

			// Remove the ID attribute if it was generated by the Format Builder.
			if (listID.match(/^ttfmake-/)) {
				$node.removeAttr('id');
			}

			// Remove classes and other attributes.
			$node.removeClass('ttfmake-list');
			$node.removeAttr('data-icon-color');

			// Remove classes from list items.
			if (iconClasses) {
				$.each(iconClasses, function(index, iconClass) {
					if (iconClass.match(/^fa-/)) {
						$node.find('li').removeClass(iconClass);
					}
				});
			}
		}
	});
})(Backbone, jQuery, _, MakeFormatBuilder);