var _ = require('underscore'),
	when = require('when'),
	PromiseObject = require('promise-object'),
	fs = require('fs'),
	crypto = require('crypto'),
	UglifyJS = require('uglify-js'),
	packer = require('packer'),
	dateFormat = require('dateformat');

var Builder = PromiseObject.create({
	initialize: function (changeset) {
		this._cache = {};

		this._changeset = changeset;
		this._timestamp = new Date();
		this._exclusions = [];
	},

	process: function ($deferred, sets, compression, exclusions) {

		this._exclusions = exclusions || [];

		when.map(sets, [this._processScript, compression]).done($deferred.resolve, $deferred.reject);
	},

	_processScript: function ($deferred, $self, set, compression) {
		if (compression) {
			this._concat(set)
				.then([this._pack, compression])
				.then(function (set) {
					set.source = $self._comment(set.files) + set.source;
					return set;
				})
				.then($deferred.resolve, $deferred.reject);
		} else {
			this._concat(set)
				.then(function (set) {
					set.source = $self._comment(set.files) + set.source;
					return set;
				})
				.then($deferred.resolve, $deferred.reject);
		}
	},

	_pack: function ($deferred, compression, set) {
		var packed = true;

		if (_.isObject(compression) && !_.isUndefined(compression.packed)) {
			packed = compression.packed;
		}

		var hash = this._hashObject({name: set.name, files: set.files, compressed: true, packed: packed});

		if (this._cache[hash]) {
			set.source = this._cache[hash];
			$deferred.resolve(set);
		} else {
			var source = UglifyJS.minify(set.source || '', {fromString: true, mangle: true}).code;

			if (packed) {
				source = packer.pack(source, true, true);
			}

			set.source = this._cache[hash] = source;

			$deferred.resolve(set);
		}
	},

	_concat: function ($deferred, $self, set) {
		var hash = this._hashObject({name: set.name, files: set.files});

		if (this._cache[hash]) {
			set.source = this._cache[hash];
			$deferred.resolve(set);
		} else {
			when.map(set.files, this._readFile).done(function (sources) {
				set.source = $self._cache[hash] = sources.join('\n\n');
				$deferred.resolve(set);
			}, $deferred.reject);
		}
	},

	_readFile: function ($deferred, $self, filepath) {
		var hash = this._hashObject(filepath);

		if (this._cache[hash] && !$self._exclusions) {
			return $deferred.resolve(this._cache[hash]);
		}

		fs.readFile(filepath, {encoding: 'utf8'}, function (error, data) {
			if (error) {
				return $deferred.reject(new Error(error));
			}

			// run through exclusions
			$self._exclusions.forEach(function (exclusion) {
				
				var re = new RegExp("<exclude="+exclusion+">",'ig');
				var blocks = data.toString().match(/\/\*\*[\s\S]*?\*\//gm);
				if (data.match(re)) {
					console.log('Warning! Code exclusions applied'.yellow,exclusion, filepath);
					var currIndex = 0;
					var indexes = [];
					var searchString = "//<exclude="+exclusion+">";
					
					while(data.indexOf(searchString,currIndex) > -1) {
						currIndex = data.indexOf("//<exclude="+exclusion+">",currIndex);
						endIndex = data.indexOf("//</exclude="+exclusion+">",currIndex) + searchString.length + 1;
						indexes.push({start:currIndex,end:endIndex,string:data.substr(currIndex, endIndex - currIndex)});
						currIndex++;
					}

					indexes.forEach(function (item) {					
						data = data.replace(item.string, "");
					})
				}
			})

			$deferred.resolve($self._cache[hash] = data);
		});
	},

	_hashObject: function (o) {
		return crypto.createHash('md5').update(typeof o === 'string' ? o : JSON.stringify(o)).digest('hex');
	},

	_comment: function(files) {
		return '' +
			'/**\n' +
			' * Copyright ' + this._timestamp.getFullYear() + ', Media Arts Lab\n' +
			' *\n' +
			' * USE OF THIS SOFTWARE IS STRICTLY LIMITED TO MEDIA ARTS LAB FOR APPLE WORK\n' +
			' * AND USE OF THIS SOFTWARE BY ANY OTHER PARTY IS PROHIBITED WITHOUT WRITTEN\n' +
			' * PERMISSION FROM MEDIA ARTS LAB.\n' +
			' *\n' +
			' * changeset ' + this._changeset + ' built on ' + dateFormat(this._timestamp, 'dd/mm/yy "at" H:MM:ss TT') + '\n' +
			' *\n' +
			files.map(function (filename) { return (' *     ' + filename + '\n'); }).join('') +
			' */\n\n';
	}
});

module.exports = Builder;