var exec = require('child_process').exec,
	glob = require('glob'),
	when = require('whenplus'),
	PromiseObject = require('promise-object'),
	util = require('util'),
	path = require('path'),
	FTP = require('ftp'),
	async = require('async'),
	malChangeset = require('node-changeset');

function promisify (func, scope) {
	return function () {
		var deferred = when.defer(),
			args = Array.prototype.slice.call(arguments);

		args.push(deferred.resolve);

		func.apply(scope, args);

		return deferred.promise;
	};
}

var Publish = PromiseObject.create({
	initialize: function ($self, $config, callback) {
		this._pipelines = $config.publish;
		this._changeset = new malChangeset({revision:$config.overrideRevision});

		this._changeset.getData().then(function () {
			when.mapUnfulfilled($self._pipelines, $self._publish).allLimit(1).done(function () {
					callback();
				}, function (error) {console.log(error.stack); });
		});

	},

	_publish: function ($deferred, $self, config) {
		console.log('publishing'.grey + ' ' + config.type);

		var rootFolder = config.year + ' ' + config.name,
			packageFolderName = config.name.replace(/ /g, '').toLowerCase();

		if (config.type === 'pointroll') {
			$self._connect(config).done(function () {
				$self._connection.mkdir(rootFolder, function () {
					$self._connection.cwd(rootFolder, function () {
						$self._connection.put(new Buffer('<?xml version="1.0" encoding="utf-8" ?><deploy><name>'+'r'+$self._changeset.active.revision+' - '+ config.name +'</name><notify>' + config.notify.join(';') + '</notify></deploy>'), 'deploy.xml', function () {
							$self._connection.mkdir('media', function () {
								$self._connection.cwd('media', function () {
									$self._connection.mkdir(packageFolderName, function () {
										$self._connection.cwd(packageFolderName, function () {
											$self._connection.mkdir('r' + $self._changeset.active.revision, function () {
												$self._connection.cwd('r' + $self._changeset.active.revision, function () {
													var files = $self._mapPaths(glob.sync('output/'+$self._changeset.active.branch+'_r' + $self._changeset.active.revision + '/pointroll/*'));

													$self._writeFiles(files).done(function () {
														$self._connection.cwd('/' + rootFolder, function () {
															$self._connection.put(new Buffer(''), 'done.flg', function () {
															 	$deferred.resolve();
															});
														});
													});
												});
											});
										});
									});
								});
							});
						});
					});
				});
			}, $deferred.reject);
		} else {
			$deferred.reject(new Error('Publishing instructions for ' + config.type + ' do not exist'));
		}
	},

	_getGit: function ($deferred) {
		
		exec('git rev-list HEAD --count',function (err,stdout,stderr) {
			if (err) throw new Error('Git/Hg Repo not found, use "--r 1" to set the changeset manually');
			$deferred.resolve(parseInt(stdout)-1);
		});
	},

	_getMercurial: function ($deferred) {
		
		exec('hg tip',function (err,stdout,stderr) {
			if (err) return $deferred.reject();			
			var changeset = stdout.match(/changeset\:\s{3}(\d+)\:/);
			$deferred.resolve(changeset ? changeset[1] : -1);
		});		
	},

	_setChangeset: function ($deferred,$self) {
		
		if (this._changeset) return $deferred.resolve(this._changeset);

		this._getMercurial().then(function (changeset) {
			$self._changeset = changeset;
			$deferred.resolve(changeset);
		},function () {
			$self._getGit().then(function (changeset) {
				$self._changeset = changeset;
				$deferred.resolve(changeset);
			});
		});
	},

	_connect: function ($deferred, $self, config) {
		$self._connection = new FTP();

		$self._connection.on('ready', function() {
			$deferred.resolve();
		});

		$self._connection.connect({ host: config.server.host, user: config.server.username, password: config.server.password });
	},

	_ls: function ($deferred, $self, dir) {
		$self._connection.list(dir, function (error, iter) {
			var items = [];

			iter.on('error', function(e) {
				$self._connection.end();
				throw new Error('ERROR during list(): ' + util.inspect(e));
			});

			iter.on('entry', function(entry) {
				items.push(entry);
			});

			iter.on('success', function() {
				$deferred.resolve(items);
			});
		});
	},

	_lswalk: function ($deferred, $self, dir) {
		var response = {files: [], dirs: []};

		function walk (dir, callback) {
			var dirs = [];

			$self._ls(dir).done(function (items) {
				items.forEach(function (item) {
					if (item.type === 'd') {
						response.dirs.push(path.join(dir, item.name));
						dirs.push(item.name);
					} else if (item.type === '-') {
						response.files.push(path.join(dir, item.name));
					}
				});

				if (!dirs.length) {
					callback();
				} else {
					async.forEachSeries(
						dirs,
						function (current, callback) {
							walk(path.join(dir, current), function () {
								callback();
							});
						},
						callback
					);

				}
			});
		}

		walk(dir, function () {
			$deferred.resolve(response);
		});
	},

	_rmFiles: function ($deferred, files) {
		when.mapUnfulfilled(files.reverse(), promisify(this._connection.delete, this._connection))
			.allLimit(1)
			.done($deferred.resolve, $deferred.reject);
	},

	_rmDirs: function ($deferred, dirs) {
		when.mapUnfulfilled(dirs.reverse(), promisify(this._connection.rmdir, this._connection))
			.allLimit(1)
			.done($deferred.resolve, $deferred.reject);
	},

	_mkDirs: function ($deferred, dirs) {
		dirs = dirs.map(function (dir) {
			return dir.dest;
		});

		when.mapUnfulfilled(dirs, promisify(this._connection.mkdir, this._connection))
			.allLimit(1)
			.done($deferred.resolve, $deferred.reject);
	},

	_writeFile: function ($deferred, file) {
		console.log('uploading'.grey + ' ' + file.dest);

		this._connection.put(file.source, file.dest, function (err) {
			if (err) {
				$deferred.reject(err);
			}

			$deferred.resolve();
		});
	},

	_writeFiles: function ($deferred, files) {
		when.mapUnfulfilled(files, this._writeFile)
			.allLimit(1)
			.done($deferred.resolve, $deferred.reject);
	},

	_recursiveDelete: function ($deferred, $self, path) {
		this._lswalk(path).done(function (items) {
			$self._rmFiles(items.files).done(function () {
				$self._rmDirs(items.dirs).done(function () {
					$deferred.resolve();
				});
			});
		});
	},

	_mapPaths: function ($self, paths) {
		return paths.map(function (source) {

			var regex = new RegExp('output/'+$self._changeset.active.branch+'_r'+$self._changeset.active.revision+'/[a-z]+/');
			var dest = source.replace(/\/$/,'').replace(regex,''),
				info = {source: source, dest: dest};

			return info;
		});
	}
});

module.exports = function(grunt) {
	grunt.registerTask('marcom-publish', 'Publish marcom project', function() {
		var callback = this.async(),
			publish = grunt.config('publish');

		new Publish({publish: publish,changeset:grunt.option('r')}, callback);
	});
};