/**
 * Copyright 2015, Media Arts Lab
 *
 * USE OF THIS SOFTWARE IS STRICTLY LIMITED TO MEDIA ARTS LAB FOR APPLE WORK
 * AND USE OF THIS SOFTWARE BY ANY OTHER PARTY IS PROHIBITED WITHOUT WRITTEN
 * PERMISSION FROM MEDIA ARTS LAB.
 *
 * changeset 217 built on 08/04/15 "aa" 10:55:00 AM
 *
 *     scripts/MALLibrary/ecma5.js
 *     scripts/MALLibrary/MAL.js
 *     scripts/MALLibrary/Util.js
 *     scripts/MALLibrary/Object.js
 *     scripts/MALLibrary/Set.js
 *     scripts/MALLibrary/Environment.js
 *     scripts/MALLibrary/Preloader.js
 *     scripts/MALLibrary/RMVProxy.js
 *     scripts/MALLibrary/Style.js
 *     scripts/MALLibrary/Stylesheet.js
 *     scripts/MALLibrary/ViewStyle.js
 *     scripts/MALLibrary/View.js
 *     scripts/MALLibrary/ViewController.js
 *     scripts/MALLibrary/TimeoutManager.js
 *     scripts/MALLibrary/ScriptLoader.js
 *     scripts/MALLibrary/RawTween.js
 *     scripts/MALLibrary/KeyframeAnimation.js
 *     scripts/MALLibrary/video/Base.js
 *     scripts/MALLibrary/video/Advanced.js
 *     scripts/MALLibrary/tween/Stage.js
 *     scripts/MALLibrary/tween/Layer.js
 *     scripts/MALLibrary/tween/LayerProxy.js
 *     scripts/MALLibrary/tween/CSSLayer.js
 *     scripts/MALLibrary/tween/CSSJSLayer.js
 *     scripts/MALLibrary/tween/JSLayer.js
 *     scripts/services/ImageLoader.js
 *     scripts/services/MarcomProxy.js
 *     scripts/services/InView.js
 *     scripts/services/MouseHotspot.js
 *     scripts/services/FrequencyCap.js
 *     scripts/services/Keyline.js
 *     scripts/services/Supertag.js
 *     scripts/services/ScrollRange.js
 *     scripts/views/Main.js
 *     scripts/views/RichMedia/Main.js
 *     scripts/controllers/Main.js
 *     scripts/controllers/RichMedia/Main.js
 *     scripts/bootstrap.js
 *     drivers/Default.js
 */

MALChangeset = '217';

/**
 * MAL Namespace
 * @class MAL
 */
if (!('bind' in Function.prototype)) {
	/**
	 * @private
	 */
	Function.prototype.bind = function(){
		var fn = this, args = Array.prototype.slice.call(arguments), object = args.shift();
		return function(){ return fn.apply(object, args.concat(Array.prototype.slice.call(arguments)));	};
	};
}

// Copyright (c) Mozilla Foundation http://www.mozilla.org/ This code is available under the terms of the MIT License
if (!Array.prototype.map) {
	/**
	 * @private
	 */
	Array.prototype.map = function(fun /*, thisp*/) {
		var len = this.length;
		if (typeof fun !== "function") {
			throw new TypeError();
		}

		var res = new Array(len);
		var thisp = arguments[1];
		for (var i = 0; i < len; i++)
		{
			if (i in this) {
				res[i] = fun.call(thisp, this[i], i, this);
			}
		}

		return res;
	};
}

// Copyright (c) Mozilla Foundation http://www.mozilla.org/ This code is available under the terms of the MIT License
if (!Array.prototype.forEach) {
	/**
	 * @private
	 */
	Array.prototype.forEach = function(fun /*, thisp*/) {
		var len = this.length >>> 0;
		if (typeof fun !== "function") {
			throw new TypeError();
		}

		var thisp = arguments[1];
		for (var i = 0; i < len; i++) {
			if (i in this) {
				fun.call(thisp, this[i], i, this);
			}
		}
	};
}

// Copyright (c) Mozilla Foundation http://www.mozilla.org/ This code is available under the terms of the MIT License
if (!Array.prototype.indexOf) {
	/**
	 * @private
	 */
	Array.prototype.indexOf = function (searchElement /*, fromIndex */ ) {
		"use strict";
		if (this === void 0 || this === null) {
			throw new TypeError();
		}
		var t = this;
		var len = t.length >>> 0;
		if (len === 0) {
			return -1;
		}
		var n = 0;
		if (arguments.length > 0) {
			n = Number(arguments[1]);
			if (n !== n) { // shortcut for verifying if it's NaN
				n = 0;
			} else if (n !== 0 && n !== Infinity && n !== -Infinity) {
				n = (n > 0 || -1) * Math.floor(Math.abs(n));
			}
		}
		if (n >= len) {
			return -1;
		}
		var k = n >= 0 ? n : Math.max(len - Math.abs(n), 0);
		for (; k < len; k++) {
			if (k in t && t[k] === searchElement) {
				return k;
			}
		}
		return -1;
	};
}

if (!Function.prototype.bind) {
	// adds param support to setTimeout and setInterval older engines
	//
	//     var timeoutID = window.setTimeout(func, delay, [param1, param2, ...]);
	//     var intervalID = window.setInterval(func, delay, [param1, param2, ...]);
	(function () {

		function helper (method) {
			return function (func, timeout) {
				var args = Array.prototype.slice.call(arguments, 2);
				return method(function () {
					func.apply(null, args);
				}, timeout);
			};
		}

		window.setTimeout = helper(window.setTimeout);
		window.setInterval = helper(window.setInterval);
	}(window));
}

/**
 * @class MAL
 * @singleton
 */
window.MAL = window.MAL || {
	define: function (path, value) {
		var current = this;

		path = path.split('.');

		path.forEach(function (segment, key) {
			if (!MAL.isObject(current)) {
				throw new Error('trying to store a value into a ' + typeof current);
			} else if (!(segment in current)) {
				current = current[segment] = key === path.length-1 ? value : (Object.create ? Object.create(null) : {});

				if (value.prototype) {
					value.prototype.objectClassName = segment;
					value.prototype.objectClassPath = path;
					value.prototype.objectClassPathToken = path.join('.').toLowerCase();
				}
			} else {
				if (key === path.length-1) {
					current[segment] = value;
				} else {
					current = current[segment];
				}
			}
		}, this);
	},

	get: function (path) {
		var current = this;

		path = path.split('.');

		for (var key = 0, segment; key < path.length; key++) {
			segment = path[key];

			if (key === path.length-1) {
				return current[segment];
			} else {
				if (!current[segment]) return undefined;
				current = current[segment];
			}
		}

		return undefined;
	}
};

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Util namespace. Provides you with a set of common functions and helpers. **Majority of Util methods are also added directly to the MAL namespace, for example you can use MAL.log instead of MAL.Util.log**
 *
 * # Usage
 *
 * lets pretend that you have a function that accepts a string, array or object as the first param and does different things based on the type
 *
 *     function something (param) {
 *      if (MAL.Util.isString(param)) {
 *       // param is a string
 *      } else if (MAL.Util.isArray(param)) {
 *       // param is a array
 *      } else if (MAL.Util.isObject(param)) {
 *       // param is  object
 *      }
 *     }
 *
 * as you can see its very useful for type checking, another useful method is parseNumber it actually returns 0 instead of NaN
 *
 *     var ensuredNumber = MAL.Util.parseNumber(param);
 *
 * also the Util namespace is house of the create method which was used to create {@link MAL.Object} you can use it like this
 *
 *     var Person = MAL.Util.create({
 *      initialize: function () {
 *       // constructor
 *      }
 *     });
 *
 * this is an ever-growing list of methods, feel free to add to it
 *
 * @class MAL.Util
 * @singleton
 */

MAL.Util = function () {};

// Async helper
MAL.Util.Async = {
	parallel: function (tasks, callback) {
		var finished = 0;

		tasks.map(function (value) {
			return function () {
				value(function () {
					finished++;
					if (finished === tasks.length) callback();
				});
			};
		}).forEach(function (value) { value(); });
	},
	serial: function (tasks, callback) {
		tasks = tasks.map(function (value) {
			return function () {
				value(next);
			};
		});

		function next () {
			if (tasks.length) {
				tasks.shift()();
			} else {
				callback();
			}
		}
		next();
	}
};

/**
 * binds all methods to scope
 *
 *     MAL.bindAll(this, 'doThis', 'doThat');
 *
 * @param {Object} scope
 * @return {Function...} methods
 */
MAL.Util.bindAll = function () {
	var self = arguments[0];
	for (var i = 1; i < arguments.length; i++) self[arguments[i]] = self[arguments[i]].bind(self);
};

/**
 * check if the param is undefined
 *
 * @param {Object} o
 * @return {Boolean}
 */
MAL.Util.isUndefined = function (o) {
	return o === void 0;
};

/**
 * check if the param is a {@link Function}
 *
 * @param {Function} o
 * @return {Boolean}
 */
MAL.Util.isFunction = function (o) {
	return !!(o && o.constructor && o.call && o.apply);
};

/**
 * check if the param is a {@link Array}
 *
 * @param {Array} o
 * @return {Boolean}
 */
MAL.Util.isArray = function (o) {
	return Object.prototype.toString.call(o) === '[object Array]';
};

/**
 * check if the param is a {@link Date}
 *
 * @param {Date} o
 * @return {Boolean}
 */
MAL.Util.isDate = function (o) {
	return Object.prototype.toString.call(o) === '[object Date]';
};

/**
 * check if the param is a {@link RegExp}
 *
 * @param {RegExp} o
 * @return {Boolean}
 */
MAL.Util.isRegExp = function (o) {
	return Object.prototype.toString.call(o) === '[object RegExp]';
};

/**
 * check if the param is a {@link String}
 *
 * @param {String} o
 * @return {Boolean}
 */
MAL.Util.isString = function (o) {
	return !!(o === '' || o && o.charCodeAt && o.substr);
};

/**
 * check if the param is null
 *
 * @param {Object} o
 * @return {Boolean}
 */
MAL.Util.isNull = function (o) {
	return o === null;
};

/**
 * check if the param is NaN
 *
 * @param {Number} o
 * @return {Boolean}
 */
MAL.Util.isNaN = function (o) {
	return o !== o;
};

/**
 * check if the param is a {@link Number}
 *
 * @param {Number} o
 * @return {Boolean}
 */
MAL.Util.isNumber = function (o) {
	return typeof o === 'number';
};

/**
 * check if the param is a {@link Object}
 *
 * @param {Object} o
 * @return {Boolean}
 */
MAL.Util.isObject = function (o) {
	return !(MAL.isNumber(o) || MAL.isNaN(o) || MAL.isNull(o) || MAL.isString(o) || MAL.isArray(o) || MAL.isFunction(o) || MAL.isUndefined(o));
};

/**
 * check if the param is a {@link Boolean}
 *
 * @param {Boolean} o
 * @return {Boolean}
 */
MAL.Util.isBoolean = function (o) {
	return o === true || o === false || Object.prototype.toString.call(o) == '[object Boolean]';
};

/**
 * shorten a number to the specified decimal points
 *
 *     MAL.Util.shortenFloat(1.0011111, 3); // returns 1.001
 *
 * @param {Number} number
 * @param {Number} points
 * @return {Number}
 */
MAL.Util.shortenFloat = function (number, points) {
	return Math.round(number*Math.pow(10,points))/Math.pow(10,points);
};

/**
 * converts an array-like object into an array
 *
 * @param {Object} arrayLike
 * @return {Array}
 */
MAL.Util.toArray = function (arrayLike) {
	// HACK: issue with StaticNodeList
	if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
		return Array.prototype.map.call(arrayLike, function (item) {return item;});
	}

	return Array.prototype.slice.call(arrayLike, 0);
};

/**
 * retuns min number inside of the array
 *
 * @param {Array} numbers
 * @return {Number}
 */
MAL.Util.arrayMax = function (array) {
	return Math.max.apply(null, array);
};

/**
 * retuns max number inside of the array
 *
 * @param {Array} numbers
 * @return {Number}
 */
MAL.Util.arrayMin = function (array) {
	return Math.min.apply(null, array);
};

/**
 * parse the float out of a string, and if it is NaN return 0
 *
 *     MAL.Util.parseNumber(" 1.01 "); // returns 1.01 as a float
 *
 * @param {String} o
 * @return {Number}
 */
MAL.Util.parseNumber = function (string) {
	var number = parseFloat(string);

	if (MAL.isNaN(number)) {
		return 0;
	} else {
		return number;
	}
};

/**
 * clones an object or array of objects
 *
 *     MAL.Util.clone({name: 'joe', details: {age: 24}}, true); // returns a deep clone of the object
 *
 * @param {Object} src
 * @param {Boolean} deep
 * @return {Number}
 */
MAL.Util.clone = function (src, deep) {
    if(!src || MAL.isNumber(src) || MAL.isString(src) || MAL.isBoolean(src) || MAL.isNaN(src)){
        return src;
    }

	// Number
    if(MAL.isNumber(src)){
        return src;
    }

    //Honor native/custom clone methods
    if(src.clone && MAL.isFunction(src.clone)){
        return src.clone(deep);
    }

    //DOM Elements
    if(src.nodeType && MAL.isFunction(src.cloneNode)){
        return src.cloneNode(deep);
    }

    //Date
    if(MAL.isDate(src)){
        return new Date(src.getTime());
    }

    //RegExp
    if(MAL.isRegExp(src)){
        return new RegExp(src);
    }

    //Function
    if(MAL.isFunction(src)){
        //Wrap in another method to make sure == is not true;
        //Note: Huge performance issue due to closures, comment this :)
        return (function(){
            src.apply(this, arguments);
        });

    }

    var ret, index;
    //Array
    if(MAL.isArray(src)){
        //[].slice(0) would soft clone
        ret = src.slice();
        if(deep){
            index = ret.length;
            while(index--){
                ret[index] = MAL.clone(ret[index], true);
            }
        }
    } else {
        ret = src.constructor ? new src.constructor() : {};
        for (var prop in src) {
            ret[prop] = deep ? MAL.clone(src[prop], true) : src[prop];
        }
    }

    return ret;
};

/**
 * iterates objects and arrays
 *
 *     MAL.Util.each([1,2,3], function (value, key, list) {
 *
 *     });
 *
 *     MAL.Util.each({a: 'a', b: 'b', c: 'c'}, function (value, key, list) {
 *
 *     });
 *
 * @param {Object} o
 * @param {Object} callback
 */
MAL.Util.each = function (o, callback) {
	if (MAL.isArray(o)) {
		Array.prototype.forEach.call(o, callback);
	} else {
		for (var key in o) if (o.hasOwnProperty(key)) callback(o[key], key, o);
	}
};

/**
 * retuns trimmed string
 *
 * @param {String} string
 * @return {String}
 */
MAL.Util.trim = (function () {
	var trimRegExp = /^\s+|\s+$/g;
	return function (string) {
		return string.replace(trimRegExp,'');
	};
}());

/**
 * easy way to ensure a delta from strings/floats
 *
 *     MAL.Util.delta("1", "5"); // returns 4
 *     MAL.Util.delta(1, "5"); // returns 4
 *     MAL.Util.delta(1, 5); // returns 4
 *
 * @param {String/Number} start
 * @param {String/Number} finish
 * @return {String}
 */
MAL.Util.delta = function (start, finish) {
	return MAL.parseNumber(finish) - MAL.parseNumber(start);
};

/**
 * Prototype Inheritance
 *
 *     var Person = MAL.Util.create({
 *      initialize: function () {
 *       // constructor
 *      }
 *     });
 *
 * @param {Object} prototype
 * @return {Function} object
 */
MAL.Util.create = function (proto) {
	var Self = this,
		BaseClass = function (_MAL) { // only if the argument doesnt match MAL
			if (MAL !== _MAL) {
				for (var method in this) {
					if (MAL.isFunction(this[method])) {
						this[method] = this[method].bind(this);
					}
				}
				if (MAL.isFunction(this.initialize)) {
					this.initialize.apply(this, arguments);
				}
			}
		};

	BaseClass.prototype = new Self(MAL); // private method as magic cookie

	function makeMethod (key) {
		(function (func, superFunc) {
			BaseClass.prototype[key] = !MAL.isFunction(func) || !MAL.isFunction(superFunc) ? func : function () {this._super = superFunc;return func.apply(this, arguments);};
		}(proto[key], BaseClass.prototype[key]));
	}

	for (var key in proto) makeMethod(key);

	BaseClass.prototype.constructor = BaseClass;
	BaseClass.extend = this.extend || this.create;
	return BaseClass;
};

/**
 * ensures that you have a real array, this is mainly useful if you want to support multiple arguments and a single array arguemnt
 *
 *     var tweens = MAL.Util.argumentsToArray(arguments);
 *     tweens.forEach(function (tween) {});
 *
 * @param {Array} arguments
 * @return {Array} arguments
 */
MAL.Util.argumentsToArray = function (args) {
	if (args.length === 1 && MAL.isArray(args[0])) {
		return args[0];
	} else {
		return Array.prototype.slice.call(args, 0);
	}
};

/**
 * converts an object to an array of objects
 *
 * @param {Object} object
 * @return {Array} arguments
 */
MAL.Util.objectToArray = function (o) {
	var array = [];
	for (var property in o) array.push({name: property, value: o[property]});
	return array;
};

/**
 * easy way to camel case a string with a lowercase prefix
 *
 *     MAL.Util.toCamelCaseWithPrefix('webkit', 'some-random-css-property'); // returns "webkitSomeRandomCssProperty"
 *
 * @param {String} prefix
 * @param {String} string
 * @return {String}
 */
MAL.Util.toCamelCaseWithPrefix = function (prefix, string) {
	return MAL.toCamelCase(prefix + '-' + string);
};

/**
 * easy way to camel case a string
 *
 *     MAL.Util.toCamelCase('some-random-css-property'); // returns "someRandomCssProperty"
 *
 * @param {String} string
 * @return {String}
 */
MAL.Util.toCamelCase = function (string) {
	for (var i = 0, result = '', split = string.split('-'); i < split.length; i++) result += i ? split[i].charAt(0).toUpperCase() + split[i].slice(1) : split[i];
	return result;
};

/**
 * easy way to convert CamelCase to hyphen-delmited strings
 * @param  {String} string
 * @return {String} string
 */
MAL.Util.toHyphenDelimited = function (string) {
	return string.replace(/([A-Z])/g, function (string, match, index) {
		return (index > 0 ? '-' : '') + match.toLowerCase();
	});
};


/**
 * lets you specify a base object and extend upon it
 *
 *     MAL.mergeOptions({
 *      one: 1,
 *      three: 3
 *     }, {one: 'NOT ONE', two: 2}); // returns {one: 'NOT ONE', two: 2, three: 3}
 *
 * this is mainly used to providing default options to classes
 *
 * @param {Object} defaults
 * @param {Object...} objects to mixin
 * @return {Object} merged options
 */
MAL.Util.mergeOptions = function () {
	var options = arguments[0];

	for (var i = 1, l = arguments.length; i < l; i++)
		for (var name in arguments[i]) options[name] = arguments[i][name];

	return options;
};

/**
 * allows you to provide a default value, and if the new value specified is undefined it fallsback
 *
 *     MAL.defaultValue(1, undefined); // 1
 *     MAL.defaultValue(1, 2); // 2
 *
 * @param {Object} fallback value
 * @param {Object} new value
 * @return {Object} value
 */
MAL.Util.defaultValue = function (fallback, value) {
	return MAL.isUndefined(value) ? fallback : value;
};


/**
 * returns a percent based on a cubic bezier applied to a percent
 *
 * @param {Number} percent
 * @param {Number} p1x
 * @param {Number} p1y
 * @param {Number} p2x
 * @param {Number} p2y
 */
MAL.Util.cubicBezierWithPercent = (function () {
	var ax, bx, cx, ay, by, cy, t, duration;

	function _solve (x, epsilon) {
		return _sCY(_solveCX(x,epsilon));
	}

	function _solveCX (x, epsilon) {
		var t0,t1,t2,x2,d2,i;
			for(t2 = x, i = 0; i < 8; i++) {
				x2 = _sCX(t2)-x;
				if(_fabs(x2)<epsilon) {
					return t2;
				}
				d2 = _sCDX(t2);
				if(_fabs(d2)<1e-6) {
					break;
				}
				t2 = t2 - x2/d2;
			}

			t0 = 0.0;
			t1 = 1.0;
			t2 = x;

			if (t2 < t0) {
				return t0;
			}

			if (t2 > t1) {
				return t1;
			}

			while (t0 < t1) {
				x2 = _sCX(t2);
				if (_fabs(x2-x) < epsilon) {
					return t2;
				}
				if (x > x2) {
					t0 = t2;
				} else {
					t1 = t2;
				}
				t2 = (t1 - t0) * 0.5 + t0;
			}
			return t2; // Failure.
	}

	function _sCX (t) {
		return ((ax*t+bx)*t+cx)*t;
	}

	function _sCY (t) {
		return ((ay*t+by)*t+cy)*t;
	}

	function _sCDX (t) {
		return (3.0*ax*t+2.0*bx)*t+cx;
	}

	function _sE (duration) {
		return 1.0/(200.0*duration);
	}

	function _fabs (n) {
		if (n >= 0) {
			return n;
		} else {
			return 0 - n;
		}
	}

	return function (percent,p1x,p1y,p2x,p2y) {

		ax = bx = cx = ay = by = cy = 0;

		cx= 3.0 * p1x;
		bx= 3.0 * (p2x - p1x) - cx;
		ax= 1.0 - cx - bx;
		cy= 3.0 * p1y;
		by= 3.0 * (p2y - p1y) - cy;
		ay= 1.0 - cy - by;
		duration = 1;
		t = percent;

		return _solve(t, _sE(duration));
	};
}());

/**
 * protects html ID's and class names, it will also prefix images with the specified path and return a Set of the images that were used
 *
 *     var data = MAL.processHTML("<div id='Test'></div>", "images/");
 *     console.log(data.images);
 *     console.log(data.html);
 *
 * @param {String} html
 * @param {String} path path for images
 * @param {String} prefix prefix for class names and id's
 * @return {Object}
 */
MAL.Util.processHTML = function (html, path, prefix) {
	var self = this,
		srcRegExp = /src=['"](?!data:|http:|https:)([^'"]+)['"]/ig,
		matches,
		images = new MAL.Set();

	prefix = prefix || 'mal-ad-';

	// TODO: look into the reasoning behind this later
	html = String(html);

	// snag images that were used
	html = html.replace(srcRegExp, function () {
		var src = arguments[1];

		// high res / retina support
		if (!window.devicePixelRatio || window.devicePixelRatio === 1) src = src.replace(/@[0-9.]+x/,'');

		images.add(src);
		return "src='" + path + src + "'";
	});

	// prefix name spaces
	html = html.replace(/(id|class)=(['"])([^'"]+)['"]/ig, function () {
		var type = arguments[1],
			quote = arguments[2],
			items = arguments[3].split(' '),
			prefixRegexp = new RegExp('^(mal-|' + prefix + ')');

		for (var i = 0; i < items.length; i++) {
			if (!prefixRegexp.test(items[i])) items[i] = prefix + items[i];
		}

		return type + '=' + quote + items.join(' ') + quote;
	});

	// return html with namespaces
	return {
		html: html,
		images: images
	};
};

/**
 * gets the absolute elements position by adding up its ancestors offsets
 *
 * @param {HTMLElement} element
 * @return {Object} positioning object {x: 0, y: 0}
 */
MAL.Util.findElementPosition = function (element) {
	var x = 0,
		y = 0;

	if (element.offsetParent) {
		do {
			x += element.offsetLeft;
			y += element.offsetTop;
		} while (!!(element = element.offsetParent));
	}
	return {x: x, y: y};
};

/**
 * get the document size (cross browser)
 *
 * @return {Object} size {width: 0, height: 0}
 */
MAL.Util.getDocumentSize = function () {
	return {
		width: document.width || Math.max(document.documentElement.clientWidth, document.body.scrollWidth, document.documentElement.scrollWidth, document.body.offsetWidth, document.documentElement.offsetWidth),
		height: document.height || Math.max(document.documentElement.clientHeight, document.body.scrollHeight, document.documentElement.scrollHeight, document.body.offsetHeight, document.documentElement.offsetHeight)
	};
};

/**
 * attaches an object of events to the specified element
 *
 * @param {HTMLElement} element
 * @param {Object} events
 */
MAL.Util.addEvents = function (element, events) {
	for (var type in events) MAL.addEvent(element, type, events[type]);
};

/**
 * attaches an event specified element with a callback
 *
 * @param {HTMLElement} element
 * @param {String} event
 * @param {Function} callback
 */
MAL.Util.addEvent = function (element, type, callback) {
	if (element.addEventListener) {
		element.addEventListener(type, callback, false);
	} else {
		element.attachEvent('on' + type, callback);
	}
};

/**
 * removes an object of events to the specified element
 *
 * @param {HTMLElement} element
 * @param {Object} events
 */
MAL.Util.removeEvents = function (element, events) {
	for (var type in events) MAL.removeEvent(element, type, events[type]);
};


/**
 * removes an event specified element with a callback
 *
 * @param {HTMLElement} element
 * @param {String} event
 * @param {Function} callback
 */
MAL.Util.removeEvent = function (element, type, callback) {
	if (element.removeEventListener) {
		element.removeEventListener(type, callback, false);
	} else {
		element.detachEvent('on' + type, callback);
	}
};

/**
 * easy way to log to the console without worrying about its existence
 *
 *     MAL.Util.log('this', 'is', 'a', 'test'); // logs out "this is a test"
 *
 * @param {String...} message
 */
MAL.Util.log = function() {
	if (MAL.Environment.browser !== 'adobeair' && (typeof console === 'undefined' || typeof console.log === 'undefined' || typeof MAL.DEBUG === 'undefined' || !MAL.DEBUG) && !document.getElementById('mal-logs')) return;

	if (MAL.Environment.browser === 'adobeair' || document.getElementById('mal-logs')) {
		var message = document.createElement('div');
		message.className = 'mal-ad-log';
		message.innerHTML = Array.prototype.slice.call(arguments, 0).join(' ');
		if (document.getElementById('mal-logs')) {
			document.getElementById('mal-logs').appendChild(message);
		} else {
			document.body.appendChild(message);
		}
	} else if (typeof console.log.apply !== 'undefined' && MAL.Environment.os !== 'iphone' && MAL.Environment.os !== 'ipad') {
		console.log.apply(console, arguments);
	} else {
		console.log(Array.prototype.slice.call(arguments, 0).join(' '));
	}
};

/**
 * will stringify and log any object without traversing
 *
 * @param {Object} item
 */
MAL.Util.logObject = function (e) {
	var props = {};
	for (var prop in e) if (MAL.isNumber(e[prop]) || MAL.isString(e[prop])) props[prop] = e[prop];
	MAL.log(JSON.stringify(props));
};

/**
 * cross browser dir logging
 *
 * @param {Object} item
 */
MAL.Util.dir = function() {
	if (typeof console === 'undefined' || typeof console.dir === 'undefined' || typeof MAL.DEBUG === 'undefined' || !MAL.DEBUG) return;
	if (typeof console.dir.apply !== 'undefined') {
		console.dir.apply(console, arguments);
	} else {
		console.dir(arguments[0]);
	}
};

/**
 * cross browser error logging
 *
 * @param {String} message
 */
MAL.Util.error = function(message) {
	if (typeof console === 'undefined' || typeof MAL.DEBUG === 'undefined' || !MAL.DEBUG) return;
	if (MAL.Environment.browser === 'firefox' && typeof console !== 'undefined' && !MAL.Util.isUndefined(console.error)) {
		console.error(message);
	} else {
		throw new Error(message);
	}
};

// expose all methods to parent namespace for convenience
(function () {
	for (var method in MAL.Util) if (method !== 'create') MAL[method] = MAL.Util[method];
})();

/**
 * @author Chad Scira <chad@mediaartslab.com>, Steve Cox <Steve.Cox@eu.mediaartslab.com>, Martijn Korteweg <martijn@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Object class. Provides a standard base object with custom event support.
 *
 * # Event Usage
 *
 *     // instantiate a extended Object
 *     var Person = MAL.Object.extend({
 *      initialize: function () {
 *       // call supers initialization method
 *       this._super();
 *      },
 *
 *      poke: function () {
 *       // dispatch poke event to listeners
 *       this.dispatchEvent('poke');
 *      }
 *     });
 *
 *     // instantiate person
 *     var person = new Person();
 *
 *     // subscribe to poke events
 *     person.addEventListener('poke', function () {
 *      MAL.log('poked');
 *     });
 *
 *     // poke the person
 *     person.poke();
 *
 * @class MAL.Object
 */
MAL.define('Object', MAL.Util.create({
	/**
	 * initialize
	 *
	 *     var Person = MAL.Object.extend({
	 *         // requires an initialize method
	 *         initialize: function () {
	 *             // call supers initialization method
	 *             this._super();
	 *         }
	 *     };
	 *
	 * @method constructor
	 */
	initialize: function () {
		this._listeners = {};
	},

	/**
	 * enable/disable all log calls to the object
	 */
	logging: false,

	/**
	 * enable/disable on any object to show event logging
	 */
	logEvents: false,

	/**
	 * enable/disable on any object to show event logging
	 */
	logEventsCollapsed: true,

	/**
	 * array of events to not log
	 */
	logEventsIgnore: [],

	/**
	 * enable/disable on any object event logging for events with no listeners
	 */
	logEventsNoListeners: true,

	/**
	 * object log method
	 * @return {Object...} message
	 */
	log: function () {
		if (!this.logging) return;

		if (MAL.Environment.browser === 'chrome') {
			console.log.apply(console, ['%c' + this.getFileAndLineNumberAtStackIndex(2) + '%c ' + this.objectClassPathToken + '%c', 'color: #de7400;', 'font-weight: bold', 'font-weight: normal'].concat(MAL.argumentsToArray(arguments)));
		} else {
			MAL.log.apply(MAL, ['Controller', '"' + this.objectClassPathToken + '"'].concat(MAL.argumentsToArray(arguments)));
		}
	},

	/**
	 * add an event listener to the object
	 *
	 *     this.addEventListener('poke', pokeFunction);
	 *
	 * @param {String} type
	 * @param {Function} listener
	 * @inheritable
	 */
	addEventListener: function(type, listener) {
		if (!MAL.isFunction(listener)) throw new Error ('the listener you tried to add for "' + (this.objectClassPathToken + ':' + type) + '" is not a function');

		type = this._prefixTypeWithPathToken(type);

		if (MAL.isUndefined(this._listeners[type])) {
			this._listeners[type] = [];
		}

		this._listeners[type].push(listener);
	},

	/**
	 * add multiple event listeners to the object
	 *
	 *     this.addEventListeners({poke: pokeFunction, punch: punchFunction});
	 *
	 * @param {Object} listeners
	 * @inheritable
	 */
	addEventListeners: function(listeners) {
		for (var type in listeners) this.addEventListener(type, listeners[type]);
	},

	/**
	 * remove an event listener from the object. if the function needs the scope binded this is a pain, and you may want to use {@link #removeAllEventListeners}
	 *
	 *     // you must pass in the original function (bind makes this a little less ideal)
	 *     this.removeEventListener('poke', originalFunction);
	 *
	 * all events are prefixed with their objects objectClassPathToken
	 *
	 * @param {String} type
	 * @param {Function} listener
	 * @inheritable
	 */
	removeEventListener: function(type, listener) {
		type = this._prefixTypeWithPathToken(type);

		if (MAL.isArray(this._listeners[type])){
			var listeners = this._listeners[type];
			for (var i=0, l=listeners.length; i < l; i++){

				if (listeners[i] === listener){
					listeners.splice(i, 1);
					break;
				}
			}
		}
	},

	/**
	 * remove an event listener of a specific type or all listeners in general from the object
	 *
	 *     // remove all subscribers of the 'poke' event
	 *     person.removeAllEventListeners('poke');
	 *
	 *     // remove all subscribers for all events
	 *     person.removeAllEventListeners();
	 *
	 * @param {String} type
	 * @inheritable
	 */
	removeAllEventListeners: function(type) {
		if (type) {
			delete this._listeners[type];
		} else {
			this._listeners = {};
		}
	},

	/**
	 * dispatches event to all listeners
	 *
	 *     // dispatch a basic event
	 *     this.dispatchEvent('poke');
	 *
	 *     // dispatch an event with details merged into the event object
	 *     this.dispatchEvent('poke', {enabled: true});
	 *
	 *     // dispatch an event with details and a original event object
	 *     this.dispatchEvent('poke', {enabled: true}, e);
	 *
	 * @param {String} event
	 * @param {Object} details
	 * @param {Object} originalEvent
	 * @inheritable
	 */
	dispatchEvent: function(event, details, originalEvent, proxy) {
		var loggingGroup;

		event = typeof event === 'string' ? { type: event } : event;
		event.type = this._prefixTypeWithPathToken(event.type);

		if (this.logging && this.logEvents && this.logEventsIgnore.indexOf(event.type) === -1 && (!(event.type.indexOf(':') && (this.logEventsIgnore.indexOf(event.type.split(':')[1]) !== -1)))) {
			if (proxy) {
				if (MAL.Environment.browser === 'chrome') {
					console.log('%cProxiedEvent%c ' + proxy.replace('controllers.', '') + ' %csent%c ' + event.type + ' %cto%c ' + this.objectClassPathToken.replace('controllers.', ''), 'color: #00993d;', 'font-weight: bold', 'font-weight: normal', 'font-weight: bold', 'font-weight: normal', 'font-weight: bold', details || '');
				} else {
					MAL.log('ProxiedEvent (' + proxy.replace('controllers.', '') + ') sent (' + event.type + ') to (' + this.objectClassPathToken.replace('controllers.', '') + ')');
				}
			} else {

				if (MAL.Environment.browser === 'chrome') {
					loggingGroup = ['%c' + (this.getFileAndLineNumberAtStackIndex(2) || 'Event') + '%c dispatched%c ' + event.type , 'color: #00993d;', 'color: #000; font-weight: normal;', 'color: #000; font-weight: bold;', details || ''];
				} else {
					MAL.log('Event (' + event.type + ')');
				}
			}
		}

		event.target = (!event.target) ? this : event.target;
		event.timestamp = new Date().getTime();
		event.details = details;
		if (!event.type){
			throw new Error('missing \'type\' property');
		}

		// attach original event
		if (typeof originalEvent !== 'undefined') event.originalEvent = originalEvent;

		// add details to event
		if (typeof details !== 'undefined') for (var detail in details) event[detail] = details[detail];

		this._callListeners(event, loggingGroup);
	},

	getFileAndLineNumberAtStackIndex: function (index) {
		var stack = new Error().stack,
			paths = stack.match(/\([^()]+\)/g),
			path;

		for (var i = 0; i < paths.length; i++) {
			path = paths[i];
			path = path.replace(/\(.*\/(.*)\)$/, '$1');
			path = path.replace(/\:\d*$/, '');

			if (i === index) {
				return path;
			}
		}
	},

	/**
	 * choose events to proxy to a specific object or array of objects. proxied events will trigger listeners on objects you are proxying to.
	 *
	 *     // proxy event to object
	 *     this.proxyEvents('poke', otherObject);
	 *     otherObject.addEventListener('poke', function);
	 *
	 * @param {Array/Object} events
	 * @param {Array/Object} targets
	 */
	proxyEvents: function (events, targets) {
		events = MAL.isArray(events) ? events : [events];
		targets = MAL.isArray(targets) ? targets : [targets];

		var self = this;

		events.forEach(function (event) {
			if (event.type) {
				event.type = self._prefixTypeWithPathToken(event.type);
			} else {
				event = self._prefixTypeWithPathToken(event);
			}

			this.addEventListener(event.type || event, function (object, event, details) {
				targets.forEach(function (target) {
					target.dispatchEvent.call(target, event.type, details, event, self.objectClassPathToken);
				});
			});
		}, this);
	},

	/**
	 * @private
	 */
	_prefixTypeWithPathToken: function (type) {
		type = type.match(':') ? type : this.objectClassPathToken + ':' + type;
		type = type.replace('controllers.', '');
		return type;
	},

	/**
	 * @private
	 */
	_callListeners: function(event, loggingGroup) {
		if (this._listeners[event.type] instanceof Array){

			if (loggingGroup) console[this.logEventsCollapsed ? 'groupCollapsed' : 'group'].apply(console, loggingGroup);

			var listeners = this._listeners[event.type];
			for (var i=0, l=listeners.length; i < l; i++){
				var details = event.details;
				delete event.details;

				try {
					listeners[i](this, event, details);
				} catch (error) {
					if (MAL.Environment.browser === 'chrome') {
						console.warn('%cerror encountered when throwing (' + event.type + ')\n\n%c' + error.stack, 'color: #ff0000;', 'color: #bcbab3');
					} else {
						if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
							MAL.log('encountered (' + error.message + ') when throwing (' + event.type + ')');
						} else {
							MAL.error('error encountered when throwing (' + event.type + ')\n\n' + error.stack);
						}
					}
				}
			}

			if (loggingGroup) console.groupEnd.apply(console, loggingGroup);
		} else {
			if (loggingGroup && this.logEventsNoListeners) {
				loggingGroup[1] = 'color: #e10000';
				console.log.apply(console, loggingGroup);
			}
		}
	}
}));

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Set class. Provides an array like object that ensures that all of its items are unique.
 *
 * # Usage
 *
 *     // adding and obtaining the layer
 *     var items = new MAL.Set();
 *     items.add('test');
 *     items.add('test');
 *     console.log(items.length); // should return 1 because a duplicate was added
 *
 * @class MAL.Set
 */
MAL.define('Set', MAL.Util.create({
	initialize: function () {
		// set length
		this.length = 0;

		// value cache
		this._values = [];

		// add items
		if (arguments.length) this.add.apply(this, arguments);
	},

	/**
	 * add items to set
	 *
	 *     items.add('test');
	 *
	 * @param {Object.../Number.../String...} items
	 */
	add: function (items) {
		if (!MAL.isArray(items)) items = Array.prototype.slice.call(arguments, 0);

		for (var i = 0; i < items.length; i++) {
			// item already exists
			if (this._values.indexOf(items[i]) !== -1) continue;

			// add to value cache
			this._values.push(items[i]);

			// update length
			this.length = this._values.length;
		}
	},

	/**
	 * remove item from set
	 *
	 *     items.remove('test');
	 *
	 * @param {Object.../Number.../String...} items
	 */
	remove: function (items) {
		if (!MAL.isArray(items)) items = Array.prototype.slice.call(arguments, 0);

		for (var i = 0, exists = -1; i < items.length; i++) {
			exists = this._values.indexOf(items[i]);

			// item doesn't exist
			if (exists === -1) continue;

			// remove item
			this._values.splice(exists, 1);

			// update length
			this.length = this._values.length;
		}
	},

	/**
	 * checks if set has item
	 *
	 *     items.has('test');
	 *
	 * @param {Object/Number/String} item
	 */
	has: function (item) {
		return this._values.indexOf(item) !== -1;
	},

	/**
	 * traverse set
	 *
	 *     items.forEach(function (value, key, array) {
	 *      console.log(value, key, array);
	 *     });
	 *
	 * @param {Function} callback
	 */
	forEach: function (callback, self) {
		this._values.forEach(callback, self);
	},

	/**
	 * map
	 *
	 *     var mapped = items.map(function (value) {
	 *      return value;
	 *     });
	 *
	 * @param {Function} callback
	 */
	map: function (callback, self) {
		return this._values.map(callback, self);
	},

	clone: function () {
		return this._values.slice(0);
	}
}));


/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * @class MAL.Environment
 *
 * User Environment class. Provides an easy way to do browser and feature detection.
 *
 * # Browser detection
 *
 * To detect a users browser you would do this
 *
 *     if (MAL.Environment.browser == 'firefox') {
 *      // browser is firefox
 *     } else {
 *      // browser is not firefox
 *     }
 *
 * A common thing you may need to check for is if the browser is IE and is not running in a lower compatibility mode
 *
 *     if (MAL.Environment.browser == 'msie' && MAL.Environment.documentMode < 9) {
 *      // browser is IE and can not do newer IE features
 *     } else {
 *      // can do ecma5 related stuff
 *     }
 *
 * # Global bugs and features
 *
 * we have a list of {@link #bugs} and {@link #features} that you can toggle
 *
 * for example if you wanted to disable all 3d transformations
 *     MAL.Environment.bugs.transform3d = true; // disabling all 3d transformations
 *
 * or if backfaces on all transition elements are causing issues you can do this
 *     MAL.Environment.features.backfacePerformanceFix = false; // disable auto backface
 *
 * @singleton
 */
MAL.define('Environment', (function () {
	function create (vendor, platform, userAgent) {
		function browserSearchString (data) {
			for (var i=0;i<data.length;i++)	{
				var string = data[i].string,
					property = data[i].property;

				versionSearchString = data[i].versionSearch || data[i].identity;

				if (string && string.indexOf(data[i].subString) !== -1 || property)
					return data[i].identity;
			}
			return '';
		}

		function browserSearchVersion (dataString) {
			var index = dataString.indexOf(versionSearchString);
			if (index === -1) return;
			return parseFloat(dataString.substring(index+versionSearchString.length+1));
		}

		function osSearchVersion (dataString, os) {
			for (var i = 0; i < data.os.length; i++) {
				if (os.indexOf(data.os[i].subString) === -1) continue;
				if (data.os[i].version) {

					return data.os[i].version(dataString);
				}
			}

			return '';
		}

		//<exclude=localStorage>
		function checkLocalStorage(browser) {
			/*
			 * Temporaray hack for IE on Surface. We needed a quick solution. It looks like in
			 * Metro mode, the browser has a bug that doesn't allow localStorage to be modified
			 * or read. A more permanent solution is to include the if statement in the try/catch.
			 */
			if (browser && browser=== 'msie' && navigator.userAgent.match(/Touch/)) {
				return false;
			}
			if ('localStorage' in window && window['localStorage'] !== null) {
				try {
					var test = 'localtest-' + Math.round(Math.random()*1000);
					localStorage.setItem(test,1);
					localStorage.removeItem(test);
					return true;
				} catch(e) {}
			}
			return false;
		}
		//</exclude=localStorage>
		
		function getVendorPrefix() {
			var regex = new RegExp('^(' + data.prefixes.join('|') + ')(?=[A-Z])');

			// latest browsers should support this
			for(var property in element.style)
				if (regex.test(property))
					return property.match(regex)[0];

			// fallback incase we need to check a specific property
			for (var i in data.prefixes) {
				var prefix = data.prefixes[i];
				if (prefix + 'Opacity' in element.style)
					return prefix;
			}

			return '';
		}

		var CSSPropertyNameCache = {normal: {}, camel: {}};

		function getCSSPropertyName (property, doNotCamelCase) {
			var cache = doNotCamelCase ? 'normal' : 'camel';

			// return if we have already cached the property
			if (CSSPropertyNameCache[cache][property] !== undefined) {
				return CSSPropertyNameCache[cache][property];
			}

			// HACK Firefox: float returns false
			if (property === 'float') return CSSPropertyNameCache[cache][property] = property;

			// HACK Firefox: src returns false
			if (property === 'src') return CSSPropertyNameCache[cache][property] = property;

			// HACK Webkit: issues with non prefixed filter returning a false positive
			if (property === 'filter' && engine === 'webkit') {
				return CSSPropertyNameCache[cache][property] = getCSSPropertyName('-' + property, doNotCamelCase);
			}

			var propertyCamelCase = MAL.toCamelCase(property);

			if (property.substr(0, 1) !== '-' && propertyCamelCase in element.style) {
				return CSSPropertyNameCache[cache][property] = property;
			} else {
				var propertyName = property;

				if (propertyName.substr(0, 1) === '-') {
					propertyName = propertyName.substr(1);
				}

				propertyCamelCase = MAL.toCamelCaseWithPrefix(prefix, propertyName);

				if (propertyCamelCase in element.style) {
					return CSSPropertyNameCache[cache][property] = doNotCamelCase ? '-' + prefix.toLowerCase() + '-' + propertyName : propertyCamelCase;
				}
			}

			return CSSPropertyNameCache[cache][property] = false;
		}

		function appleOSVersionMatch (string) {
			var matches = string.match(/(?:OS|Mac[^;]+) ([0-9_.]+)/);
			return matches ? parseFloat(matches[1].split('_').slice(0,2).join('.')) : '';
		}

		var versionSearchString = '',
			element = document.createElement('div'),
			data = {
				prefixes: 'Webkit Moz O ms'.split(' '),
				engines: {
					applewebkit: 'webkit',
					safari: 'webkit',
					chrome: 'webkit',
					msie: 'trident',
					firefox: 'gecko',
					opera: 'webkit',
					opera_legacy: 'presto',
					adobeair: 'webkit'
				},
				browser: [
					{
						string: userAgent,
						subString: 'MSIE',
						identity: 'MSIE',
						versionSearch: 'MSIE'
					},
					{
						string: userAgent,
						subString: 'Trident',
						identity: 'Trident',
						versionSearch: 'Trident'
					},
					{
						string: userAgent,
						subString: 'OPR/',
						identity: 'Opera',
						versionSearch: 'OPR'
					},
					{
						string: userAgent,
						subString: 'Chrome',
						identity: 'Chrome'
					},
					{
						string: userAgent,
						subString: 'Opera',
						identity: 'opera_legacy',
						versionSearch: 'Version'
					},
					{
						string: userAgent,
						subString: 'AdobeAIR',
						identity: 'AdobeAIR'
					},
					{
						string: vendor,
						subString: 'Apple',
						identity: 'Safari',
						versionSearch: 'Version'
					},
					{
						string: userAgent,
						subString: 'Firefox',
						identity: 'Firefox'
					},
					{
						string: userAgent,
						subString: 'AppleWebKit',
						identity: 'AppleWebKit'
					},
					{
						string: userAgent,
						subString: 'AppleWebkit',
						identity: 'AppleWebkit'
					}
				],
				os: [
					{
						string: platform,
						subString: 'Win',
						identity: 'Windows',
						version: function (string) {
							var matches = string.match(/Windows NT ([0-9.]+)/i);
							return matches ? parseFloat(matches[1]) : '';
						}
					},
					{
						string: platform,
						subString: 'Mac',
						identity: 'Mac',
						version: appleOSVersionMatch
					},
					{
						string: platform,
						subString: 'Android',
						identity: 'Android',
						version: function (string) {
							var matches = string.match(/Android ([0-9.]+)/i);
							return matches ? matches[1] : '';
						}
					},
					{
						string: userAgent,
						subString: 'iPod',
						identity: 'iPod',
						version: appleOSVersionMatch
					},
					{
						string: userAgent,
						subString: 'iPhone',
						identity: 'iPhone',
						version: appleOSVersionMatch
					},
					{
						string: userAgent,
						subString: 'iPad',
						identity: 'iPad',
						version: appleOSVersionMatch
					},
					{
						string: platform,
						subString: 'Linux',
						identity: 'Linux'
					}
				],
				transitionEndEventNames: {
					webkit: 'webkitTransitionEnd',
					gecko: 'transitionend',
					presto: 'oTransitionEnd',
					trident: 'transitionend'
				},
				animationEndEventNames: {
					webkit: 'webkitAnimationEnd',
					gecko: 'animationend',
					presto: 'oAnimationEnd',
					trident: 'animationend'
				}
			},
			browser = browserSearchString(data.browser).toLowerCase(),
			browserVersion = browserSearchVersion(userAgent);

		// IE11+ pretends not to be IE
		if (browser === 'trident') {
			browser = 'msie';

			// assumes that they will bump the trident version every time there is a new IE version (this has worked since IE7)
			browserVersion += 4;
		}

		var	os = browserSearchString(data.os),
			osVersion = osSearchVersion(userAgent, os),
			prefix = getVendorPrefix(),
			engine = data.engines[browser],
			transitionEnd = data.transitionEndEventNames[engine],
			animationEnd = data.animationEndEventNames[engine],
			documentMode,
			mobile = false;

		// mobile os check
		if (userAgent.match(/\bMobile\b/)) mobile = true;
		if (os === 'Android') mobile = true;
		if (browser === 'opera_legacy') browser = 'opera';

		var result = {
			/**
			* @property {String} browser
			*
			* - safari
			* - chrome
			* - msie
			* - firefox
			* - opera
			*/
			browser: browser,

			/**
			* @property {String} engine
			*
			* - webkit
			* - trident
			* - gecko
			* - presto
			*/
			engine: engine,

			/**
			* @property {Boolean} mobile
			*/
			mobile: mobile,

			/**
			* @property {Number} browserVersion
			*
			* float value of the browser version
			*/
			browserVersion: browserVersion,

			/**
			* @property {String} os
			*
			* - windows
			* - mac
			* - ipod
			* - iphone
			* - ipad
			* - linux
			*/
			os: os.toLowerCase(),

			/**
			* @property {String} osVersion
			*
			* usually a numerical representation of the os version
			*/
			osVersion: osVersion,

			/**
			* @property {String} prefix
			*
			* - -webkit-
			* - -moz-
			* - -o-
			* - -ms-
			*/
			prefix: '-' + prefix.toLowerCase() + '-',

			/**
			* @property {String} domPrefix
			*
			* - Webkit
			* - Moz
			* - O
			* - ms
			*/
			domPrefix: prefix,

			/**
			* returns the correct property name if the browser supports it
			*
			*     if (MAL.Environment.getCSSPropertyName('transform')) {
			*      // logs out the browser prefixed transform css property (if its still prefixed)
			*      console.log(MAL.Environment.getCSSPropertyName('transform'));
			*     } else {
			*      // browser does not support css transformations
			*     }
			*
			* @param {String} property
			* @param {String} [doNotCamelCase] use if you want dashed based properties
			* @return {String/Boolean} false or the property
			*/
			getCSSPropertyName: getCSSPropertyName,

			/**
			* @property {String/Boolean} requestAnimationFrame
			*
			* provides the correct name of the browser's requestAnimationFrame method, or false if the browser doesn't support the method
			*/
			requestAnimationFrame: window[prefix.toLowerCase() + 'RequestAnimationFrame'] ? prefix.toLowerCase() + 'RequestAnimationFrame' : false,

			/**
			* @property {Boolean} has3d
			*
			* informs you if the browser supports 3d transformations
			*/
			has3d: getCSSPropertyName('perspective') && (engine !== 'webkit' || !window.matchMedia || window.matchMedia && window.matchMedia('(transform-3d),(-' + prefix.toLowerCase() + '-transform-3d)').matches),

			/**
			* @property {Boolean} hasCanvas
			*
			* informs you if the browser supports canvas
			*/
			hasCanvas: !!document.createElement('canvas').getContext,

			//<exclude=localStorage>
			/**
			* @property {Boolean} hasLocalStorage
			*
			* informs you if the browser supports localStorage
			*/
			hasLocalStorage: checkLocalStorage(browser),
			//</exclude=localStorage>
			
			/**
			* @property {Boolean} hasKeyframeAnimations
			*
			* informs you if the browser supports css keyframe animations
			*/
			hasKeyframeAnimations: !!getCSSPropertyName('animation-name'),

			/**
			* @property {Object} bugs namespace helper for maintaining browser bugs
			*
			* - transformations
			* - transform3d
			* - scale
			*/
			bugs: {
				transformations: false,
				transform3d: !!(browser === 'safari' && userAgent.match(/Version\/(4.0 Safari|4.0.1|4.0.2|4.0.5)/)),
				forceWebm: browser === 'firefox',
				scale: false // (browser == 'firefox' && browserVersion < 7) || (browser == 'msie' && browserVersion == 9)
			},

			/**
			* @property {Object} features namespace allows you to enable/disable features
			*
			* - backfacePerformanceFix
			* - perspectivePerformanceFix
			* - translateZPerformanceFix
			*/
			features: {
				// adds rotation to elements in firefox and ie9 to smooth animation
				//rotationPerformanceFix: false,

				// adds performance fixes relating to css backfaces, you need to disable this when you need to use backfaces and tweening
				backfacePerformanceFix: true,

				// adds performance fixes relating to css aliasing, you need to disable this when you need to use 3d perspective and tweening
				perspectivePerformanceFix: true,

				// adds translateZ to all css animations
				translateZPerformanceFix: true
			}
		};

		/**
		* @property {String} transitionEnd
		*
		* provides the correct name of the browser's transitionEnd event
		*/
		if (getCSSPropertyName('transition')) result.transitionEnd = transitionEnd;

		/**
		* @property {String} animationEnd
		*
		* provides the correct name of the browser's animationEnd event
		*/
		if (getCSSPropertyName('animation')) result.animationEnd = animationEnd;

		/**
		* @property {String} documentMode
		*
		* provides you with a consistent way to figure out what documentMode IE is in
		*/
		if (browser === 'msie') {
			if (document.documentMode) {
				// IE8 or later
				result.documentMode = document.documentMode;
			} else {
				// IE 5-7
				result.documentMode = document.compatMode && document.compatMode === 'CSS1Compat' ? 7 : 5;
			}
		}
		result.create = create;

		return result;
	}

	return create(navigator.vendor, navigator.platform, navigator.userAgent);
})());

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * @class MAL.Preloader
 * @extends MAL.Object
 *
 * Preloader class. Provides a way to preload a set of images.
 *
 * # Usage Example
 *
 * initialize preloader with an array of images
 *
 *     var preloader = new MAL.Preloader({
 *      // array of initial images
 *      images: ['one.png', 'two.png', 'three.png'],
 *
 *      // path for all images
 *      imagePath: 'images/',
 *
 *      // file path proxy method (required for google)
 *      filePathProxy: function (filepath, filename) {
 *       // every file request goes through this method, this is where you would add additional filepath logic
 *       return filepath + filename;
 *      }
 *     });
 *
 * add additional images after the preloader has been initialized (but before it has been instructed to start loading)
 *
 *     preloader.addImages(['four.png', 'five.png', 'six.png']);
 *
 * subscribe to the preloaders completion event
 *
 *     preloader.addEventListener('load', function (o,e) {
 *      console.log('all images have been loaded');
 *     });
 *
 * start preloading
 *
 *     preloader.load();
 *
 */

/**
 * dispatched when loading starts
 *
 * @event start
 * @param {MAL.Preloader} this
 */

/**
 * dispatched when an image has loaded starts
 *
 * @event image-loaded
 * @param {MAL.Preloader} this
 * @param {Object} image
 */

/**
 * dispatched when all the images are loaded
 *
 * @event load
 * @param {MAL.Preloader} this
 */
MAL.define('Preloader', MAL.Object.extend({
	/**
	 * initializes MAL.Preloader
	 *
	 *     var preloader = new MAL.Preloader({
	 *      // array of initial images
	 *      images: ['one.png', 'two.png', 'three.png'],
	 *
	 *      // path for all images
	 *      imagePath: 'images/',
	 *
	 *      // file path proxy method (required for google)
	 *      filePathProxy: function (filepath, filename) {
	 *       // every file request goes through this method, this is where you would add additional filepath logic
	 *       return filepath + filename;
	 *      }
	 *     });
	 *
	 * @method constructor
	 * @param {Object} $config
	 */

	/**
	 * @cfg {Function} filePathProxy
	 * event loop delay in milliseconds
	 */

	/**
	 * @cfg {String} imagePath
	 * event loop delay in milliseconds
	 */

	/**
	 * @cfg {Function} callback
	 *
	 * callback function that should be called upon completion
	 */

	/**
	 * @cfg {Boolean} autoload
	 *
	 * autoload the images upon initialization
	 */
	initialize: function ($config) {
		$config = $config || {};

		this._super();

		this._images = new MAL.Set();

		if ($config.imagePath) this._imagePath = $config.imagePath;
		if ($config.filePathProxy) this._filePathProxy = $config.filePathProxy;

		if (typeof MAL.rmv !== 'undefined') {
			this._imagePath = MAL.rmv.paths.images;
			this._filePathProxy = MAL.rmv.filePathProxy;
		}

		this._callback = $config.callback;

		// garbage collection fix for IE7?
		this.images = {};

		// add initial images if passed
		if ($config.images) this.addImages($config.images);
		if ($config.autoload) this.load();
	},

	/**
	 * image path
	 * @private
	 */
	_imagePath: '',

	/**
	 * image path resolution
	 *
	 * @private
	 * @param {String} filepath
	 * @param {String} filename
	 * @return {String} resolved filepath
	 */
	_filePathProxy: function (filepath, filename) {
		return filepath + filename;
	},

	/**
	 * Adds a set of images to the Preloader
	 *
	 *     preloader.addImages(['four.png', 'five.png', 'six.png']);
	 *
	 * @param {Array} images
	 */
	addImages: function (images) {
		images.forEach(function(value, key, ref) {
			this.addImage(value);
		}, this);
	},

	/**
	 * Adds a image to the Preloader
	 *
	 *     preloader.addImage('four.png');
	 *
	 * @param {String} image
	 */
	addImage: function (image) {
		this._images.add(image);
	},

	/**
	 * Starts the load process
	 *
	 *     preloader.load();
	 *
	 */
	load: function () {
		if (!this._images.length) {
			MAL.log('WARNING: empty image preloader');
			this._onImagesLoaded();
			return;
		}

		var self = this;

		this.dispatchEvent('start');

		MAL.Async.parallel(
			this._images.map(function(value, key, array) {
				var src = this._filePathProxy(this._imagePath, value);

				return function (callback) {
					var error,
						cache = new Image();

					if (cache.addEventListener) {
						// toss the image info in a closure
						error = (function (self, cache) {
							return function () {
								self._onImageError.apply(self, [cache]);
							};
						}(this, cache));
					} else {
						error = this._onImageError;
					}

					MAL.addEvents(cache, {
						load: function () {
							// DO NOT THROW LOAD EVENTS FOR DATA URI's
							if (cache.src.substr(0, 4) === 'data') return false;

							self.dispatchEvent('image-loaded', {image: cache});
							callback();
						},
						error: error,
						abort: error
					});

					// start loading
					cache.src = src;

					// hold image ref
					this.images[value] = cache;
				}.bind(this);
			}, this),
			this._onImagesLoaded
		);
	},

	/**
	 * @private
	 */
	_onImagesLoaded: function (e) {
		if (this._callback) this._callback();
		this.dispatchEvent('load');
	},

	/**
	 * @private
	 */
	_onImageError: function (image) {
		MAL.error('MALPreloader: can not load "' + image.src + '"');
	}
}));

/**
 * @author Martijn Korteweg <martijn@mediaartslab.com>
 * @docauthor Martijn Korteweg <martijn@mediaartslab.com>
 *
 * Used to pipe events from an internal source into a RMV
 *
 * @class MAL.RMVProxy
 * @extends MAL.Object
 * @private
 */

MAL.define('RMVProxy', MAL.Object.extend({
	initialize: function (RMV) {
		this._super();

		this.startTime = new Date().getTime();
		this.userDetails = MAL.Environment.os+'_'+MAL.Environment.browser+'_'+MAL.Environment.browserVersion;
		var _scope = this;
		this.RMV = RMV;
		this.RMVReady = false;
		this.queue = [];
		this.uniqueEvents = [];

		// copy some stuff to this object
		this.paths = RMV.paths;
		this.filePathProxy = RMV.filePathProxy;
		this.siteAnimations = RMV.siteAnimations || false;
	
		var self = this;		
		if (RMV.events) {
			MAL.each(RMV.events,function (value, key, ref) {				
				var shortName = key.slice(2).toLowerCase();
				self.addEventListener(shortName, function(o,e,c) {
					if (!self.RMVReady) {
						self.queue.push({eventKey:key,eventArgs:c});
						return;
					}

					self._runEventCall(key,c);
				}.bind(self), true);
			});
		} else {
			MAL.log('RMV does not implement any events');
		}

		this.RMVReadyCheck();
	},

	/**
	 * @private	 
	 */
	_runEventCall: function (eventKey, args) {
		
		if (this.RMV.events[eventKey].unique && this.uniqueEvents.indexOf(eventKey) !== -1) return;

		if (this.RMV.events[eventKey].unique) {
			this.uniqueEvents.push(eventKey);
		}

		if (MAL.ad && MAL.isFunction(MAL.ad.site[eventKey])) {
			MAL.ad.site[eventKey].call(this.RMV,args);
		}

		if (MAL.isFunction(this.RMV.events[eventKey])) {
			this.RMV.events[eventKey].call(this.RMV,args);	
		} else if (this.RMV.events[eventKey].handler) {
			this.RMV.events[eventKey].handler.call(this.RMV,args);
		} else {
			this.log('Event does not implement a handler');
		}		
	},

	/**
	 * if functionality is implemented this should make sure all events get run that are queued up.
	 */
	RMVReadyCheck: function () {
		if (this.RMV.isRMVReady) {
			if (this.RMV.isRMVReady()) {
				this.RMVReady = true;
				this.queue.forEach(function (value) {
					this._runEventCall(value.eventKey,value.eventArgs);					
				}, this);
			} else {
				setTimeout(this.RMVReadyCheck, 50);
			}
		} else {
			this.RMVReady = true;
		}
	}
}));

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Style instances are passed into {@link MAL.Stylesheet} via {@link MAL.Stylesheet#addStyle}
 *
 * ## example
 *
 *     var style = new MAL.Style({
 *      '.button': {
 *          width: 200,
 *          height: 200
 *      }
 *     });
 *
 * ## keyframe animations
 *
 *     var style = new MAL.Style({
 *       '@keyframes rotation-reverse': [
 *         {
 *             time: 'from',
 *             properties: {
 *                 transform: 'translateZ(0) rotate(0deg) scale(1)'
 *             }
 *         },
 *         {
 *             time: 'to',
 *             properties: {
 *                 transform: 'translateZ(0) rotate(-180deg) scale(0)'
 *             }
 *         }
 *       ]
 *     });
 *
 * @class MAL.Style
 * @extends MAL.Object
 */
MAL.define('Style', MAL.Object.extend({
	/**
	 * initializes MAL.Style
	 *
	 *     var style = new MAL.Style({
	 *       body: {
	 *         background: '#ccc'
	 *       }
	 *     });
	 *
	 * @method constructor
	 * @param {Object} config
	 */

	/**
	* @cfg {String} prefix
	* the className prefix (defaults to mal-ad-)
	*/

	/**
	* @cfg {Boolean} raw
	* if you set raw to true the rules will not have auto prefixed properties
	*/

	/**
	* @cfg {Object} defaults
	* an object of default rules
	*/

	/**
	* @cfg {Function} filePathProxy
	* event loop delay in milliseconds
	*/

	/**
	* @cfg {String} imagePath
	* event loop delay in milliseconds
	*/
	initialize: function (rules, $config) {
		this._super();

		$config = $config || {};

		this._images = new MAL.Set();
		this._prefix = $config.prefix || 'mal-ad-';
		this._selectorPrefix = $config.selectorPrefix || '';
		this._raw = $config.raw;
		this._defaults = $config.defaults || null;

		if ($config.imagePath) this._imagePath = $config.imagePath;
		if ($config.filePathProxy) this._filePathProxy = $config.filePathProxy;

		if (typeof MAL.rmv !== 'undefined') {
			this._imagePath = MAL.rmv.paths.images;
			this._filePathProxy = MAL.rmv.filePathProxy;
		}

		this.rules = {};
		this.animations = {};

		if (this._defaults) {
			this._addRulesWithDefaults(this._defaults, rules);
		} else {
			this._addRules(rules);
		}
	},

	INT_PROPERTIES: ['z-index', 'opacity'],

	/**
	 * image path
	 * @private
	 */
	_imagePath: '',

	/**
	 * image path resolution
	 *
	 * @private
	 * @param {String} filepath
	 * @param {String} filename
	 * @return {String} resolved filepath
	 */
	_filePathProxy: function (filepath, filename) {
		return filepath + filename;
	},

	/**
	 * add a rule to the rules array
	 *
	 * @private
	 * @param {String} selector
	 * @param {Object} properties
	 */
	_addRule: function (selector, properties, recursive) {
		if (selector.match(/^@keyframes/)) {
			return this._addKeyframeAnimation(selector, properties);
		}

		if (!this._raw && !recursive && this._prefix !== 'mal-' && !selector.match(/\:\:\:/) && selector.match(/[.#]mal\-/)) {
			return MAL.error('Style: bad selector (' + selector + ') you should never reference "mal-", use ::: instead');
		}

		if (selector.substr(0, 3) === ':::') {
			if (this._raw) return MAL.error('Style: bad selector (' + selector + ') you should not use ::: inside of a raw style object');

			return this._addRule(this._resolveStateWithSelector(selector), properties, true);
		} else if (selector.match(/^\:\:?[a-z]/)) {
			selector = this._selectorPrefix.slice(0, -1) + selector;

			return this._addRule(selector, properties, true);
		}

		// properly process the selector
		selector = selector.split(',').map(this._resolveSelector).join(', ');

		// track altered properties
		var _properties = this.rules[selector] || {};

		for (var property in properties) {
			if (MAL.isObject(properties[property])) {
		 		if (property.substr(0, 3) === ':::') {
					this._addRule(this._resolveStateWithSelector(property, selector), properties[property], true);
		 		} else {
		 			this._addRule(this._addParentSelectorToSelector(selector, property), properties[property], true);
		 		}
			} else {
				_properties[property] = this.resolveProperty(property, properties[property]) || undefined;
			}
		}

		// add to rules
		this.rules[selector] = _properties;
	},

	/**
	 * resolves a state selector (:::playing)
	 *
	 * @private
	 * @param {String} state
	 * @param {String} selector
	 */
	_resolveStateWithSelector: function (state, selector) {
		state = state.substr(3);
		selector = selector || '';

		var viewSelector = '.mal-view.' + this._prefix.substr(0, this._prefix.length-1);

		if (!state.match(/^mal/)) {
			// prepend with state
			state = state.split('.').map(function (state) {
				return 'state-' + state;
			}).join('.');

			if (!selector) {
				selector += this._resolveSelector(this._selectorPrefix.slice(0, -1) + '.' + this._prefix + state);
			} else {
				selector = selector.replace(viewSelector, viewSelector + '.' + this._prefix + state);
			}
		} else {
			if (!selector) {
				selector = this._resolveSelector('.' + state) + ' ' + viewSelector;
			} else {
				selector = this._resolveSelector('.' + state) + ' ' + selector;
			}
		}

		return selector;
	},

	/**
	 * resolves the original selector
	 *
	 * @private
	 * @param {String} selector
	 */
	_resolveSelector: function (selector) {
		if (this._prefix !== 'mal-') {
			if (/(^|\s|,)view/.test(selector)) {
				selector = selector.replace(/(^|\s|,)view/g, '$1.mal-view.' + this._prefix.substr(0, this._prefix.length-1));
			} else if (!selector.match(/.mal-/) && selector.match(/\.|\#/) && !selector.match(/\@instance/)) {
				selector = this._selectorPrefix + selector;
			}
		}

		if (MAL.isUndefined(this._raw)) selector = selector.replace(new RegExp('([#\\.])(?!mal-)([A-Z0-9\\-_]*)', 'ig'), '$1' + this._prefix + '$2');

		if (/\:\:-(\S+)/.test(selector)) {
			selector = selector.replace(/\:\:-(\S+)/, '::' + MAL.Environment.prefix + '$1');
		}

		return MAL.trim(selector);
	},

	/**
	 * adds the parent selectors to the selector
	 *
	 * @private
	 * @param {String} parent
	 * @param {String} selector
	 * @return {String} result
	 */
	_addParentSelectorToSelector: function (parent, selector) {
		return selector.split(',').map(function (item) {
			return parent + (item.substr(0, 1) === ':' ? '' : ' ') + item;
		}).join(',');
	},

	/**
	 * resolves a property so its crossbrowser
	 *
	 * @param {String} property
	 * @param {String} propertyValue
	 * @return {Object} resolved
	 */
	resolveProperty: function (property, propertyValue) {
		var methodName = MAL.toCamelCaseWithPrefix('_css', property);

		if (!MAL.isUndefined(this[methodName])) {
			return this[methodName](property, propertyValue);
		} else {
			var name = MAL.Environment.getCSSPropertyName(property, true),
				// auto append px if value is an int and does
				value = (MAL.Util.isNumber(propertyValue) && this.INT_PROPERTIES.indexOf(property) === -1) ? propertyValue + 'px' : propertyValue;

			// skip bugged or unsupported properties
			if (!name || (MAL.Environment.bugs.transform3d && (property === 'backface-visibility' || property === 'perspective'))) {
				return null;
			}

			// auto append px if value is an int and does
			return {
				name: name,
				value: value
			};
		}
	},

	/**
	 * add an animation to the animation object
	 *
	 * @private
	 * @param {String} selector
	 * @param {Object} list
	 */
	_addKeyframeAnimation: function (selector, list) {
		var identifier = selector.replace('@keyframes ', ''),
			keyframesName = (MAL.Environment.getCSSPropertyName('animation') === 'animation' ? '' : MAL.Environment.prefix) + 'keyframes',
			value = '';

		list.forEach(function (item) {
			value += item.time + ' {';
			for (var property in item.properties) {
				var resolved = MAL.Style.prototype.resolveProperty(property, item.properties[property]);
				if (resolved) value += resolved.name + ': ' + resolved.value + ';';
			}
			value += '}';
		}, this);

		delete this.rules[selector];
		this.animations['@' + keyframesName + ' ' + identifier] = value;
	},


	/**
	 * adds a set of rules to the rules array
	 *
	 * @private
	 * @param {Object} rules
	 */
	_addRules: function (rules) {
		// traverse arguments and run addRule on each item
		for (var rule in rules)
			this._addRule(rule, rules[rule]);
	},

	/**
	 * adds a set of rules with default rules to the rules array
	 *
	 * @private
	 * @param {Object} default rules
	 * @param {Object} rules
	 */
	_addRulesWithDefaults: function (defaults, rules) {
		// process rules with defaults
		for (var rule in rules)
			for (var defaultProperty in defaults)
				if (MAL.isUndefined(rules[rule][defaultProperty]))
					rules[rule][defaultProperty] = defaults[defaultProperty];

		this._addRules(rules);
	},

	/**
	 * background css property override
	 *
	 * @private
	 */
	_cssBackground: function (property, value) {
		// add no-repeat as default if it is not set
		if (!value.match(/repeat/)) value += ' no-repeat';
		return this._cssBackgroundImage(property, value);
	},

	/**
	 * background image property override
	 *
	 * @private
	 */
	_cssBackgroundImage: function (property, value) {
		var imageRegExp = /url\(['"]?(?!data:)([^'"]+)['"]?\)/,
			backgrounds = value.split(',');

		// multiple backgrounds
		backgrounds = backgrounds.map(function (background) {
			var image;

			if (imageRegExp.test(background)) {
				// fix src
				background = background.replace(imageRegExp, function (fullmatch, filename) {
					// high res / retina support
					if (!window.devicePixelRatio || window.devicePixelRatio === 1) {
						filename = image = filename.replace(/@[0-9.]+x/,'');
					} else {
						image = filename;
					}

					return 'url("' + this._filePathProxy(this._imagePath, filename) + '")';
				}.bind(this));

				// add to image set
				this._images.add(image);
			}

			return background;
		}, this);

		return {
			name: property,
			value: backgrounds.join(',')
		};
	},

	/**
	 * css opacity property override
	 *
	 * @private
	 */
	_cssOpacity: function (property, value) {
		if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			return {
				name: 'filter',
				value: 'Alpha(opacity=' + (value * 100) + ')'
			};
		} else {
			return {
				name: property,
				value: value
			};
		}
	},

	/**
	 * returns an array of all images used in the style
	 *
	 * @return {Array} images
	 */
	getImages: function () {
		return this._images.clone();
	}
}));

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Stylesheets class. Provides an easy way to interface with stylesheets in a cross browser way, and also tracks images. **Anything related to transformations or animated filters should be done through {@link MAL.tween.CSSLayer} or {@link MAL.tween.JSLayer}**.
 *
 * # Usage
 *
 * **NOTE: default functionality of the stylesheet is to prefix all ID's and classes with mal-ad-, in order to override this convenience you have to use {@link #addRawRules}**
 *
 * create a stylesheet object that manages all ad related styles
 *
 *     var stylesheet = new MAL.Stylesheet({
 *      callback: function(){
 *          // stylesheet ready
 *      }
 *     });
 *
 * add some basic rules
 *
 *     stylesheet.addRules({
 *      '#Box': {
 *       width: 50,
 *       height: 50,
 *       left: 0,
 *       top: 0,
 *       position: 'absolute'
 *      },
 *      '#Box2': {
 *       width: 50,
 *       height: 50,
 *       left: 50,
 *       top: 50,
 *       position: 'absolute'
 *      }
 *     });
 *
 * add some rules with defaults
 *
 *     stylesheet.addRulesWithDefaults(
 *      // defaults
 *      {
 *       left: 0,
 *       top: 0,
 *       position: 'absolute'
 *      },
 *
 *      // rules
 *      {
 *       // because of the defaults this rule will actually also include {left:0,top:0,position:'absolute'}
 *       '#Box': {},
 *
 *       // because of the defaults this rule will actually also include {left:0,top:0}
 *       '#Box2': {
 *        left: 50,
 *        top: 50
 *       }
 *      }
 *     );
 *
 * write css rules to the stylesheets
 *
 *     stylesheet.writeStyles()
 *
 * get images that have been used in the rules
 *
 *     var images = stylesheet.getImages();
 *
 * @class MAL.Stylesheet
 * @extends MAL.Object
 */
MAL.define('Stylesheet', MAL.Object.extend({
	/**
	 * initializes MAL.Stylesheet
	 *
	 *     var stylesheet = new MAL.Stylesheet({
	 *      callback: function(){
	 *          // stylesheet ready
	 *      }
	 *     });
	 *
	 * @method constructor
	 * @param {Object} config
	 */

	/**
	* @cfg {String} callback
	* callback called when DOM is ready
	*/
	initialize: function ($config) {
		this._super();

		// check if we already have existing styleSheets on the page
		this._element = window.document.createElement('style');

		// append our stylesheet to the head
		window.document.getElementsByTagName('head')[0].appendChild(this._element);

		// basic properties
		this._ready = false;
		this._styles = [];
		this._images = new MAL.Set();

		// tracks instances
		this._views = new MAL.Set();

		// tracks view classes
		this._viewSignatures = new MAL.Set();

		this._viewsToWrite = [];

		if ($config.callback) this._checkCSS($config.callback);
	},

	/**
	 * make sure that the stylesheet object is ready to be written to
	 *
	 * @private
	 */
	_checkCSS: function (callback) {
		var style = this._getStylesheet(),
			self = this;

		// check if the stylesheet is processed and ready
		try {
			if (style && style.rules && !MAL.isUndefined(style.rules.length)) this._ready = true;
			if (style && style.cssRules && !MAL.isUndefined(style.cssRules.length)) this._ready = true;
			if (MAL.Environment.browser === 'safari' && style && style.cssRules === null) this._ready = true;
		} catch (e) {
			// TODO: figure out a way to avoid using this... for some reason firefox thinks its hacking
			//console.log(e.message);
		}

		// write html if we are really ready
		if (this._ready) {
			if (callback) window.setTimeout(callback.bind(this), 0);
			return;
		}

		window.setTimeout(function () {
			self._checkCSS(callback);
		}, 0);
	},

	/**
	 * obtain the actual stylesheet object that matches the element we made
	 *
	 * @private
	 */
	_getStylesheet: function () {
		var stylesheet = false;

		for (var i = 0; i < window.document.styleSheets.length; i++) {
			if ((window.document.styleSheets[i].ownerNode || window.document.styleSheets[i].owningElement) === this._element)	{
				stylesheet = window.document.styleSheets[i];
				break;
			}
		}

		return stylesheet;
	},

	/**
	 * writes rules to documents style sheet
	 *
	 * @param {Object} rules
	 * @param {Object} animations
	 */
	writeRules: function (rules, animations) {
		// obtain the real stylesheet
		var stylesheet = this._getStylesheet(),
			stylesheetText = '';

		// HACK: stylesheetText is a hack so we can use the chrome development tools (when this is fixed we can remove this)
		for (var selector in rules) {
			var propertiesString = '',
				rule = rules[selector];

			for (var property in rule) if (rule[property])
				propertiesString += rule[property].name + ':' + rule[property].value + ';';

			if (MAL.Environment.browser === 'chrome') {
				stylesheetText += selector + ' {' + propertiesString + '}\n';
			} else if (MAL.Environment.browser === 'safari' && !MAL.isUndefined(stylesheet.addRule)) {
				stylesheet.addRule(selector, propertiesString);
			} else if (!MAL.isUndefined(stylesheet.insertRule)) {
				stylesheet.insertRule(selector + ' {' + propertiesString + '}', stylesheet.cssRules.length);
			} else {
				// in IE we can't use comma seperated selectors... so we need to just keep adding rules
				var selectors = selector.split(',');

				for (var i = 0; i < selectors.length; i++) {
					stylesheet.addRule(selectors[i], propertiesString, stylesheet.rules.length);
				}
			}
		}

		if (MAL.Environment.getCSSPropertyName('animation-name')) {
			for (var animation in animations) {
				if (MAL.Environment.browser === 'chrome') {
					stylesheetText += animation + ' {' + animations[animation] + '}';
				} else if (MAL.Environment.browser === 'safari' && !MAL.isUndefined(stylesheet.addRule)) {
					stylesheet.addRule(animation, animations[animation]);
				} else if (!MAL.isUndefined(stylesheet.insertRule)) {
					stylesheet.insertRule(animation + ' {' + animations[animation] + '}', stylesheet.cssRules.length);
				}
			}
		}

		if (stylesheetText) this._element.innerHTML += stylesheetText;
	},

	/**
	 * adds views to stylesheet
	 *
	 * @param {Array...} views
	 */
	addViews: function () {
		var views = MAL.argumentsToArray(arguments);
		views.forEach(this.addView);
	},

	/**
	 * adds view to stylesheet
	 *
	 * @param  {MAL.View/MAL.ViewController} view
	 */
	addView: function (view) {
		if (view instanceof MAL.ViewController) {
			view = view.view;
		}

		if (this._views.has(view)) return;
		this._views.add(view);
		this._viewsToWrite.push(view);
	},

	/**
	 * adds a set of rules to the rules array
	 *
	 *     stylesheet.addRules({
	 *      '#Box': {
	 *       width: 50,
	 *       height: 50,
	 *       left: 0,
	 *       top: 0,
	 *       position: 'absolute'
	 *      },
	 *      '#Box2': {
	 *       width: 50,
	 *       height: 50,
	 *       left: 50,
	 *       top: 50,
	 *       position: 'absolute'
	 *      }
	 *     });
	 *
	 * @param {Object} rules
	 */
	addRules: function (rules) {
		this.addStyle(new MAL.Style(rules));
	},

	/**
	 * adds a set of rules with default rules to the rules array
	 *
	 *     stylesheet.addRulesWithDefaults(
	 *      // defaults
	 *      {
	 *       left: 0,
	 *       top: 0,
	 *       position: 'absolute'
	 *      },
	 *
	 *      // rules
	 *      {
	 *       // because of the defaults this rule will actually also include {left:0,top:0,position:'absolute'}
	 *       '#Box': {},
	 *
	 *       // because of the defaults this rule will actually also include {left:0,top:0}
	 *       '#Box2': {
	 *        left: 50,
	 *        top: 50
	 *       }
	 *      }
	 *     );
	 *
	 * @param {Object} default rules with default (first argument is default)
	 * @param {Object} rules everything else is a rule
	 */
	addRulesWithDefaults: function (defaults, rules) {
		this.addStyle(new MAL.Style(rules, {defaults: defaults}));
	},

	/**
	 * adds a set of raw rules to the rules object (**does not prefix classes or IDs with mal-ad-**)
	 *
	 *     stylesheet.addRawRules({
	 *      '#Box': {
	 *        width: 50,
	 *        height: 50,
	 *        left: 0,
	 *        top: 0,
	 *        position: 'absolute'
	 *      },
	 *      '#Box2': {
	 *        width: 50,
	 *        height: 50,
	 *        left: 50,
	 *        top: 50,
	 *        position: 'absolute'
	 *      }
	 *     });
	 *
	 * @param {Object} rules
	 */
	addRawRules: function (rules) {
		this.addStyle(new MAL.Style(rules, {raw: true}));
	},

	/**
	 * add a style object to the write cue
	 *
	 *     stylesheet.addStyle(
	 *          new MAL.Style({
	 *              '.button': {
	 *                  width: 200,
	 *                  height: 200
	 *               }
	 *          })
	 *     );
	 *
	 * @param {Object} rules
	 */
	addStyle: function () {
		var styles = MAL.argumentsToArray(arguments);
		styles.forEach(function (style) {
			this._styles.push(style);
		}, this);
	},

	/**
	 * writes styles to stylesheet
	 */
	writeStyles: function () {
		var views = this._viewsToWrite,
			view,
			rules = {},
			animations = {};

		while ((view = views.shift())) {
			var viewStyles = view.getStyles();

			viewStyles.forEach(function (viewStyle) {
				var viewSignature = viewStyle.view.objectClassPath.join('.'),
					additionalRules = {};

				if (!this._viewSignatures.has(viewSignature)) {
					this._viewSignatures.add(viewSignature);
					this._images.add(viewStyle.getImages());

					additionalRules = MAL.mergeOptions(additionalRules, viewStyle.rules);
					animations = MAL.mergeOptions(animations, viewStyle.animations);
				}

				additionalRules = MAL.mergeOptions(additionalRules, viewStyle.instanceRules);

				for (var rule in additionalRules) {
					rules = MAL.mergeOptions(rules, additionalRules);
					break;
				}
			}, this);
		}

		this.writeRules(rules, animations);

		var style;

		while((style = this._styles.shift())) {
			this._images.add(style.getImages());
			this.writeRules(style.rules, style.animations);
		}
	},


	/**
	 * returns the images that have been used in the stylesheet
	 *
	 *     var images = stylesheet.getImages();
	 *
	 * @return {Array} images
	 */
	getImages: function () {
		return this._images.clone();
	}
}));

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * ViewStyle is a Style object that is made to be used directly inside of {@link MAL.View}, the interface remains exactly the same as {@link MAL.Style}
 *
 * @class MAL.ViewStyle
 * @extends MAL.Style
 */
MAL.define('ViewStyle', MAL.Style.extend({
	/**
	 * initializes MAL.ViewStyle
	 *
	 *     var style = new MAL.ViewStyle({
	 *       body: {
	 *         background: '#ccc'
	 *       }
	 *     });
	 *
	 * @method constructor
	 * @param {Object} config
	 */

	/**
	* @cfg {String} prefix
	* the className prefix (defaults to mal-ad-)
	*/

	/**
	* @cfg {Boolean} raw
	* if you set raw to true the rules will not have auto prefixed properties
	*/

	/**
	* @cfg {Object} defaults
	* an object of default rules
	*/

	/**
	* @cfg {Function} filePathProxy
	* event loop delay in milliseconds
	*/

	/**
	* @cfg {String} imagePath
	* event loop delay in milliseconds
	*/
	initialize: function (rules, $config) {
		$config = $config || {};

		this.view = $config.view;
		this.instanceRules = {};

		this._super(rules, MAL.mergeOptions({prefix: this.view.prefix, selectorPrefix: '.mal-view.' + this.view.prefix.substr(0, this.view.prefix.length-1) + ' '}, $config));

		this._parseInstanceRules();
	},

	/**
	 * @private
	 */
	_parseInstanceRules: function () {
		for (var rule in this.rules) {
			if (rule.match(/@instance/)) {
				this.instanceRules[rule.replace(/@instance/, '#mal-view' + this.view.id)] = this.rules[rule];
				delete this.rules[rule];
			}
		}
	}
}));

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * View class is made to be extended upon and should never be directly initialized, when extending upon it you need to specify the following properties
 *
 * * name
 * * styles
 * * html
 *
 * NOTE: MAL.rmv is required for imagePath and filePathProxy.
 *
 * ## example
 *
 *      MAL.ExampleView = MAL.View.extend({
 *          name: 'example',
 *
 *			className: 'card', // accepts an array too
 *
 *          styles: function () { return {
 *              view: {
 *                  width: 200,
 *                  height: 200
 *              },
 *
 *              '.button': {
 *                  width: 200,
 *                  height: 200
 *              }
 *          }},
 *
 *          html: '<div class="button"><div class="item mal-external-item"><img src="image2.png"/></div></div>'
 *      });
 *
 * ## styles
 *
 * use the 'view' selector to target the views container element
 *
 * ## html
 *
 * every class gets prefixed except for when you already prefix it so *class="something mal-something-else"* becomes *class="mal-{VIEW_NAME}-something mal-something-else"*
 *
 * @class MAL.View
 * @extends MAL.Object
 *
 */
MAL.define('View', MAL.Object.extend({
	/**
	 * initializes MAL.View
	 *
	 * @method constructor
	 * @param {Object} config
	 */
	initialize: function ($config) {
		$config = $config || {};

		this._super();

		this.id = MAL.View.prototype.id++;

		this.controller = $config.controller;

		this.name = this.name || MAL.toHyphenDelimited(this.objectClassName);

		this._images = [];
		this.prefix = 'mal-' + this.name + '-';

		if (typeof MAL.rmv !== 'undefined') {
			this._imagePath = MAL.rmv.paths.images;
			this._filePathProxy = MAL.rmv.filePathProxy;
		} else {
			MAL.error('Missing MAL.rmv');
		}

		this.element = this._createContainer(this.name, this.html);
		this.element.MALController = this.controller;

		this.layer = MAL.stage[$config.layerType ? 'add' + $config.layerType + 'Layer' : 'getLayer'](this.element);

		this._styles = new MAL.Set();
		this._views = new MAL.Set();

		this.style = this._createStyle(this.styles());
		this._styles.add(this.style);
	},

	/**
	 * the view instance ID
	 * @type {Number}
	 */
	id: 0,

	/**
	 * node used for converting HTML into DOM
	 *
	 * @private
	 */
	_DOMContainer: document.createElement('div'),

	/**
	 * add subview references to the view
	 * @param  {Array} subviews
	 */
	addSubviews: function () {
		var subviews = MAL.argumentsToArray(arguments);
		subviews.forEach(this.addSubview, this);
	},

	/**
	 * adds a subview reference to the view
	 * @param  {Object} subview
	 */
	addSubview: function (subview) {
		if (subview instanceof MAL.ViewController) subview = subview.view;
		this._views.add(subview);
	},

	/**
	 * adds a subview reference to the view that doesnt exist yet
	 *
	 * HACK: this is a hack, and does generate an additional unused view. In the future we can fix this though.
	 *
	 * @param  {Object} subview
	 */
	willHaveSubview: function (Subview) {
		var subview = new Subview();
		this.addSubview(subview);
	},

	/**
	 * creates the container element and injects it with html
	 *
	 * @private
	 * @param  {String} name
	 * @param  {String} html
	 * @return {Object} element
	 */
	_createContainer: function (name, html) {
		var classes = ['view', this.name];
		if (MAL.isArray(this.className)) {
			classes = classes.concat(this.className);
		} else if (MAL.isString(this.className)) {
			classes.push(this.className);
		}
		classes = classes.map(function (className) {
			return 'mal-' + className;
		});

		var element = document.createElement('div');
		element.className = classes.join(' ');
		element.id = 'mal-view' + this.id;

		var processed = MAL.processHTML(html, this._imagePath, this.prefix);

		element.innerHTML = processed.html;
		this._images = processed.images;

		return element;
	},

	/**
	 * creates a Style with the provided rules
	 *
	 * @private
	 * @param  {Object} rules
	 * @return {MAL.Style} style
	 */
	_createStyle: function (rules) {
		//
		var style = new MAL.ViewStyle(rules, {view: this}),
			images = style.getImages();

		this._images.add(images);

		return style;
	},

	/**
	 * adds view prefix to selector string
	 *
	 * @param {String} selector
	 * @return {String} selector
	 */
	addPrefixToSelector: function (selector) {
		return selector.replace(new RegExp('([#\\.])(?!mal-)([A-Z0-9\\-_]*)', 'ig'), '$1' + this.prefix + '$2');
	},

	/**
	 * adds view prefix to className string
	 *
	 * @param {String} selector
	 * @return {String} selector
	 */
	addPrefixToClassName: function (className) {
		return this.prefix + className;
	},

	/**
	 * adds view prefix to html string
	 *
	 * @param {String} selector
	 * @return {String} selector
	 */
	addPrefixToHTML: function (html) {
		return MAL.processHTML(html, this._imagePath, this.prefix).html;
	},

	/**
	 * add additional images that are used in the view
	 *
	 * @return {Array} images
	 */
	addImages: function () {
		this._images.add(MAL.argumentsToArray(arguments));
	},

	/**
	 * returns images that were used in the HTML and CSS for the view
	 *
	 * @return {Array} images
	 */
	getImages: function () {
		var images = this._images.clone();

		this._views.forEach(function (view) {
			images = images.concat(view.getImages());
		});

		return images;
	},

	/**
	 * returns a flat array of styles that are related to the view
	 *
	 * @return {Array} styles
	 */
	getStyles: function () {
		var styles = this._styles.clone();

		this._views.forEach(function (view) {
			styles = styles.concat(view.getStyles());
		});

		return styles;
	},

	/**
	 * works just like element.querySelector but operates directly on the views container
	 *
	 * @param  {String} selector
	 * @param  {Object} [target]
	 * @return {Object} element
	 */
	query: function (selector, target) {
		target = target || this.element;
		return target.querySelector(this.addPrefixToSelector(selector));
	},

	/**
	 * returns a layer object
	 *
	 * @param  {String} selector
	 * @param  {Object} [target]
	 * @return {Object} layer or layerProxy
	 */
	queryLayer: function (selector, target) {
		return MAL.stage.getLayer(this.query(selector, target));
	},

	/**
	 * works just like element.querySelectorAll but operates directly on the views container
	 *
	 * @param  {String} selector
	 * @param  {Object} [target]
	 * @return {Array} elements
	 */
	queryAll: function (selector, target) {
		target = target || this.element;
		return MAL.toArray(target.querySelectorAll(this.addPrefixToSelector(selector)));
	},

	/**
	 * gets a layer proxy for the selectors results
	 *
	 * @param  {String} selector
	 * @param  {Object} [target]
	 * @return {Object} layer or layerProxy
	 */
	queryLayerAll: function (selector, target) {
		return MAL.stage.getLayers.apply(MAL.stage, this.queryAll(selector, target));
	},

	/**
	 * appends the elements container to the provided element
	 *
	 * @param  {Object} target target element
	 */
	appendTo: function (target) {
		this._getElement(target).appendChild(this.element);
	},

	/**
	 * appends the elements to target element
	 *
	 * @param  {Object} target target element
	 * @param  {Array} views array of elements
	 */
	appendElementsTo: function (elements, target) {
		target = this._getElement(target) || this.element;

		elements.forEach(function (element) {
			if (element instanceof MAL.ViewController || element instanceof MAL.View) {
				this.addSubview(element);
			}

			target.appendChild(this._getElement(element));
		}, this);
	},

	/**
	 * gets the element from a View, Controller, or queryString
	 *
	 * @param  {Object} target target element
	 */
	_getElement: function (item) {
		if (item instanceof MAL.View) {
			item = item.element;
		} else if (item instanceof MAL.ViewController) {
			item = item.view.element;
		} else if (MAL.isString(item)) {
			item = this.query(item);
		}

		return item;
	},

	/**
	 * checks if element has a class
	 *
	 * @param  {String} className
	 * @param  {Object/String} element element or query
	 */
	hasClass: function (className, element) {
		if (MAL.isUndefined(element)) {
			element = this.element;
		} else if (MAL.isString(element)) {
			element = this.query(element);
		}

		return MAL.stage.getLayer(element).hasClass(className, className.match(/^mal-/) ? '' : this.prefix);
	},

	/**
	 * adds a class to the elements
	 *
	 * @param  {String} className
	 * @param  {Object/String/Array} elements elements, query, or layer
	 */
	addClass: function (className, elements) {
		if (MAL.isUndefined(elements)) {
			elements = [this.element];
		} else if (MAL.isString(elements)) {
			elements = this.queryAll(elements);
		} else if (!MAL.isArray(elements)) {
			elements = [elements];
		}

		MAL.stage.getLayers(elements).addClass(className, className.match(/^mal-/) ? '' : this.prefix);
	},

	/**
	 * removes a class from the elements
	 *
	 * @param  {String} className
	 * @param  {Object/String/Array} elements elements, query, or layer
	 */
	removeClass: function (className, elements) {
		if (MAL.isUndefined(elements)) {
			elements = [this.element];
		} else if (MAL.isString(elements)) {
			elements = MAL.toArray(this.queryAll(elements));
		} else if (!MAL.isArray(elements)) {
			elements = [elements];
		}

		MAL.stage.getLayers(elements).removeClass(className, className.match(/^mal-/) ? '' : this.prefix);
	},

	/**
	 * converts HTML string into property prefixed DOM
	 *
	 * @param  {String} html
	 * @return {Object} element
	 */
	createElement: function (html) {
		var processed = MAL.processHTML(html, this._imagePath, this.prefix);

		this._DOMContainer.innerHTML = processed.html;

		var elements = MAL.toArray(this._DOMContainer.childNodes);

		this._DOMContainer.removeChild(elements[0]);

		this._images.add(processed.images.clone());

		return elements[0];
	}
}));

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * ViewController class is made to be extended upon and should never be directly initialized
 *
 * ## example
 *
 *    var viewController = MAL.ViewController.extend({
 *      intialize: function(){
 *         this._super();
 *      }
 *    });
 *
 * @class MAL.ViewController
 * @extends MAL.Object
 */
MAL.define('ViewController', MAL.Object.extend({
	/**
	 * initializes MAL.ViewController
	 *
	 * @method constructor
	 * @param {Object} config
	 */
	initialize: function ($config) {
		$config = $config || {};

		this._super($config);
		this._autoInitializeView($config.layerType);

		this._stateListeners = {};
	},

	_autoInitializeView: function (layerType) {
		var View = MAL.get('views.' + this.objectClassPath.slice(1).join('.'));

		if (View) {
			this.view = new View({controller: this, layerType: layerType});
		}
	},

	/**
	 * transitioning state var
	 * @type {Boolean}
	 */
	transitioning: false,

	/**
	 * enabled state var
	 * @type {Boolean}
	 */
	enabled: true,

	/**
	 * provides a buffer that ensures that *transitioning=false* and *enabled=true*
	 *
	 * @param  {Function} callback
	 * @return {Function} callback
	 */
	interactionCheck: function (callback) {
		var self = this;

		return function (event) {
			if(event && event.stopPropagation){
				event.stopPropagation();
			}
			if (!self.transitioning && self.enabled) {
				callback.apply(self, arguments);
			}
		};
	},

	/**
	 * enable/disable all state logging
	 */
	logStates: false,

	/**
	 * array of states to not log
	 */
	logStatesIgnore: [],

	/**
	 * add a state listener to the view controller
	 *
	 *     this.addStateListener('poking', onPokeToggle);
	 *
	 * @param {String} name
	 * @param {Function} listener
	 */
	addStateListener: function (name, listener) {
		if (!MAL.isFunction(listener)) throw new Error ('the listener you tried to add for "' + (this.objectClassPathToken + ':' + type) + '" is not a function');

		if (MAL.isUndefined(this._stateListeners[name])) {
			this._stateListeners[name] = [];
		}

		this._stateListeners[name].push(listener);
	},

	/**
	 * remove a state listener from the view controller.
	 *
	 *     // you must pass in the original function (bind makes this a little less ideal)
	 *     this.removeStateListener('poking', onPokeToggle);
	 *
	 * @param {String} name
	 * @param {Function} listener
	 */
	removeStateListener: function (name, listener) {
		if (this._stateListeners[name] instanceof Array){
			var listeners = this._stateListeners[name];

			for (var i=0, l=listeners.length; i < l; i++){
				if (listeners[i] === listener){
					listeners.splice(i, 1);
					break;
				}
			}
		}
	},

	/**
	 * @private
	 */
	_callStateListeners: function(name, active, _internalCall) {
		var listeners = this._stateListeners[name];

		if (this.logging && this.logStates && this.logStatesIgnore.indexOf(name) === -1) {
			if (MAL.Environment.browser === 'chrome') {
				console.log('%c' + this.getFileAndLineNumberAtStackIndex(_internalCall ? 4 : 3) + '%c state %c' + name + '%c ' + (active ? 'added to' : 'removed from') + ' %c' + this.objectClassPathToken, 'font-weight: bold; color: #1795de;', 'color: #000', 'font-weight: bold', '', 'font-weight: bold');
			} else {
				MAL.log('StateChange ' + this.objectClassPathToken + (active ? ' added ' : ' removed ') + name);
			}
		}

		if (MAL.isArray(listeners)) {
			listeners.forEach(function (listener) {
				listener(active);
			});
		}
	},

	/**
	 * toggles a view controllers state
	 *
	 *     this.toggleState('poking');
	 *
	 * @param {String} name
	 */
	toggleState: function (name) {
		if (this.hasState(name)) {
			this.removeState(name, true);
		} else {
			this.addState(name, true);
		}
	},

	/**
	 * adds the state to the view controller
	 *
	 *     this.addState('poking');
	 *
	 * @param {String} name
	 */
	addState: function (name, _internalCall) {
		if (!this.hasState(name)) {
			this.view.addClass('state-' + name);
			this._callStateListeners(name, true, _internalCall);
		}
	},

	/**
	 * removes the state from the view controller
	 *
	 *     this.removeState('poking');
	 *
	 * @param {String} name
	 */
	removeState: function (name, _internalCall) {
		if (this.hasState(name)) {
			this.view.removeClass('state-' + name);
			this._callStateListeners(name, false, _internalCall);
		}
	},

	/**
	 * tells you if the view controller has the state
	 *
	 *     this.hasState('poking');
	 *
	 * @param {String} name
	 * @return {Boolean} active
	 */
	hasState: function (name) {
		if (name.match(/^mal-/)) {
			if (MAL.toArray(document.querySelectorAll('.' + name + ' .mal-view.mal-' + this.view.name)).indexOf(this.view.element) !== -1) {
				return true;
			} else {
				return false;
			}
		} else {
			return this.view.hasClass('state-' + name);
		}
	}
}));

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * TimeoutManager class. Provides a way to ensure that timeouts with the same timeout run grouped. **This should not be used for timeouts that need to possibly be cleared.**
 *
 * # Usage
 *
 *     // create a timeout manager (usually you only want one of these ever running)
 *     var timeouts = new MAL.TimeoutManager();
 *
 *     // add a timeout
 *     timeouts.addTimeout(function () {
 *
 *     }, 1000);
 *
 * @class MAL.TimeoutManager
 */
MAL.define('TimeoutManager', MAL.Util.create({
	/**
	 * initializes MAL.TimeoutManager
	 *
	 *     // create a timeout manager (usually you only want one of these ever running)
	 *     var timeouts = new MAL.TimeoutManager();
	 *
	 * @method constructor
	 * @param {Object} config
	 */

	/**
	* @cfg {Number} leniency
	* event loop delay in milliseconds
	*/
	initialize: function ($config) {
		// bind all event based methods
		MAL.bindAll(this, '_loop');

		// optional config object
		$config = $config || {};

		this.queue = {};
		this._loop();
		this.leniency = $config.leniency || 0;
	},

	/**
	 * internal loop
	 * @private
	 */
	_loop: function () {
		for (var item in this.queue) {
			var tasks = this.queue[item];

			window.setTimeout((function (tasks) {
				return function () {
					for (var task = 0; task < tasks.length; task++) tasks[task]();
				};
			}(tasks)), item);

			delete this.queue[item];
		}

		window.setTimeout(this._loop, this.leniency);
	},

	/**
	 * add a timeout similar to window.setTimeout but managed
	 *
	 *     // wait 1000 ms then fire callback
	 *     timeouts.addTimeout(function () {
	 *
	 *     }, 1000);
	 *
	 *     // you can also bind a scope
	 *     timeouts.addTimeout((function () {
	 *
	 *     }).bind(this), 1000);
	 *
	 * @param {Function} callback
	 * @param {Number} delay
	 */
	addTimeout: function (callback, delay) {
		if (!this.queue[delay]) this.queue[delay] = [];
		this.queue[delay].push(callback);
	}
}));

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Script loader class. Provides an easy way to preload javascript files.
 *
 * # Usage
 *
 * to load and ensure that a set of javascript files are loaded you may do the following
 *
 *     new MAL.ScriptLoader({
 *      path: 'this/is/a/path/',
 *      scripts: ['one.js', 'two.js', 'three.js'],
 *      callback: function () {
 *       // everything has been loaded
 *      }
 *     });
 *
 * if you want to build up a loader and start it later on you can do this
 *
 *     var loader = new MAL.ScriptLoader({
 *      autoload: false,
 *      path: 'this/is/a/path/',
 *      callback: function () {
 *       // everything has been loaded
 *      }
 *     });
 *
 *     loader.addScript('one.js');
 *
 *     loader.addScript('two.js');
 *
 *     loader.start();
 *
 *
 * @class MAL.ScriptLoader
 */
MAL.define('ScriptLoader', MAL.Util.create({
	/**
	 * initializes MAL.ScriptLoader
	 *
	 *     new MAL.ScriptLoader({
	 *      async: true,
	 *      path: 'this/is/a/path/',
	 *      scripts: ['one.js', 'two.js', 'three.js'],
	 *      callback: function () {
	 *       // everything has been loaded
	 *      }
	 *     });
	 *
	 * @method constructor
	 * @param {Object} config
	 */

	/**
	* @cfg {Function} callback
	*
	* callback function that should be called upon completion
	*/
	/**
	* @cfg {Boolean} async
	*
	* load scripts asynchronously instead of serially
	*/
	/**
	* @cfg {String} path
	*
	* path all the scripts should be prefixed with
	*/
	/**
	* @cfg {Array} scripts
	*
	* an array of the scripts you wish to load
	*/
	/**
	* @cfg {Boolean} autoload
	*
	* autoload the scripts upon initialization
	*/
	initialize: function ($config) {
		$config = $config || {};

		this.scripts = new MAL.Set();

		this.callback = $config.callback || false;
		this.async = $config.async || false;
		this.path = $config.path || '';

		// determine if we should be loading in library files normally
		this.packed = !MAL.isUndefined(MAL.CHANGESET);

		if ($config.scripts) this.addScripts($config.scripts);
		if ($config.scripts && $config.autoload !== false) this.start();
	},

	/**
	 * adds script to the load stack
	 *
	 *     loader.addScript('two.js');
	 *
	 * @param {String} src
	 */
	addScript: function (src) {
		// append .js file extension
		if (!/\.js$/.test(src)) src += '.js';

		// make special library related replacements
		if (this.packed && /^(.\/)?MALLibrary\/(.+)/.test(src)) {
			src = src.replace(/^(.\/)?MALLibrary\/(.+)/, function (match, prefix, path) {
				return ((prefix || '') + path).replace(/\//, '_');
			});
		}

		// prepend file path if relative
		if (!/^http\:\/\//.test(src)) src = this.path + src;

		this.scripts.add(src);
	},

	/**
	 * adds multiple scripts to the load stack
	 *
	 *     loader.addScript('two.js', 'three.js');
	 *
	 * @param {Array} scripts
	 */
	addScripts: function (scripts) {
		for (var script = 0; script < scripts.length; script++)
			this.addScript(scripts[script]);
	},

	/**
	 * starts script loader
	 *
	 *     preloader.start();
	 *
	 * @param {Array} scripts
	 */
	start: function () {
		MAL.Async[this.async ? 'parallel' : 'serial'](
			this.scripts.map(function (src) {

				return function (callback) {
					var self = this;

					var	script = window.document.createElement('script');
					script.setAttribute('type','text/javascript');
					script.setAttribute('src', src);

					if (script.addEventListener) {
						MAL.addEvent(script, 'load', callback);
					} else if (script.attachEvent) {
						// IE 6 7 8
						MAL.addEvent(script, 'readystatechange', function (event) {
							if(/loaded|complete/.test(script.readyState)) callback();
						});
					}

					window.document.getElementsByTagName('head')[0].appendChild(script);
				};
			}),
			this.callback
		);
	}
}));

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * RawTween class. Provides a basic method for number tweening.
 *
 * # Usage
 *
 *  example of tweening some numbers
 *
 *     var tweens = new MAL.RawTween();
 *
 *     tweens.addTween({
 *     delay: 1,
 *      duration: 1,
 *		start: {a: 0, b: 50},
 *		finish: {a: 100, b: -1200},
 *		easing: [0, 0, 1, 1],
 *		callbacks: {
 *		start: function (tween, current) {console.log('start')},
 *		update: function (tween, current) {console.log('update', current.a, current.b)},
 *		complete: function (tween, current) {console.log('complete')}
 *		}
 *     });
 *
 *     tweens.start();
 *
 * @class MAL.RawTween
 */
MAL.define('RawTween', MAL.Util.create({
	/**
	 * initializes MAL.RawTween
	 *
	 *     var rawTweens = new MAL.RawTween({fps: 60});
	 *
	 * @method constructor
	 * @param {Object} $config
	 */

	/**
	 * @cfg {Number} fps
	 * fps for all tweens
	 */
	initialize: function ($config) {
		$config = $config || {};
		this.fps = $config.fps || 60;
		this.timeout = null;
		this.tweens = [];
	},

	/**
	 * Starts the raw tween loop
	 *
	 *     rawTweens.start()
	 */
	start: function () {
		this.startTime = new Date().getTime();
		this._loop();
	},

	/**
	 * animation loop
	 *
	 * @private
	 */
	_loop: function () {
		clearTimeout(this.timeout);

		var time = new Date().getTime();
		for (var i = 0; i < this.tweens.length; i++)
			this._update(this.tweens[i], time);

		this.timeout = setTimeout(this._loop.bind(this), 1000 / this.fps);
	},

	/**
	 * update tween with time
	 *
	 * @private
	 */
	_update: function (tween, time) {
		if (tween.startTime >= time) return;

		if (!tween.started) {
			tween.started = true;
			if (tween.callbacks.start) tween.callbacks.start(tween);
		}

		var finished = time > (tween.startTime + tween.duration),
			percent = (time - tween.startTime) / tween.duration;

		if (MAL.isArray(tween.easing)) {
			percent = MAL.cubicBezierWithPercent(percent, tween.easing[0], tween.easing[1], tween.easing[2], tween.easing[3]);
		} else {
			percent = tween.easing(percent);
		}

		if (percent > 1) percent = 1;

		tween.percent = percent;

		for (var i in tween.values.finish) {
			if (finished) {
				tween.values.current[i] = tween.values.finish[i];
			} else {
				tween.values.current[i] = tween.values.start[i] + (tween.values.delta[i] * percent);
			}
		}

		if (tween.callbacks.update) tween.callbacks.update(tween, tween.values.current);
		if (finished && tween.callbacks.complete) tween.callbacks.complete(tween, tween.values.current);
		if (finished) this.tweens.splice(this.tweens.indexOf(tween), 1);
	},

	/**
	 * Add a raw tween
	 *
	 *     tweens.addTween({
	 *			delay: 1,
	 *			duration: 1,
	 *			start: {a: 0, b: 50},
	 *			finish: {a: 100, b: -1200},
	 *			easing: [0, 0, 1, 1],
	 *			callbacks: {
	 *			start: function (tween, current) {console.log('start')},
	 *			update: function (tween, current) {console.log('update', current.a, current.b)},
	 *			complete: function (tween, current) {console.log('complete')}
	 *		}
	 *     });
	 *
	 * @param {Object} tween
	 */
	addTween: function ($tween) {
		var tween = {};
		tween.started = false;
		tween.finished = false;
		tween.obj = $tween.obj;
		tween.delay = $tween.delay * 1000 || 0;
		tween.duration = $tween.duration * 1000 || 1000;
		tween.startTime = new Date().getTime() + tween.delay;
		tween.easing = $tween.easing || [0.25, 0.1, 0.25, 1];
		tween.values = {
			start: $tween.start || {},
			delta: {},
			current: {},
			finish: $tween.finish
		};
		tween.callbacks = {
			start: $tween.callbacks.start || null,
			update: $tween.callbacks.update || null,
			complete: $tween.callbacks.complete || null
		};

		// if start value is not supplied set it to zero
		for (var i in tween.values.finish) {
			if (!tween.values.start[i]) tween.values.start[i] = 0;
			tween.values.delta[i] = tween.values.finish[i] - tween.values.start[i];
		}

		this.tweens.push(tween);
	}
}));

/**
 * @author Chad Scira <chad@mediaartslab.com>, Martijn Korteweg <martijn@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Keyframe Animation class. Provides an really useful timer based tool that is usually used for keyframe animations, but can be used for other things.
 *
 * # Usage
 *
 * auto calculated timing based on time/frames
 *
 *     new MAL.KeyframeAnimation({
 *       delay: 1,
 *       time: 10,
 *       frames: 20
 *      },
 *      function () {
 *       // this.frame
 *      }
 *     );
 *
 * Specific timing on frame 1:
 *
 *     new MAL.KeyframeAnimation({
 *       time: {defaultTime: .5, '1': .75},
 *       frames: 20
 *      },
 *      function () {
 *       // this.frame
 *      }
 *     );
 *
 * @class MAL.KeyframeAnimation
 * @extends MAL.Object
 */
MAL.define('KeyframeAnimation', MAL.Object.extend({
	/**
	 * initialize
	 *
	 *     new MAL.KeyframeAnimation({
	 *       delay: 1,
	 *       time: 10,
	 *       frames: 20
	 *      },
	 *      function () {
	 *       // this.frame
	 *      }
	 *     );
	 *
	 * @method constructor
	 * @param {Object} config
	 * @param {Function} callback
	 */

	/**
	* @cfg {Object/Array} time
	*
	* controls the frame timing
	*
	* default time
	*     {defaultTime: .5}
	* default time and specific frame timing
	*     {defaultTime: .5, '23': .2}
	* specific timing for every frame (array length must not be less than the amount of frames)
	*     [.1, .2., .3, .4]
	*
	*/
	/**
	* @cfg {Number} delay
	*
	* event loop delay in milliseconds
	*/
	/**
	* @cfg {Number} frames
	*
	* amount of frames
	*/
	/**
	* @cfg {Number} startFrame
	*
	* frame to start animation on
	*/
	/**
	* @cfg {Boolean} autoplay
	*
	* autoplay the animation upon initialization
	*/
	/**
	* @cfg {Boolean} loop
	*
	* loops animation
	*/
	initialize: function ($config, $callback) {
		this._super();

		// set private vars
		this.options = $config;
		this.callback = $callback;
		this.frame = 0;
		this.totalTime = 0;
		this.nextTime = 0;
		this.frameTime = 0;
		this.autoplay = true;
		this._timeout = null;

		// throw error if we do not have the required options set
		if (!this.options || !this.callback || !this.options.frames || !this.options.time) MAL.error('[MAL.KeyframeAnimation] InvalidInitialization');

		// if there is a start frame and set the nessary info
		if (!MAL.isUndefined(this.options.startFrame)) {
			this.frame = this.options.startFrame;
			this.options.frames = this.options.frames+this.options.startFrame;
		}

		// make frames 1 based
		this.options.frames--;

		// stop here if its not suppose to play
		if (this.options.autoplay === false) return;

		this.play();

	},

	/**
	 * start animation
	 *
	 *     keyframeAnimation.start();
	 *
	 */
	play: function () {
		// fire delayed loop
		if (this.options.delay) {
			setTimeout(this._delayedStart, this.options.delay * 1000);
		} else {
			this._delayedStart();
		}
	},

	/**
	 * stops animation
	 *
	 *     keyframeAnimation.stop();
	 *
	 */
	stop: function () {
		clearTimeout(this._timeout);
	},

	/**
	 * replay the animation
	 *
	 *     keyframeAnimation.replay({delay: 1});
	 *
	 */
	replay: function ($config) {

		if (typeof $config !== 'undefined')
			this.options.delay = MAL.isUndefined($config.delay) ? this.options.delay : $config.delay;

		// reset frame
		this.frame = 0;

		// start the animation loop
		this.play();
	},

	/**
	 * delays the start of the loop
	 *
	 * @private
	 */
	_delayedStart: function () {
		// set the nessary timestamps
		this.startTime = this.timeSpend = new Date().getTime();

		// start the animation loop
		this._loop();
	},

	/**
	 * recursive loop function that adjusts the delay based on real time spent
	 *
	 * @private
	 */
	_loop: function () {
		// check if we are finished
        if (this.frame > this.options.frames) {
          if (this.options.loop) {
            this.frame = 0;
          } else {
            return;
          }
        }

		// get the current time
		var dateNow = new Date().getTime();

		// fire callback
		this.callback.apply(this);

		var time = this.options.time,
			frameTime = this._getFrameTime();

		// time that should be spent
		this.frameTime += frameTime;

		// time actually spent
		this.timeSpend = (dateNow-this.timeSpend)/1000;

		// keep track of the total time spent
		this.totalTime += this.timeSpend;

		// incriment frame
		this.frame++;

		// calculate when the next frame should be
		this.nextTime = this.frameTime;
		// figure out the difference
		var tempNext = this.nextTime-this.totalTime;

		// reset timeSpend
		this.timeSpend = dateNow;

		// fire recursive loop
		this._timeout = setTimeout(this._loop, tempNext * 1000);
	},

	/**
	 * figures out what the next delay should be
	 *
	 * @private
	 */
	_getFrameTime: function() {
		var frameTime;
		if (MAL.isNumber(this.options.time)) {
			// time will be divided upon the total keyframes
			frameTime = this.options.time/this.options.frames;
		} else if (MAL.isArray(this.options.time)) {
			// time is explicitly set for each and every keyframe
			frameTime = this.options.time[this.frame-1];
		} else if (MAL.isObject(this.options.time)) {
			// time is an object that provides a default time
			frameTime = !MAL.isUndefined(this.options.time[this.frame]) ? this.options.time[this.frame] : this.options.time.defaultTime;
		}

		return frameTime;
	}
}));

/**
 * @author Martijn Korteweg <martijn@mediaartslab.com>
 * @docauthor Martijn Korteweg <martijn@mediaartslab.com>
 *
 *
 * video should control all video element interactions.
 *
 * #usage
 *
 * Initialize your video object first
 *
 * 		var video = new MAL.video.Base({
 *	  			controls: false,
 * 		 	minimumTime: 1,
 * 		  	volume: 0,
 * 		   	autobuffer: true
 *      	});
 *
 * Then you can set your source object
 *
 * 		video.setSource({
 * 			path: 'http://somwehre',
 * 		 	file: 'videoname'
 * 		  	type: 'mp4', // override only
 * 		   	filePathProxy: RMV.filePathProxy
 *       	});
 *
 * @class MAL.video.Base
 * @extends MAL.Object
 */
MAL.define('video.Base', MAL.Object.extend({
	/**
	 * initializes MAL.video.Base
	 *
	 * @method constructor
	 * @param {Object} $config
	 */

	/**
	 * @cfg {Boolean} autobuffer
	 * autobuffer setting to preload video when the source is set
	 */

	/**
	 * @cfg {Boolean} controls
	 * turn on controls on the video element
	 */

	/**
	 * @cfg {Number} minimumTime
	 * minimum time before the progress event gets fired
	 */

	/**
	 * @cfg {Boolean} loop
	 * Make the video loop
	 */

	/**
	 * @cfg {Number} volume
	 * starting volume
	 */

	/**
	 * @event start
	 * Fired when the video starts to play
	 */

	/**
	 * @event cached
	 * Fired on the first progress event reading
	 * @param {Number} percent the percentage of the video that is loaded on cached
	 */

	/**
	 * @event ended
	 * Fired when video is ended
	 */

	/**
	 * @event canplay
	 * Passthrough from the video element
	 */

	/**
	 * @event loadeddata
	 * Passthrough from the video element
	 */

	/**
	 * @event error
	 * Passthrough from the video element
	 */

	/**
	 * @event stalled
	 * Passthrough from the video element
	 */

	/**
	 * @event abort
	 * Passthrough from the video element
	 */

	/**
	 * @event waiting
	 * Passthrough from the video element
	 */

	/**
	 * @event loadstart
	 * Passthrough from the video element
	 */

	/**
	 * @event emptied
	 * Passthrough from the video element
	 */

	/**
	 * @event play
	 * Passthrough from the video element
	 */

	/**
	 * @event pause
	 * Passthrough from the video element
	 */

	/**
	 * @event playing
	 * Passthrough from the video element
	 */

	/**
	 * @event seeking
	 * Passthrough from the video element
	 */

	/**
	 * @event seeked
	 * Passthrough from the video element
	 */

	/**
	 * @event ratechange
	 * Passthrough from the video element
	 */

	/**
	 * @event durationchange
	 * Passthrough from the video element
	 */

	/**
	 * @event volumechange
	 * Passthrough from the video element
	 */

	/**
	 * @event timeupdate
	 * Passthrough from the video element
	 */

	initialize: function ($config) {
		this._super();

		// setup options from $options passed in
		this._options = $config || {};

		// create the video element with buffer set from options
		this.element = document.createElement('video');

		this._setAutoBuffer(this._options.autobuffer);

		this.element.loop = this._options.loop;

		this.element.controls = this._options.controls || false;

		this.minimumTime = this._options.minimumTime || 1;

		// progress loops reset
		this._loops = 0;

		// start progress loop
		setTimeout(this._loop, 200);


		var events = [
			'_onCanPlayThrough',
			'_onLoadedMetaData',
			'_onEnded'
		];

		events.forEach(function(value, key, ref) {
			this.element.addEventListener(value.slice(3).toLowerCase(), this[value].bind(this), true);
		}.bind(this));

		var PassThroughEvents = [
			'_onCanPlay',
			'_onLoadedData',
			'_onError',
			'_onStalled',
			'_onSuspend',
			'_onAbort',
			'_onWaiting',
			'_onLoadStart',
			'_onEmptied',
			'_onPlay',
			'_onPause',
			'_onPlaying',
			'_onSeeking',
			'_onSeeked',
			'_onRateChange',
			'_onDurationChange',
			'_onVolumeChange',
			'_onTimeUpdate'
		];

		// making sure that we can listen to all element events on the video object
		PassThroughEvents.forEach(function(value, key, ref) {
			this.element.addEventListener(value.slice(3).toLowerCase(), function(e,o) {

				if (!this._forceload) this.dispatchEvent(e.type, false, e);
			}.bind(this), true);
		}.bind(this));

		// the internal object with all data related to the video load process
		this._loadDetails = {
			timeTaken: 0,
			percent: 0,
			cached: 0,
			percentNeeded: 100,
			percentLoaded: 0,
			timeNeeded: null, // time needed till 100%
			timeNeededTotal: null, // time till 100% from start
			bufferPercent: 0,
			bufferTime: null, // time needed till play
			averageBufferTime: 0 // average needed till play
		};

		// current playing state
		this.playing = false;

		// current can play through state
		this._canPlayThrough = false;

		// duration of the video
		this._duration = 0;

		// if the video has played
		this._played = false;

		// current volume of the video
		this._volume = null;

		// if our calculations think it can be played
		this.malCanPlayThrough = false;

		// if the source has been set
		this._hasSource = false;

		if (!MAL.isUndefined(this._options.volume)) {
			this._volume = this._options.volume;
			this.element.addEventListener('loadeddata',this._setLoadedVolume.bind(this),false);
		}

		this.types = {
			mp4: 'video/mp4; codecs="avc1.42E01E, mp4a.40.2"',
			webm: 'video/webm; codecs="vp8, vorbis"',
			ogg: 'video/ogg; codecs="theora, vorbis"'
		};

		// check what type of video can be played
		for (var i in this.types) {
			if (this.element.canPlayType(this.types[i]) === 'probably') {
				this.type = i; break;
			}
		}

		// Browser Bug: Windows Firefox 23+ has an issue with hardware acceleration (http://support.mozilla.org/en-US/questions/963666)
		if (MAL.Environment.bugs.forceWebm) this.type = 'webm';

		// force load variables
		this._lastPercentChangeTime = null;
		this._stalled = false;
		this._forceload = false;

		// check if the element has suspended the load
		this.element.addEventListener('suspend', function () {
			if (!this.malCanPlayThrough && !this._stalled && this._loadDetails.percent > 0) {
				this._stalled = true;
				this._forceload = true;
				this._forceLoadChangeCurrentTime();
			}
		}.bind(this), false);

	},

	/**
	 * Setting the source of the video
	 *
	 * 	video.setSource({
	 *	 		path: 'http://somwehre',
	 *	  		file: 'videoname',
	 *	   		type: 'mp4',
	 *	     	filePathProxy: RMV.filePathProxy
	 *	    });
	 *
	 * @param {Object} $config source settings
	 * @param {String} $config.path Path to the file
	 * @param {String} $config.file name of the file
	 * @param {String} $config.type override file type
	 * @param {Function} $config.filePathProxy the proxy function that runs through the RMV object
	 *
	 */

	setSource: function($config) {

		var opts = $config || {};

		if (MAL.isUndefined(opts.path) || MAL.isUndefined(opts.file)) MAL.error('source is required');
		if (!MAL.isUndefined(opts.duration)) this.setDuration(opts.duration);


		this.type = opts.type || this.type;
		this._dispatchedCanPlayThrough = false;

		var source = opts.path + opts.file + '.' + this.type;
		if (MAL.isFunction(opts.filePathProxy))
			source = opts.filePathProxy(opts.path,opts.file + '.' + this.type);

		this.element.sourceElement = document.createElement('source');
		this.element.sourceElement.src = source;
		this.element.sourceElement.type = this.types[this.type];

		this.element.appendChild(this.element.sourceElement);
		this._startTime = new Date().getTime();

		this._loaded = false;
		this._hasSource = true;
	},

	/**
	 * Start the buffering of the video when autobuffer was turned off initially
	 *
	 *		video.Base.startBuffer();
	 *
	 */
	startBuffer: function() {
		this._setAutoBuffer(true);
	},

	/**
	 * sets buffer state
	 *
	 * @private
	 */
	_setAutoBuffer: function (state) {
		this.element.autobuffer = state;
		this.element.preload = state ? 'auto' : 'none';
		this._autobuffer = state;
	},

	/**
	 * Getter for muted state
	 * @return {Boolean} true or false based on volume
	 *
	 *		video.Base.isMuted();
	 *
	 */
	isMuted: function () {
		return this.volume === 0 ? true : false;
	},

	/**
	 * Playing the video
	 * @param  {Number} volume volume between 0 and 1 of the volume the video should be playing with
	 * @param  {Number} delay  Number of seconds you want the video to be delayed before playing.
	 *
	 *		video.Base.play(1,0);
	 *
	 *
	 */
	play: function(volume, delay) {
		var self = this;

		if (this.volume !== volume && volume !== undefined) this.setVolume(volume);

		if (this._played || this._forceload)
			try{this.element.currentTime = 0;}catch(e){}


		this._played = true;

		if (!delay) {
			this.element.play();
			this.dispatchEvent('start');
			this.playing = true;
			this._forceload = false;
		} else {
			setTimeout(function(){
				self.element.play();
				self.dispatchEvent('start');
				self.playing = true;
				self._forceload = false;
			}, delay);
		}
	},

	/**
	 * Stopping the video
	 *
	 * 		video.Base.stop();
	 *
	 */
	stop: function() {
		this.playing = false;
		this.element.pause();
	},

	/**
	 * Set the volume of the video
	 * @param {Number} volume Number between 0 and 1 of the volume
	 *
	 * 		video.Base.setVolume(1);
	 *
	 */
	setVolume: function(volume) {
		this.element.volume = (volume) ? volume : 0;
		this.volume = volume;
		this.element.muted = (volume) ? false : true;
	},

	/**
	 * Change the current time of hte video
	 * @param {Number} time
	 *
	 *		video.Base.setCurrentTime(10); // set the video to 10 seconds
	 *
	 */
	setCurrentTime: function(time) {
		this.element.currentTime = time;
	},

	/**
	 * Setting specific duration of the video
	 * @param {Number} duration
	 *
	 * 		video.Base.setDuration(12); // set the duration of the video to 12
	 *
	 */
	setDuration: function (duration) {
		this._duration = duration;
	},

	/**
	 * Get the current videos duration
	 * @return {Number} Seconds of the duration
	 *
	 *		video.Base.duration(); // returned current duration
	 *
	 */
	duration: function () {
		return MAL.isNumber(this.element.duration) ? this.element.duration : this._duration;
	},

	/**
	 * Getter for _played
	 * @return {Boolean} If the video has played or not
	 *
	 *		video.Base.hasPlayed();
	 *
	 */
	hasPlayed: function() {
		return this._played;
	},

	/**
	 * interval for progress events
	 * @private
	 */
	_loop: function () {
		// progress event calculation/dispatch
		if (this._loaded) {
			return;
		} else if (this._autobuffer === true && this._hasSource) {
			this._dispatchProgress();
		}

		setTimeout(this._loop, 200);
	},

	/**
	 * Checking if the loading progress has stalled, and if we should start forceload
	 * @param  {Number} percent current percentage the video is loaded
	 * @private
	 */
	_checkStalled: function (percent) {
		var now = Date.now();
		if (!this._stalled && (
								// cached
								this._loadDetails.cached > 0 && this._loadDetails.cached === this._loadDetails.percent && now - this._lastPercentChangeTime > 500 ||
								// normal load
								this._lastPercentChangeTime !== null && now - this._lastPercentChangeTime > 3000
							)) {
			this._stalled = true;
			this._forceload = true;
			this._forceLoadChangeCurrentTime();
		} else if (this._lastPercentChangeTime !== null && now - this._lastPercentChangeTime < 1000) {
			this._stalled = false;
		}

		if (this._loadDetails.percent !== percent && !(this._loadDetails.cached > 0 && this._loadDetails.cached === this._loadDetails.percent)) this._lastPercentChangeTime = now;
	},

	/**
	 * Setting current time to for forceload
	 * @private
	 */
	_forceLoadChangeCurrentTime: function () {
		var time = this.duration() * (this._loadDetails.percent / 100);
		this.log('Video: Force Loading (time: ' + time + ')');
		this.element.currentTime = time;
	},

	/**
	 * manual progress event that does additional calculations
	 *
	 * @private
	 */
	_dispatchProgress: function () {

		// set some current values
		var currentTime = new Date().getTime(),
			percent = this._loadDetails.percent;

		// check the time taken to load
		this._loadDetails.timeTaken = (currentTime - this._startTime) / 1000;
		// percent calculation
		if (this.element && this.element.buffered && this.element.buffered.length) {
			percent = (this.element.buffered.end(0) / this.duration()) * 100;
		}

		// check if IE is 90% cached to force to 100
		if ((MAL.Environment.browser === 'msie' && percent > 90) || (MAL.Environment.browser === 'chrome' && percent > 98) || (MAL.Environment.os === 'ipad' && MAL.Environment.osVersion >= 8 && percent > 98)) percent = 100;

		// check if we are 100% cached, then dispatch a cached event
		if (percent === 100 && this._loadDetails.percent === 0)
			this.dispatchEvent('cached', {percent:percent});

		if (!this._played) this._checkStalled(percent);

		// set current percent to load details
		this._loadDetails.percent = percent;

		// when the video is fully loaded
		if (this._loadDetails.percent >= 100) {
			this.canPlayThrough = this._loaded = true;
			this._loadDetails.bufferPercent = 100;
			this._loadDetails.bufferTime = 0;
			this._loadDetails.averageBufferTime = 0;
			this._loadDetails.timeNeeded = 0;
			this.dispatchEvent('progress', this._loadDetails);
			if (this._forceload) this.element.currentTime = 0;

			if (this.malCanPlayThrough === false) {
				this.malCanPlayThrough = true;
				this._malCanPlayThrough();
			}

			return;
		}

		// we start the load calculations beyond this point, this gets set to true in _onLoadedMetaData
		if (!this._loadstart) return;

		// initial loop after the data loaded event
		if (this._loops === 0) {
			this._loadDetails.cached = this._loadDetails.percent;
			this.dispatchEvent('cached', {percent:percent});
			this._startTime = currentTime;
		}

		// increment the loops after initial chcek has been done
		this._loops++;


		// get percent loaded from the current percent minus the cachedPercent, and percent needed till full load
		var	percentLoaded = (this._loadDetails.percent - this._loadDetails.cached)/100,
			percentNeeded = (100-this._loadDetails.percent) / 100;

 		this._loadDetails.percentLoaded = (percentLoaded*100);
 		this._loadDetails.percentNeeded = (percentNeeded*100);


		// time needed till 100% loaded
		if (percentLoaded > 0 ) this._loadDetails.timeNeeded = (this._loadDetails.timeTaken/percentLoaded) * percentNeeded;

		// if timeNeeded is Infinity we set it to 0
		if (isNaN(this._loadDetails.timeNeeded)) this._loadDetails.timeNeeded = 0;

		// total time Needed till 100% from timeNeeded plus how long it allready took
		this._loadDetails.timeNeededTotal = this._loadDetails.timeNeeded + this._loadDetails.timeTaken;

		// calculate time needed till we can play, based on time needed to 100% and susctract the play time from that
		this._loadDetails.bufferTime = (this._loadDetails.timeNeeded - this.duration());

		// when the bufferTime Needed to play is lower then 0 it should be ready and set reset it to 0
		if (this._loadDetails.bufferTime < 0) this._loadDetails.bufferTime = 0;

		// calculate buffer percent based on time needed and time taken
		if (this._loadDetails.bufferPercent !== 100 && (this._loadDetails.timeTaken != this._loadDetails.timeNeededTotal)) {
			this._loadDetails.bufferPercent = (this._loadDetails.timeTaken/this._loadDetails.timeNeededTotal) * 100;
		}

		// calculate average buffer time needed
		var averageAdjustment = (this._loadDetails.timeNeeded-this._loadDetails.averageBufferTime)/this._loops;
		this._loadDetails.averageBufferTime += averageAdjustment;

		// dispatch the progress event after minimum load time is over
		if (this._loadDetails.timeTaken > this.minimumTime) {

			this.dispatchEvent('progress', this._loadDetails);
			if (!this._forceload && this._loadDetails.bufferPercent >= 100 && this.malCanPlayThrough === false) {
				this.malCanPlayThrough = true;
				this._malCanPlayThrough();
			}
		}

		//console.log(this._loadDetails);


	},

	/**
	 * When video can play through based on browser and our own determination, we fire the malcanplaythrough event
	 * @private
	 */
	_malCanPlayThrough: function() {

		if (this._dispatchedCanPlayThrough === true) return;

		if (this.malCanPlayThrough === true && this.canPlayThrough === true) {
			this._dispatchedCanPlayThrough = true;
			this.dispatchEvent('malcanplaythrough', false);
		}
	},

	/**
	 * When video is ready based on the browser, the canplaythrough event will be fired
	 * @param  {Event} e the original event from the video
	 * @private
	 */
	_onCanPlayThrough: function (e) {
		this.canPlayThrough = true;
		this._malCanPlayThrough();
		this._dispatchProgress();
		this.dispatchEvent('canplaythrough', false, e);
	},

	/**
	 * Setting volume when metadata is loaded, this is the best time to set volume
	 * @param {Event} e
	 * @private
	 */
	_setLoadedVolume: function (e) {
		this.setVolume(this._volume);
		this.log('setting volume to ',this._volume,' on metadataloaded');
	},

	/**
	 * Start the load process when metadata has been loaded.
	 * @param  {Event} e original event from the element
	 * @private
	 */
	_onLoadedMetaData: function (e) {
		this._loadstart = true;
		this.dispatchEvent('loadedmetadata', false, e);
	},

	/**
	 * When the video has ended
	 * @param  {Event} e original event from the video element
	 * @private
	 */
	_onEnded: function (e) {
		this.dispatchEvent('ended', false, e);
		this.playing = false;
	}
}));

/**
 * @author Martijn Korteweg <martijn@mediaartslab.com>
 * @docauthor Martijn Korteweg <martijn@mediaartslab.com>
 *
 *
 * video should control all video element interactions. The Advanced video class added several options to the base video class
 *
 * - Bandwidth Calculation
 * - Animated Volume fade
 * - Que Points
 *
 * #Usage
 *
 * Initialize your video object first
 *
 * 		var video = new MAL.video.Advanced({
 *	  			controls: false,
 * 		 	minimumTime: 1,
 * 		  	volume: 0,
 * 		   	autobuffer: true
 *      	});
 *
 * Then you can set your source object
 *
 * 		video.Advanved.setSource({
 * 			path: 'http://somwehre',
 * 		 	file: 'videoname'
 * 		  	type: 'mp4', // override only
 * 		   	filePathProxy: RMV.filePathProxy
 *       	});
 *
 * #Bandwidth
 * To calculate the bandwidth from the video file you can used calculateBandwidth when the object is initilized
 *
 * 		video.Advanced.calculateBandwidth({
 * 			mp4: 4.2,
 * 			webm: 4.1
 * 		});
 *
 * #Fade Volume
 * You can fade the volume of the video over time
 *
 * 		video.Advanced.fadeVolume({
 * 			delay: 0,
 * 			volume: 1,
 * 			duration: 2,
 * 			ease: [0,0,1,1]
 * 		});
 *
 * #Que Points
 * Add que points to any moment in time of the video, all Quepoints are based on seconds in the video
 *
 * 		video.Advanced.addCuePoint(3,'bigboom');
 *
 * 		video.Advanced.addEventListener('bigboom',function () {
 * 			//do something
 * 		});
 * 		
 * @class MAL.video.Advanced
 * @extends MAL.video.Base
 *
 */
MAL.define('video.Advanced', MAL.video.Base.extend({
	/**
	 * initializes MAL.video.Advanced
	 *
	 * @method constructor	 
	 *
	 *
	 */
	
	/**
	 * @cfg {Number} cuePointInterval
	 * By default quepoints get fired on the timeupdate event, this option can override that to a timeout.
	 */
	initialize: function ($config) {
		this._options = $config || {};

		this._super(this._options);

		// the quepoints can be set to a specific check interval to tweak hit timing
		this._cuePointInterval = this._options.cuePointInterval;

		// quepoints array
		this._cuePoints = [];

		this._volumeOptions = {};

		// have we allready given a bandwidth reading
		this._dispatchedBandwidth = false;

		// video size set in Mb
		this._videoSize = 0;
	},

	/**
	 * Changing volume with a fade
	 * @param  {Object} $config
	 * @param {Number} $config.delay delay before fade happens
	 * @param {Number} $config.volume the target volume to fade towards
	 * @param {Number} $config.duration duration of the fade
	 * @param {Array} $config.ease Cubic Bezier Ease array
	 *
	 *		video.Advanced.fadeVolume({
	 *	 		delay: 0,
	 *	   		volume: 1,
	 *	     	duration: 2,
	 *	      	ease: [0,0,1,1]
	 *		});
	 * 
	 */
	fadeVolume: function($config) {

		$config = $config || {};

		// delay of volume change
		this._volumeOptions.delay = $config.delay || 0;

		// new volume
		this._volumeOptions.volume = $config.volume;

		// duration of the volume change
		this._volumeOptions.duration = $config.time || $config.duration || 0.25;

		// easing of the volume change
		this._volumeOptions.ease = $config.ease || [0,0,1,1];

		// if there is a startwidth it will be changed to that volume instantly
		this.element.volume = this._volumeOptions.currVolume = $config.startWidth || this.element.volume;

		// set the muted to false
		this.element.muted = false;

		// start time of the animation
		this._volumeOptions.startTime = new Date().getTime();

		// kickoff the fade loop
		this._fadeLoop();
	},

	/**
	 * 
	 * Loop for fading animation
	 * @private
	 *
	 */
	_fadeLoop: function() {

		var now = new Date().getTime(),
			cT = ((now-this._volumeOptions.startTime)/1000)-this._volumeOptions.delay;

		// wait for delay
		if (cT <= 0) {
			setTimeout(this._fadeLoop,(1000/60));
			return;
		}

		var	b = this._volumeOptions.currVolume,
			c = this._volumeOptions.volume - b,
			d = this._volumeOptions.duration,
			a = ((1 / d) * cT),
			ease = this._volumeOptions.ease,
			secpos = MAL.cubicBezierWithPercent(a, ease[0], ease[1], ease[2], ease[3]),
			newVolume = ((c * secpos) + b);

		if (cT > this._volumeOptions.duration) {
			newVolume = this._volumeOptions.volume;
		} else {
			setTimeout(this._fadeLoop,(1000/60));
		}

		var volume = Math.round((newVolume)*1000)/1000;
		this.setVolume(volume);
	},

	/**
	 * Stop video from loading by overriding the source
	 *
	 * 		video.Advanced.setBlankSource();
	 */
	setBlankSource: function() {
		this._loaded = true;
		this.canPlayThrough = false;
		this.malCanPlayThrough = false;
		this.element.src = 'data:video/' + this.type + ';base64,' + this._blanks[this.type];
		this.element.parentNode.removeChild(this.element);
		this.element = {};
	},

	/**
	 * Adding quepoint to the video
	 *
	 * Quepoints are based on events, when you add a quepoint there will be an event on the this
	 * instance with the same name as your quepoint.
	 *
	 * @param {Number} time At what time does this need to get fired
	 * @param {String} name Event name that gets fired
	 *
	 * 		video.Advanced.addCuePoint(3,'bigboom');
	 *
	 * 		video.Advanced.addEventListener('bigboom',function () {
	 * 			//do something
	 * 		});
	 */
	addCuePoint: function(time, name) {

		if (this._cuePoints.length === 0) {
			if (!MAL.isUndefined(this._cuePointInterval)) {
				setInterval(this._fireCuePoints, this._cuePointInterval);
			} else {
				this.element.addEventListener('timeupdate', this._fireCuePoints, true);
			}
			// when a seek has happend on the video, the quepoints will be reset.
			this.element.addEventListener('seeked', this._resetCuePoints, true);
		}

		// push the quepoint to the array
		this._cuePoints.push({
			time: time,
			name: name,
			fired: false
		});
	},

	/**
	 * Reset quepoint to make sure it can fire again
	 * @param  {String} name name of the quepoint
	 * @private
	 */
	_resetCuePoint: function (name) {
		for (var i = 0, l = this._cuePoints.length; i < l; i++) {
			if (this._cuePoints[i].name === name) {
				this._cuePoints[i].fired = false;
				break;
			}
		}
	},

	/**
	 * Reset all quepoints
	 *
	 * Bugzid: 4903, added setTimeout for safari 5.1 to catch up on seeked event.
	 * @private
	 */
	_resetCuePoints: function (e) {
		setTimeout(function () {
			this._cuePoints.forEach(function(value, key, ref) {
				if (this.element.currentTime < value.time)
					value.fired = false;
			}.bind(this));
		}.bind(this),10);
	},

	/**
	 * Fire the event for the quepoint
	 * @private
	 */
	_fireCuePoints: function () {
		// check if the video is playing
		if (!this.playing) return;

		// currentTime of the video
		var time = this.element.currentTime;

		// set current scope
		var _scope = this;

		// run through all quepoints to fire them
		this._cuePoints.forEach(function(value, key, ref) {
			if (value.fired || time < value.time) return;
			_scope.dispatchEvent(value.name);
			value.fired = true;
		});

	},

	/**
	 * start bandwidth calculation
	 *
	 * @param {Object} sizeObject object of Mb sizes from all videos served
	 * @param {Number} sizeObject.mp4
	 * @param {Number} sizeObject.webm
	 * 
	 */
	calculateBandwidth: function(sizeObject) {
		if (typeof sizeObject[this.type] === 'undefined')
			return;

		// initial bandwidth set
		this._bandwidth = {
			average:0,
			readings:0,
			history:[]
		};

		// set the video size
		this._videoSize = sizeObject[this.type];

		// listen to the progress event on this object
		this.addEventListener('progress',this._calculateBandwidthLoop.bind(this), false);
	},

	/**
	 * Loop bandwidth
	 * @private
	 * @param  {Object} o Event
	 * @param  {Object} e Passed variabled from the calculations
	 */
	_calculateBandwidthLoop: function (o,e) {

		// lets not calculate when cached
		if (MAL.isUndefined(e.cached) || e.cached === 100 || e.timeTaken === 0) return;

		// if there is a video size we should calculate the current bandwidth
		if (this._videoSize > 0) {
			this._bandwidth.readings++;

			var bandwidth = (((this._videoSize * 1024) * ((e.percent-e.cached)/100) * 8) / e.timeTaken);

			this._bandwidth.average += (bandwidth-this._bandwidth.average)/this._bandwidth.readings;
			this._bandwidth.history.push(bandwidth);
		}

		// when load is done, lets show the average bandwidth.
		if (e.percent === 100 && this._dispatchedBandwidth === false) {
			this._dispatchedBandwidth = true;
			this.dispatchEvent('averagebandwidth', this._bandwidth);
		}

		// dispatch current bandwidth
		this.dispatchEvent('bandwidth', this._bandwidth);
	}

}));

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Stage class. Provides an animation environment for layers ({@link MAL.tween.CSSLayer}/{@link MAL.tween.JSLayer})
 *
 * **Usually you want to only have ONE stage**
 *
 * # Creating a stage
 *
 * create stage
 *     var stage = new MAL.tween.Stage();
 *
 * alias a getter
 *     var $ = stage.getLayers;
 *
 * add a tween to a layer
 *     $('Box').addTween({delay: .2, duration: .5, finish: {x: 100}});
 *
 * start stage animations
 *     stage.start();
 *
 * # Creating a stage with both JS and CSS layers (make sure you need to do this, this is usually a last resort)
 *
 * so sometimes you may want to force some elements to be CSS driven animations and some to be JS driven
 *
 * create the special stage that forces all tweens to be in JS unless overridden
 *     var stage = new MAL.tween.Stage({dualLayer: true, layerType: 'JS'});
 *
 * you can override a layer type like this
 *     stage.addCSSLayers('Box'); // this layer will animate with CSS
 *
 * so this is an example of two layers both being driven by CSS and JS
 *     $('Box', 'Box2').addTween({delay: .2, duration: .5, finish: {x: 100}});
 *
 * Box will be animated with CSSLayer and Box2 will be animated with JSLayer
 *
 * @class MAL.tween.Stage
 */

 /**
 * dispatched when a layer has been added to the stage
 *
 * @event layer-add
 * @param {Event} this
 * @param {Object} message {layer:}
 */

MAL.define('tween.Stage', MAL.Object.extend({
	/**
	 * initialize
	 *
	 * @method constructor
	 * @param {Object} config
	 */

	/**
	 * @cfg {String} layerType
	 * type of layer to use (CSS or JS). **be sure that the layer type is loaded before doing this**
	 */

	/**
	 * @cfg {String} dualLayer
	 * type you should set this to true if you will be using the layerType over ride methods
	 * - {@link #addJSLayer}
	 * - {@link #addCSSLayer}
	 */
	initialize: function ($config) {
		this._super();

		// handle options
		$config = $config || {};

		// object of layers on the stage
		this._layerCache = {};
		this._layers = [];
		this._animating = new MAL.Set();
		this._startAnimations = new MAL.Set();
		this._dualLayer = !!$config.dualLayer;
		this._positionAnchorPointElement = $config.positionAnchorPointElement;

		// set started default
		this.started = false;
		this._startTime = null;

		//
		this._layerType = $config.layerType || (MAL.Environment.bugs.transform3d ? 'JS' : false) || (MAL.Environment.getCSSPropertyName('transition') ? 'CSS' : 'JS');

		this._timeout = null;

		// timeout manager
		this.timeouts = new MAL.TimeoutManager({leniency: 0});

		if (this._dualLayer || this._layerType === 'JS') this._loop();
	},

	/**
	 * update loop
	 *
	 * @private
	 */
	_loop: function () {
		var time = new Date().getTime();

		this._animating.forEach(function (layer) {
			if (layer instanceof MAL.tween.JSLayer || layer instanceof MAL.tween.CSSJSLayer) layer._update(time);
		});

		if (MAL.Environment.requestAnimationFrame) {
			window[MAL.Environment.requestAnimationFrame](this._loop);
		} else {
			window.clearTimeout(this._timeout);
			this._timeout = window.setTimeout(this._loop, 0);
		}
	},

	/**
	 * simple CSS layer flagging
	 *
	 *     stage.addCSSLayers('Box', 'Cube');
	 *
	 * or
	 *
	 *     stage.addJSLayers(['Box', 'Cube']);
	 */
	addCSSLayers: function (layers) {
		layers = MAL.argumentsToArray(arguments);

		layers.forEach(function (layer) {
			this.addCSSLayer(layer);
		}, this);
	},

	/**
	 * simple JS layer flagging
	 *
	 *     stage.addJSLayers('Box', 'Cube');
	 *
	 * or
	 *
	 *     stage.addJSLayers(['Box', 'Cube']);
	 */
	addJSLayers: function (layers) {
		layers = MAL.argumentsToArray(arguments);

		layers.forEach(function (layer) {
			this.addJSLayer(layer);
		}, this);
	},

	/**
	 * simple CSSJS layer flagging
	 *
	 *     stage.addCSSJSLayers('Box', 'Cube');
	 *
	 * or
	 *
	 *     stage.addCSSJSLayers(['Box', 'Cube']);
	 */
	addCSSJSLayers: function (layers) {
		layers = MAL.argumentsToArray(arguments);

		layers.forEach(function (layer) {
			this.addCSSJSLayer(layer);
		}, this);
	},

	/**
	 * add forced CSS layer to the dualLayer stage
	 *
	 *     var Box = stage.addCSSLayer('Box');
	 *
	 * you can also add a layer with properties
	 *
	 *     var Box = stage.addCSSLayer('Box', {x: 50, y: 50, scale: 2});
	 *
	 * @param {String/HTMLElement} reference ID of the element (element id or instance)
	 * @param {Object} properties properties the layer should have
	 * @return {MAL.tween.CSSLayer}
	 */
	addCSSJSLayer: function (reference, properties) {
		if (!this._dualLayer) MAL.error('Stage: dualLayer support needs to be enabled');
		if (!MAL.tween.CSSLayer) MAL.error('MAL.tween.CSSLayer is missing');
		return this.addLayer(reference, properties, 'CSSJS');
	},

	/**
	 * add forced CSS layer to the dualLayer stage
	 *
	 *     var Box = stage.addCSSLayer('Box');
	 *
	 * you can also add a layer with properties
	 *
	 *     var Box = stage.addCSSLayer('Box', {x: 50, y: 50, scale: 2});
	 *
	 * @param {String/HTMLElement} reference ID of the element (element id or instance)
	 * @param {Object} properties properties the layer should have
	 * @return {MAL.tween.CSSLayer}
	 */
	addCSSLayer: function (reference, properties) {
		if (!this._dualLayer) MAL.error('Stage: dualLayer support needs to be enabled');
		if (!MAL.tween.CSSLayer) MAL.error('MAL.tween.CSSLayer is missing');
		return this.addLayer(reference, properties, 'CSS');
	},

	/**
	 * add forced JS layer to the dualLayer stage
	 *
	 *     var Box = stage.addJSLayer('Box');
	 *
	 * you can also add a layer with properties
	 *
	 *     var Box = stage.addJSLayer('Box', {x: 50, y: 50, scale: 2});
	 *
	 * @param {String/HTMLElement} reference ID of the element (element id or instance)
	 * @param {Object} properties properties the layer should have
	 * @return {MAL.tween.JSLayer}
	 */
	addJSLayer: function (reference, properties) {
		if (!this._dualLayer) MAL.error('Stage: dualLayer support needs to be enabled');
		if (!MAL.tween.JSLayer) MAL.error('MAL.tween.JSLayer is missing');
		return this.addLayer(reference, properties, 'JS');
	},

	/**
	 * add layer to the stage
	 *
	 *     var Box = stage.addLayer('Box');
	 *
	 * you can also add a layer with properties
	 *
	 *     var Box = stage.addLayer('Box', {x: 50, y: 50, scale: 2});
	 *
	 * @param {String/HTMLElement} reference ID of the element (element id or instance)
	 * @param {Object} properties properties the layer should have
	 * @return {MAL.tween.JSLayer/MAL.tween.CSSLayer}
	 */
	addLayer: function (reference, properties, _layerType) {
		// restrict scope
		var element;

		// detect if element exists
		if (typeof reference === 'object' && reference.nodeType > 0) {
			element = reference;
		} else {
			MAL.error('InvalidElement:', reference);
		}

		// check if layer is in stage
		if (element.MALTweenLayer) MAL.error('LayerExists: layer "' + reference + '" is already in stage');

		// use the correct layer object
		var layerType = _layerType || this._layerType;

		var layer = new MAL.tween[layerType + 'Layer']({stage: this, element: element, begin: properties, positionAnchorPointElement: this._positionAnchorPointElement});

		// add listeners
		layer.addEventListeners({
			'animation-start': this._onAnimationStart,
			'animation-end': this._onAnimationEnd,
			'animation-add': this._onAnimationAdd
		});

		// directly link layer to element
		element.MALTweenLayer = layer;

		// dispatch layer add event
		this.dispatchEvent('layer-add', {layer: element.MALTweenLayer});

		// push the layer onto the stack
		this._layers.push(layer);

		// add helper reference
		if (typeof reference === 'string') this._layerCache[reference] = element.MALTweenLayer;

		// return layer
		return element.MALTweenLayer;
	},

	/**
	 * add multiple layers to stage without start properties
	 *
	 *     var layers = stage.addLayers('Box', 'Rectangle', 'Circle');
	 *
	 * obtain one of the layers you added and animate on it
	 *
	 *     layers.Box.addTween({duration: 1, finish: {x: 50, y: 50}});
	 *
	 * @param {Array} layers set of layers
	 * @return {Object} layers
	 */
	addLayers: function (layers) {
		// convert arguments into an array
		layers = MAL.argumentsToArray(arguments);

		var references = {};

		// traverse layer array
		layers.forEach(function (layer) {
			// add layer to stack
			references[layer] = this.addLayer.apply(this, [layer]);
		}, this);

		return references;
	},

	/**
	 * get a layer reference from the element ID
	 *
     *     var Box = stage.getLayer('Box');
	 *
	 * @param {String/Object} reference ID of the element (element id or instance)
	 * @return {MAL.tween.JSLayer/MAL.tween.CSSLayer}
	 */
	getLayer: function (reference) {
		// restrict scope
		var element;

		if (typeof reference === 'object' && MAL.isUndefined(reference.nodeType) && MAL.tween.Layer) {
			return reference;
		} else if (typeof reference === 'object' && reference.nodeType > 0) {
			// detect if element exists
			element = reference;
		}

		if (!element.MALTweenLayer) this.addLayer(reference);//MAL.error('MissingLayer: layer "' + reference + '" is not on the stage');

		return element.MALTweenLayer;
	},

	/**
	 * creates a proxy layers object that can perform multiple methods on a set of layers
	 *
     *     var numbers = stage.getLayers('One', 'Two', 'Three');
     *
     * or
     *
     *     var numbersArray = ['One', 'Two', 'Three'],
     *     numbers = stage.getLayers(numbersArray);
     *
     * then we can set a property on all the layers
     *
     *     numbers.setProperty('opacity', 0);
     *
     * you can also use the each method to do something with the layers
     *
     *     numbers.each(function (layer) {
	 *      console.log(layer.element);
     *     });
     *
     * or you can just pull an element from the proxy object like this
     *
     *     var first = numbers[0];
	 *
	 * @param {Array} reference ID of the element (element id or instance)
	 * @return {Object} proxy layer object
	 */
	getLayers: function (layers) {
		// convert arguments into an array
		layers = MAL.argumentsToArray(arguments);

		// if we only have one layer there is no huge advantage of using the proxy
		if (layers.length === 1) return this.getLayer(layers[0], true);

		// update query array to hold the actual Layer objects
		layers.forEach(function (value, key, array) {
			array[key] = this.getLayer(value, true);
		}, this);

		//new ExampleProxy({self: window, layers: [function () {}]});
		return new MAL.tween[this._layerType + 'LayerProxy']({layers: layers});
	},

	/**
	 * layer animation starts
	 *
	 * @private
	 */
	_onAnimationStart: function (o, e) {
		// add layer to animating pool
		this._animating.add(o);
	},

	/**
	 * layer animation ends
	 *
	 * @private
	 */
	_onAnimationEnd: function (o, e) {
		if (o._tweens.length) return;

		// remove layer to animating pool
		this._animating.remove(o);
	},

	/**
	 * layer aniamtion adds
	 *
	 * @private
	 */
	_onAnimationAdd: function (o, e) {
		// add layer to start animations pool if the stage hasn't started yet
		if (!this.started) this._startAnimations.add(o);
	},

	/**
	 * start stage animation
	 *
	 *     stage.start();
	 *
	 */
	start: function () {
		this.started = true;
		this._startTime = new Date().getTime();
		this._layers.forEach(function (layer) {
			if (layer.layerType === 'CSS') layer._setCSSTextFromBuffer();
		});

		this._startAnimations.forEach(function (layer) {
			layer._startNextAnimation();
		});
	}
}));

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * basics for any layer (should only be extended upon)
 *
 * @class MAL.tween.Layer
 * @extends MAL.Object
 * @private
 */
MAL.define('tween.Layer', MAL.Object.extend({
	/**
	 * initializes MAL.tween.Layer
	 *
	 * @method constructor
	 * @param {Object} config
	 */
	initalize: function ($config) {
		this._super();
	},

	/**
	 * set the layers data
	 *
	 * @param  {Object} data
	 */
	setData: function (data) {
		if (!this._data) this._data = {};

		for (var property in data) {
			this._data[property] = data[property];
		}

		return this;
	},

	/**
	 * get the layers data
	 *
	 * @param  {String} name
	 * @return {String/Object/Number} data
	 */
	getData: function (name) {
		if (!this._data) this._data = {};

		if (name) {
			return this._data[name];
		} else {
			return this._data;
		}
	},

	/**
	 * add an object of events to the layers element
	 *
	 *     layer.addEvents({
	 *      click: function (e) {},
	 *      mouseover: function (e) {}
	 *     });
	 *
	 * @param {Object} events
	 */
	addEvents: function (events) {
		for (var type in events)
			this.addEvent(type, events[type]);
	},

	/**
	 * add a event to the layers element
	 *
	 *     layer.addEvent('click', function (e) {
	 *     }};
	 *
	 * @param {String} type event name
	 * @param {Function} callback
	 */
	addEvent: function (type, callback) {
		MAL.addEvent(this.element, type, callback);
	},

	/**
	 * returns a RegExp that correctly matches for a className
	 *
	 * @private
	 * @param {String} className
	 */
	_classRegExp: function (className){
		return new RegExp("(^|\\s)" + className + "(\\s|$)");
	},

	/**
	 * checks if the layer has the requested class
	 *
	 *     if (layer.hasClass('Selected')) {
	 *      // layer has the class
	 *     }
	 *
	 * @param {String} className
	 * @param {String} prefix optional prefix override
	 */
	hasClass: function (className, prefix) {
		prefix = MAL.isUndefined(prefix) ? 'mal-layer-' : prefix;
		return this.element.classList ? this.element.classList.contains(prefix + className) : this._classRegExp(prefix + className).test(this.element.className);
	},

	/**
	 * add a set of classes to the layer
	 *
	 *     layer.addClass(['Box', 'Selected']);
	 *
	 * or add one
	 *
	 *     layer.addClass('Selected');
	 *
	 * @param {String/Array} classNames
	 * @param {String} prefix optional prefix override
	 */
	addClass: function (classNames, prefix) {
		classNames = MAL.isArray(classNames) ? classNames : classNames.split(' ');
		prefix = MAL.isUndefined(prefix) ? 'mal-layer-' : prefix;

		var i, l;

		// use native method if supported
		if (this.element.classList) {
			for (i = 0; i < classNames.length; i++) this.element.classList.add(prefix + classNames[i]);
			return;
		}

		// use manual method because its not native
		var classArray = this.element.className.split(' '),
			changed = false;

		for (i = 0, l = classNames.length; i < l; i++) {
			if (this.hasClass(classNames[i], arguments[1])) continue;
			classArray.push(prefix + classNames[i]);
			changed = true;
		}

		if (changed) this.element.className = MAL.trim(classArray.join(' '));

		return this;
	},

	/**
	 * remove set of classes from the layer
	 *
	 *     layer.removeClass(['Box', 'Selected']);
	 *
	 * or remove one
	 *
	 *     layer.removeClass('Selected');
	 *
	 * @param {String/Array} classNames
	 * @param {String} prefix optional prefix override
	 */
	removeClass: function (classNames, prefix) {
		classNames = MAL.isArray(classNames) ? classNames : classNames.split(' ');
		prefix = MAL.isUndefined(prefix) ? 'mal-layer-' : prefix;

		var i, l;

		// use native method if supported
		if (this.element.classList) {
			for (i = 0; i < classNames.length; i++) this.element.classList.remove(prefix + classNames[i]);
			return this;
		}

		// use manual method because its not native
		var classArray = this.element.className.split(' '),
			changed = false;

		for (i = 0, l = classNames.length; i < l; i++ ) {
			if (!this.hasClass(classNames[i], arguments[1])) continue;
			classArray.splice(classArray.indexOf(prefix + classNames[i]), 1);
			changed = true;
		}

		if (changed) this.element.className = MAL.trim(classArray.join(' '));

		return this;
	},

	/**
	 * overridden left setter
	 *
	 * @private
	 */
	_cssSetLeft: function (value) {
		this._initialProperties.left = value;

		return {
			name: 'left',
			value: value + this._suffixes.left
		};
	},

	/**
	 * overridden top setter
	 *
	 * @private
	 */
	_cssSetTop: function (value) {
		this._initialProperties.top = value;

		return {
			name: 'top',
			value: value + this._suffixes.top
		};
	},

	/**
	 * pull the value from the elements computed/current style
	 *
	 * @param {String} property name of css property
	 * @param {Boolean} raw return string value
	 */
	cssGetComputedProperty: function (property, raw) {
		var string;

		property = MAL.Environment.getCSSPropertyName(property, true);

		if (!property) return false;

		if (window.document.defaultView && window.document.defaultView.getComputedStyle) {
			string = window.document.defaultView.getComputedStyle(this.element, '').getPropertyValue(property);
			return raw ? string : MAL.parseNumber(string);
		} else if (this.element.currentStyle) {
			string = this.element.currentStyle[property];
			return raw ? string : MAL.parseNumber(string);
		}
	},

	/**
	 * turns string values into floats if they match the criteria
	 *
	 * @private
	 * @param  {String} value
	 * @return {String/Number}
	 */
	_parsePropertyValue: function (value) {
		var floatValue = parseFloat(value);

		if (MAL.isString(value) && value.replace(/(?:px|deg)$/, '') === String(floatValue)) value = floatValue;
		return value;
	}
}));

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * creates a proxy object for layers that allows us to group operations through a simple interface
 *
 * @class MAL.tween.LayerProxy
 * @private
 */

/**
 * initializes MAL.tween.LayerProxy
 *
 * @method constructor
 * @param {Object} config
 */
MAL.define('tween.LayerProxy', function (object) {
	var Proxy = function ($config) {
		this.layers = $config.layers;
		this.layers.forEach(function (layer, key) {
			this[key] = layer;
		}, this);
	};

	Proxy.prototype = {
		/**
		 * will run function against all matching layers
		 *
		 *     $('One','Two').each(function (layer) {
		 *      console.log(layer);
		 *     });
		 *
		 * @param {Object} events
		 */
		each: function (callback) {
			this.layers.forEach(function (layer) {
				callback.call(layer, layer);
			});

			return this;
		}
	};

	for (var method in object.prototype) {
		// ensure that the methods are public
		if (method.substr(0,1) !== '_') {
			Proxy.prototype[method] = (function (method) {
				return function () {
					var args = MAL.toArray(arguments);

					this.layers.forEach(function (layer) {
						MAL.tween[layer.layerType + 'Layer'].prototype[method].apply(layer, MAL.clone(args, true));
					});

					return this;
				};
			}(method));
		}
	}

	return Proxy;
});

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * @class MAL.tween.CSSLayer
 * @extends MAL.tween.Layer
 *
 * **You should not have to instantiate this class directly, it is managed by {@link MAL.tween.Stage}.**
 *
 * # Layers
 *
 * we have a standard of aliasing the layer getter with $
 *     var $ = stage.getLayers;
 *
 * obtaining a layer (this actually creates a Layer object if one does not already exist)
 *     $('box');
 *
 * obtaining multiple layers (this will create a layer proxy object)
 *     $('box', 'box2');
 *
 * setting a property on a layer
 *     $('box').setProperty('opacity', 0);
 *
 * setting a property on multiple layers
 *     $('box', 'box2').setProperty('opacity', 0);
 *
 * logging out the selected layers out of the proxy
 *
 *     $('box', 'box2').each(function (layer) {
 *      console.log(layer);
 *     });
 *
 * if for some reason you cant to find out if a layer has went through a proxy you can do this
 *     document.getElementById('mal-ad-box').MALTweenLayer; // this property would reference the layer instance
 *
 * # Helpers
 *
 * you can also string things
 *
 *     $('box', 'box2').clearAll().addTween({duration: 1, finish: {x: 100}});
 *
 * # Animating on a layer
 *
 * we only want to do one animation with this layer so we use {@link #addTween}
 *     layer.addTween({duration: 1, finish: {x: 100}}); // animate layer to {x:100} in a second
 *
 * but if we want to do a set of animations in a specific order we would use {@link #addTweenSet}
 *
 *     layer.addTweenSet(
 *      {duration: 1, finish: {x: 100}},
 *      {duration: 1, finish: {x: 0}},
 *      {duration: 1, finish: {y: 100}},
 *      {duration: 1, finish: {y: 0}}
 *     );
 *
 * here is a runnable example of adding a tween set
 *
 *     @example
 *     // add a layer (there is an element already in the dom with an ID of mal-ad-box)
 *     // we are also setting some initial css properties
 *     var layer = stage.addLayer('box', {width: 25, height: 25, 'background-color': '#085083'});
 *
 *     // you can also add multiple tweens like this
 *     layer.addTweenSet(
 *
 *         // wait 500 ms then animate to {x:500} with an ease of 'ease-in-out'
 *         {delay: .5, duration: .5, finish: {x: 500}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {x: 0, y: 100}
 *         {duration: .5, finish: {x: 0, y: 100}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {rotate: 45}
 *         {duration: .5, finish: {rotate: 45}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {x: 500, y: 0}
 *         {duration: .5, finish: {x: 500, y: 0}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {x: 0} then disappear
 *         {duration: .5, finish: {x: 0, y: 0}, easing: 'ease-in-out', callback: function (layer) {
 *          layer.setProperty('visibility', 'hidden');
 *         }}
 *
 *     );
 *
 * # Dealing with bugs and features
 *
 * all layers will inherit the bugs/features settings that are set through the {@link MAL.Environment}, but you can override them on a per layer basis with {@link #setFeatures} and {@link #setBugs}.
 *
 * because a lot of our animations depend on the bleeding edge
 */

/**
 * dispatched when an animation has been added
 *
 * @event animation-add
 * @param {Event} this
 */

/**
 * dispatched when an animation has started
 *
 * @event animation-start
 * @param {Event} this
 */

/**
 * dispatched when an animation has ended
 *
 * @event animation-end
 * @param {Event} this
 */
MAL.define('tween.CSSLayer', MAL.tween.Layer.extend({
	/**
	 * initializes MALTweenCSSLayer
	 *
	 * @method constructor
	 */

	/**
	 * @cfg {MAL.tween.Stage} stage that the layer will be added to
	 */

	/**
	 * @cfg {HTMLElement} element direct element reference
	 */

	/**
	 * @cfg {Object} begin properties that the layer should begin with
	 */

	/**
	 * @cfg {HTMLElement} positionAnchorPointElement (optional)
	 */
	initialize: function ($config) {
		this._super($config);

		// type of layer
		this.layerType = 'CSS';

		// establish element association
		this.element = $config.element;
		this.element.MALTweenLayer = this;

		// layer name / ref
		this.name = this.element.id ? this.element.id : this.element;

		// animation iteration count
		this.iteration = 0;

		// positioning element
		this._positionAnchorPointElement = $config.positionAnchorPointElement;

		/**
		 * @property {Boolean} useAnchorPointOffset
		 * If set to true the layer will always add the positionAnchorPointOffset to x/y movement
		 */
		this.useAnchorPointOffset = false;

		/**
		 * @property {Object} positionAnchorPointOffset
		 * Manages the positionAnchorPointOffset, you an get the x offset by using layer.positionAnchorPointOffset.x / layer.positionAnchorPointOffset.y
		 */
		this.positionAnchorPointOffset = {x: 0, y: 0};

		if (this._positionAnchorPointElement) {
			var anchorPointElementPosition = MAL.findElementPosition(this._positionAnchorPointElement),
				elementPosition = MAL.findElementPosition(this.element);

			this.positionAnchorPointOffset.x = anchorPointElementPosition.x - elementPosition.x;
			this.positionAnchorPointOffset.y = anchorPointElementPosition.y - elementPosition.y;
		}

		this._transformationsDisabled = false;

		// features and bugs
		this._initializedFixes = false;
		this._features = {};
		this._bugs = {};
		this._initializeBugsAndFeatures();

		// sets a reference to the parent stage
		this._stage = $config.stage;

		// animation state
		this._animating = false;

		// signature for override verification
		this._signature = null;

		// persistant callback
		this._persistantCallback = null;

		// holds all tweens
		this._tweens = [];

		// started
		this._started = false;

		// cached properties
		this._cachedProperties = {};

		// css property buffer
		this._bufferedCSSProperties = {};

		// current applied transformations
		this._transformations = {};

		// properties that have been animated on
		this._animatedProperties = {};

		/** @private empty transition callback */
		this._transitionCallback = function () {};

		// get initial transformations and copy them into the layer
		// TODO: look into removing this.. as it doesn't do anything currently
		//this._initialTransformations = this._getComputedTransformations();

		// cached properties
		this._initialProperties = {
			left: this.cssGetComputedProperty('left'),
			top: this.cssGetComputedProperty('top'),
			width: this.cssGetComputedProperty('width'),
			height: this.cssGetComputedProperty('height')
		};

		// startWithTimeout
		this._startWithTimeout = null;

		// default anchor point (used for manual position calculations)
		this._transformOrigin = {x: '50%', y: '50%', z: 0};

		// should we be using backface on the layer?
		this._backface = false;

		// establish event listener (webkit is still using a browser prefix)
		MAL.addEvent(this.element, MAL.Environment.transitionEnd, this._onTransitionEnd);

		// adds a small amount of rotation to all elements to smooth animations
//		if (MAL.Environment.features.rotationPerformanceFix) this._transformations.rotate = '.01deg';

		// set begin properties
		if ($config.begin) this.setProperties($config.begin);
	},

	// ensures that the function operations happen in this order
	_transformationFunctionOrder: [
		'perspective',

		'preRotate',
		'preRotateX',
		'preRotateY',
		'preRotateZ',

		'translateX',
		'translateY',
		'translateZ',

		'skew',
		'skewX',
		'skewY',

		'scale',
		'scaleX',
		'scaleY',
		'scaleZ',

		'rotate',
		'rotateX',
		'rotateY',
		'rotateZ',

		'postX',
		'postY',
		'postZ'
	],

	// custom transformation function aliases
	_transformationFunctionAlias: {
		preRotate: 'rotate',
		preRotateX: 'rotateX',
		preRotateY: 'rotateY',
		preRotateZ: 'rotateZ',

		postX: 'translateX',
		postY: 'translateY',
		postZ: 'translateZ'
	},

	// property suffix list
	_suffixes: {
		perspective: 'px',
		width: 'px',
		height: 'px',
		left: 'px',
		top: 'px',
		skew: 'deg',
		skewX: 'deg',
		skewY: 'deg',
		x: 'px',
		y: 'px',
		z: 'px',
		preRotate: 'deg',
		preRotateX: 'deg',
		preRotateY: 'deg',
		preRotateZ: 'deg',
		rotate: 'deg',
		rotateX: 'deg',
		rotateY: 'deg',
		rotateZ: 'deg',
		postX: 'px',
		postY: 'px',
		postZ: 'px'
	},

	// property dictionary (used for ensuring default values)
	_transformationPropertyDictionary: {
		'x': {name: 'translateX', value: 0},
		'y': {name: 'translateY', value: 0},
		'z': {name: 'translateZ', value: 0},
		'transformOrigin': {name: 'transformOrigin', value: {x: '50%', y: '50%', z: 0}},
		'transformPerspective': {name: 'transformPerspective', value: 0},
		'skew': {name: 'skew', value: 0},
		'skewX': {name: 'skewX', value: 0},
		'skewY': {name: 'skewY', value: 0},
		'scale': {name: 'scale', value: 1},
		'scaleX': {name: 'scaleX', value: 1},
		'scaleY': {name: 'scaleY', value: 1},
		'scaleZ': {name: 'scaleZ', value: 1},
		'preRotate': {name: 'preRotate', value: 0},
		'preRotateX': {name: 'preRotateX', value: 0},
		'preRotateY': {name: 'preRotateY', value: 0},
		'preRotateZ': {name: 'preRotateZ', value: 0},
		'rotate': {name: 'rotate', value: 0},
		'rotateX': {name: 'rotateX', value: 0},
		'rotateY': {name: 'rotateY', value: 0},
		'rotateZ': {name: 'rotateZ', value: 0},
		'postX': {name: 'postX', value: 0},
		'postY': {name: 'postY', value: 0},
		'postZ': {name: 'postZ', value: 0}
	},

	/**
	 * copies the enviorments default bugs and features
	 *
	 * @private
	 */
	_initializeBugsAndFeatures: function () {
		for (var feature in MAL.Environment.features) this._features[feature] = MAL.Environment.features[feature];
		for (var bug in MAL.Environment.bugs) this._bugs[bug] = MAL.Environment.bugs[bug];
	},

	/**
	 * this should be called the first time we try to interface with the object
	 *
	 * @private
	 */
	_initializeFixes: function () {
		if (MAL.Environment.has3d && this._features.translateZPerformanceFix && MAL.isUndefined(this._transformations.translateZ)) this._transformations.translateZ = '0px';
		this._initializedFixes = true;
	},

	/**
	 * enables or disables bugs for the layer
	 *
	 *     layer.setBugs({
	 *      transform3d: true,
	 *      scale: true
	 *     });
	 *
	 * @param {Object} bugs
	 */
	setBugs: function (bugs) {
		for (var bug in bugs) this._bugs[bug] = bugs[bug];
		return this;
	},

	/**
	 * enables or disables features for the layer
	 *
	 *     layer.setFeatures({
	 *      backfacePerformanceFix: false,
	 *      translateZPerformanceFix: false
	 *     });
	 *
	 * @param {Object} features
	 */
	setFeatures: function (features) {
		for (var feature in features) this._features[feature] = features[feature];
		return this;
	},

	/**
	 * add a tween to a layer
	 *
	 *     layer.addTween({duration: 1, finish: {x: 100}});
	 *
	 * @param {MAL.tween.TweenObject} tween
	 */
	addTween: function (options) {
		if (this._transformationsDisabled) MAL.error('BadCommand: layer is currently disabled "' + this.name + '"');

		this._tweens.push(options);

		if (this._stage.started && !this._started) {
			this._started = true;
			this._startNextAnimation();
		}

		this.dispatchEvent('animation-add');
	},

	/**
	 * add tweens to a layer
	 *
	 *     layer.addTweenSet(
	 *      {duration: 1, finish: {x: 100}},
	 *      {duration: 1, finish: {x: 0}},
	 *      {duration: 1, finish: {y: 100}},
	 *      {duration: 1, finish: {y: 0}}
	 *     );
	 *
	 * @param {MAL.tween.TweenObject...} tweens
	 */
	addTweenSet: function (tweens) {
		// convert arguments into an array
		tweens = MAL.argumentsToArray(arguments);

		// add tweens
		tweens.forEach(function (tween) {
			this.addTween(tween);
		}, this);
	},

	/**
	 * add a persistant callback, usually used to dynamic animations like this
	 *
	 *     layer.addTweenPersistantCallback(function (layer) {
	 *      // this get fired after every tween
	 *      if (layer.iteration <= 5) {
	 *       // add a tween with an alternating finish animation based on the next iteration
	 *       layer.addTween({
	 *        duration: .2,
	 *        finish: {
	 *         x: this.iteration % 2 ? 50 : 0
	 *        }
	 *       });
	 *      }
	 *     });
	 *
	 * @param {Function} callback
	 */
	addTweenPersistantCallback: function (callback) {
		callback(this);
		this._persistantCallback = callback;
	},

	/**
	 * add a tween that overrides all other tweens
	 *
	 * @param {MAL.tween.TweenObject} tween
	 */
	addTweenOverride: function (options) {		
		this.clearTweens();
		this.addTween(options);
		this._startNextAnimation();
	},

	/**
	 * add a tween set overrides all other tweens
	 *
	 * @param {MAL.tween.TweenObject...} tweens
	 */
	addTweenSetOverride: function (tweens) {
		// convert arguments into an array
		tweens = MAL.argumentsToArray(arguments);

		this._clearTransformations();		
		this.clearTweens();
		this.addTweenSet(tweens);
		this._startNextAnimation();
	},

	/**
	 * add a persistant callback that overrides all other tweens
	 *
	 * @param {Function} callback
	 */
	addTweenPersistantCallbackOverride: function (callback) {
		this.clearTweens();
		this.addTweenPersistantCallback(callback);
	},

	/**
	 * remove all remaining animations from the layer
	 *
	 *     layer.clearTweens();
	 *
	 */
	clearTweens: function () {
		this._started = false;
		this._animating = false;
		this._signature = null;
		this._tweens = [];
		this._cachedProperties = {};
	},

	/**
	 * remove all animations from the layer, and returns current layer
	 *
	 * @return {MAL.tween.CSSLayer} layer
	 */
	clearAll: function () {
		this.clearTweens();
		this._persistantCallback = null;
		this._clearTransformations();
		this._setCSSTextFromBuffer();

		// This fixes issues when adding another tween onto a layer thats been cleared of transformations.
		this._transformations = {};

		return this;
	},

	/**
	 * set layer transform origin
	 *
	 * @private
	 * @param {Object} origin X and Y origin
	 */
	_setLayerTransformOrigin: function (origin) {
		if (origin.x) origin.x = origin.x + (typeof origin.x === 'number' ? 'px' : '');
		if (origin.y) origin.y = origin.y + (typeof origin.y === 'number' ? 'px' : '');

		origin = MAL.mergeOptions(this._transformOrigin, origin);

		// set it to the CSS
		this._css('transform-origin', origin.x + ' ' + origin.y + (MAL.Environment.has3d ? ' ' + origin.z + 'px' : ''));

		// set it in the layer, so that we can do manual transforms
		this._transformOrigin = MAL.mergeOptions(this._transformOrigin, origin);
	},

	/**
	 * get layer transform origin
	 *
	 * @private
	 * @return {Object} origin
	 */
	_getLayerTransformOrigin: function () {
		return this._transformOrigin;
	},

	/**
	 * transform positioning
	 *
	 * @private
	 * @param {Number} value
	 */
	_setLayerX: function (value) {
		if (MAL.isString(value)) return this._transformations.translateX = value;

		var offset = this.useAnchorPointOffset ? this.positionAnchorPointOffset.x : 0;

		this._transformations.translateX = (value - this._initialProperties.left + offset) + this._suffixes.x;
	},
	_setLayerY: function (value) {
		if (MAL.isString(value)) return this._transformations.translateY = value;

		var offset = this.useAnchorPointOffset ? this.positionAnchorPointOffset.y : 0;

		this._transformations.translateY = (value - this._initialProperties.top + offset) + this._suffixes.y;
	},

	_getLayerX: function () {
		var offset = this.useAnchorPointOffset ? this.positionAnchorPointOffset.x : 0;
		return this._parsePropertyValue(this._transformations.translateX) + this._initialProperties.left - offset;
	},
	_getLayerY: function () {
		var offset = this.useAnchorPointOffset ? this.positionAnchorPointOffset.y : 0;
		return this._parsePropertyValue(this._transformations.translateY) + this._initialProperties.top - offset;
	},

	/**
	 * transform scaling
	 *
	 * @private
	 * @param {Number} value
	 */
	_setLayerScale: function (value) {
		// check for native support
		if (this._bugs.scale) {
			// cache value
			this._transformations._scale = value;

			// apply scales
			this._setLayerScaleX(value, true);
			this._setLayerScaleY(value, true);
		} else {
			this._transformations.scale = value;
		}
	},
	_setLayerScaleX: function (value) {
		// check for native support
		if (this._bugs.scale) {
			// not that awesome but its a middleground
			var width = this._initialProperties.width * value,
				left = this._initialProperties.left - (MAL.delta(this._initialProperties.width, width) * this._transformOrigin.x);

			// apply changed properties to cache
			this.setProperties({
				left: left,
				width: width
			}, true);

			// cache value
			this._transformations._scaleX = value;
		} else {
			this._transformations.scaleX = value;
		}
	},
	_setLayerScaleY: function (value) {
		// check for native support
		if (this._bugs.scale) {
			// not that awesome but its a middleground
			var height = this._initialProperties.height * value,
				top = this._initialProperties.top - (MAL.delta(this._initialProperties.height, height) * this._transformOrigin.y);

			// apply changed properties to cache
			this.setProperties({
				top: top,
				height: height
			}, true);

			// cache value
			this._transformations._scaleY = value;
		} else {
			this._transformations.scaleY = value;
		}
	},

	_getLayerScale: function () {
		// check for native support
		var value = this._bugs.scale ? this._transformations._scale : this._transformations.scale;

		return MAL.defaultValue(1, value);
	},
	_getLayerScaleX: function () {
		// check for native support
		var value = this._bugs.scale ? this._transformations._scaleX : this._transformations.scaleX;

		return MAL.defaultValue(1, value);
	},
	_getLayerScaleY: function () {
		// check for native support
		var value = this._bugs.scale ? this._transformations._scaleY : this._transformations.scaleY;

		return MAL.defaultValue(1, value);
	},

	_setLayerTransformPerspective: function (value) {
		this._transformations.perspective = value;
	},

	_getLayerTransformPerspective: function (value) {
		return this._transformations.perspective;
	},

	/**
	 * starts the next animation
	 *
	 * @private
	 */
	_startNextAnimation: function () {
		var tween = this._tweens.shift();
		if (tween) {
			for (var property in tween.finish) {
				var propertyInfo = this._transformationPropertyDictionary[property];

				if (this._transformationPropertyDictionary[property] && MAL.isUndefined(this._transformations[propertyInfo.name])) {
					this._transformations[propertyInfo.name] = propertyInfo.value;
				}
			}

			this._css('transform', this._constructTransformationString());
			this._setCSSTextFromBuffer();

			// we have to do this incase someone runs an addTween right after an addLayer with start transforms
			this._stage.timeouts.addTimeout((function (self, tween) {
				return function () {
					self._signature = new Date().getTime() * Math.random();
					self._applyAnimation.apply(self, [self._signature, tween]);
				};
			}(this, tween)), 0);

			this._started = true;
		} else {
			this._started = false;
		}
	},

	/**
	 * apply an animation to the layer
	 *
	 * @private
	 * @param {Object} options
	 */
	_applyAnimation: function (signature, options) {
		// if signature missmatch, animation has been overridden
		if (signature !== this._signature) return;

		// set start properties
		if (options.delay) {
			// complex timeout closure...
			this._stage.timeouts.addTimeout((function (self, signature, options) {
				return function () {
					// remove delay
					options.delay = 0;

					// apply animation with start values
					self._applyAnimation.apply(self, [signature, options]);
				};
			}(this, signature, options)), options.delay * 1000);

			// break chain
			return;
		} else if (options.start) {
			// no delay
			this._setStartProperties(options.start);

			this._stage.timeouts.addTimeout((function (self, signature, options) {
				return function () {
					options.start = false;
					self._applyAnimation.apply(self, [signature, options]);
				};

				// TODO: look into the delay bug for FF (http://jsfiddle.net/uaukc/2/)
			}(this, signature, options)), 25);

			return;
		}

		// reestablish options object (prevents undefined errors)
		if (!options) options = {finish:false};

		// set transitionCallback if passed
		if (options.callback) this._transitionCallback = options.callback;

		// set finish properties
		var finishProperties = options.finish || {};

		// set layers animation state
		this._animating = true;

		// increment iteration count
		this.iteration++;

		// clear
		this._clearTransformations();

		// record actual property changes
		var changes = 0;

		for (var name in finishProperties) {
			var property = finishProperties[name],
				value;

			// check for advanced options
			if (MAL.isObject(property) && property.value) {
				value = property.value;
			} else {
				value = property;
			}

			// break out of set command, nothing has changed
			if (this._cachedProperties[name] === value) continue;

			// set property

			this.setProperty(name, value);

			// record changes (we do this to allow us to actually account for an empty animation)
			changes++;
		}

		// reset
		this._setTransformations(finishProperties, options, true);

		// fire animation start
		this.dispatchEvent('animation-start');

		// throw error if a tween was ran that actually does nothing...
		//if (options.finish && !changes) MAL.error('BadTween: being applyed to "' + this.name + '"');

		// apply changes
		this._setCSSTextFromBuffer();

		// instantly fire animation end (this should prolly be fixed)
		if (MAL.isUndefined(options.finish)) {
			//this.dispatchEvent('animation-end');

			// if we are only doing a "start" and there are no more tweens left
			if (!this._tweens.length) this._started = false;
		}
	},


	/**
	 * clears css transitions and transform
	 *
	 * @private
	 */
	_clearTransformations: function () {
		// clear
		this._css({
			'transform': null,
			'transition': 'none',
			'transition-property': null,
			'transition-delay': null,
			'transition-duration': null,
			'transition-timing-function': null
		});

		// clear webkit specific properties
		if (MAL.Environment.has3d && !this._bugs.transform3d) {
			if (this._features.backfacePerformanceFix && !this._backface) this._css('backface-visibility', null);
			if (this._features.perspectivePerformanceFix) this._css('perspective', null);
		}
	},

	/**
	 * sets css transitions and transform
	 *
	 * @private
	 * @param {Object} properties
	 * @param {Object} options
	 */
	_setTransformations: function (properties, options, transition) {
		if (transition) {
			// construct transition string
			transition = this._constructTransition(properties, options);

			// reset
			this._css({
				'transform': this._constructTransformationString(),
				'transition-property': transition.property,
				'transition-delay': '0s',
				'transition-duration': transition.duration,
				'transition-timing-function': transition.easing
			});

			// reset webkit specific properties
			if (MAL.Environment.has3d && !this._bugs.transform3d) {
				if (this._features.backfacePerformanceFix && !this._backface) this._css('backface-visibility', 'hidden'); // fixes flickers for elements that do not utilize backfaces
				if (this._features.perspectivePerformanceFix) this._css('perspective', 1000); // fixes aliasing performance issues
			}

		} else {
			// reset
			this._css({
				'transform': this._constructTransformationString()
			});
		}
	},

	/**
	 * removes transformations (this is used to prevent flickering in some cases)
	 */
	disableTransformations: function () {
		if (this._animating) MAL.error('BadCommand: layer is currently animating "' + this.name + '"');
		this._transformationsDisabled = true;
		this._css({'transform': this._constructTransformationString()});
		this._clearTransformations();
		this._setCSSTextFromBuffer();
		return this;
	},

	/**
	 * restore transformations
	 */
	enableTransformations: function () {
		this._transformationsDisabled = false;
		this._css({'transform': this._constructTransformationString()});
		this._setCSSTextFromBuffer();
		return this;
	},

	/**
	 * sets start properties
	 *
	 * @private
	 * @param {Object} properties
	 */
	_setStartProperties: function (properties) {
		this._clearTransformations();

		for (var name in properties) {
			this.setProperty(name, properties[name]);
		}

		this._setTransformations(properties, {});
	},

	/**
	 * set multiple properties on a layer
	 *
	 *     layer.setProperties({
	 *      opacity: .5,
	 *      x: 100,
	 *      y: 100,
	 *      'background-color': '#000000'
	 *     });
	 *
	 * refer to {@link MAL.tween.TransformObject} for possible properties
	 *
	 * @param {Object} properties
	 */
	setProperties: function (properties, dontApplyToDOM) {
		for (var property in properties)
			this.setProperty(property, properties[property], true);

		// regenerate the transform string
		if (!dontApplyToDOM) {
			this._css('transform', this._constructTransformationString());
			this._setCSSTextFromBuffer();
		}

		return this;
	},

	/**
	 * set a layers property
	 *
	 *     layer.setProperty('rotate', 45);
	 *
	 * refer to {@link MAL.tween.TransformObject} for possible properties
	 *
	 * @param {String} name
	 * @param {String/Number} value
	 */
	setProperty: function (name, value, dontApplyToDOM) {
		if (!this._initializedFixes) this._initializeFixes();

		// method is used to detect if we need any additional setProperty overrides
		var method = MAL.toCamelCaseWithPrefix('_setLayer', name);

		// cache new property value
		this._cachedProperties[name] = value;

		if (this._transformationPropertyDictionary[name]) {
			// set advanced property
			if (this[method]) {
				this[method](value);
			} else {
				var transform = this._transformationPropertyDictionary[name];
				this._transformations[transform.name] = MAL.defaultValue(transform.value, value) + (this._suffixes[name] || '');
			}
		} else {
			// set basic styling property
			this._css(name, value);
		}

		// regenerate the transform string
		if (!dontApplyToDOM) {
			this._css('transform', this._constructTransformationString());
			this._setCSSTextFromBuffer();
		}

		return this;
	},

	/**
	 * get a layers property value
	 *
	 *     layer.getProperty('rotate');
	 *
	 * @param {String} name
	 */
	getProperty: function (name) {
		var method = MAL.toCamelCaseWithPrefix('_getLayer', name),
			value;

		if (this._transformationPropertyDictionary[name]) {
			// get transform property
			value = this[method] ? this[method]() : this._transformations[this._transformationPropertyDictionary[name].name];
		} else {
			// get basic styling property
			value = this._css(name);
		}

		return this._parsePropertyValue(value);
	},

	/**
	 * get layers current position after transformations and positioning
	 *
	 *     layer.getPosition();
	 *
	 * @return {Object} {x: Float, y: Float}
	 */
	getPosition: function () {
		return {
			x: this.cssGetComputedProperty('left') + parseFloat(this._getLayerX()),
			y: this.cssGetComputedProperty('top') + parseFloat(this._getLayerY())
		};
	},

	/**
	 * produces a transformation property string from the _transformations object
	 *
	 * @private
	 */
	_constructTransformationString: function () {
		if (this._transformationsDisabled) return null;

		var string = '';

		this._transformationFunctionOrder.forEach(function (transformation) {
			if (transformation.substr(0,1) === '_' || MAL.isUndefined(this._transformations[transformation])) return;

			// look up alias name
			var name = this._transformationFunctionAlias[transformation] ? this._transformationFunctionAlias[transformation] : transformation;

			string += ' ' + name + '(' + this._transformations[transformation] + ')';
		}, this);

		return string.substr(1);
	},

	/**
	 * produces a transformation property object from a transform string
	 *
	 * @private
	 */
	_constructTransformationObject: function (string) {
		var match,
			transformation = {},
			pattern = new RegExp(/([a-z0-9]+)\((.+?)\)/gi);

		while ((match = pattern.exec(string))) transformation[match[1]] = match[2];

		return transformation;
	},

	/**
	 * produces a transition object
	 *
	 * @private
	 * @param {Object} properties
	 * @param {Object} options
	 */
	_constructTransition: function (properties, options) {
		// option defaults
		var defaultDuration = options.duration || 0,
			defaultDelay = options.delay || 0,
			defaultTiming = options.easing || [0,0,1,1],
			transformSettings = null;

		// transition strings
		var transition = {
			property: '',
			delay: '',
			duration: '',
			easing: ''
		};

		for (var property in properties) {
			// local vars
			var duration,
				delay,
				easing,
				callback,
				value,
				info = properties[property],
				propertyName = MAL.Environment.getCSSPropertyName(property, true);

			if (MAL.isObject(info)) {
				// property has a custom transition
				duration = info.duration || defaultDuration;
				delay = info.delay || defaultDelay;
				easing = info.easing || defaultTiming;
				callback = info.callback || false;
			} else {
				// property has a default transition
				duration = defaultDuration;
				delay = defaultDelay;
				easing = defaultTiming;
				callback = false;
			}

			// externalize property ref
			if (this._transformationPropertyDictionary[property]) {
				// keep ref for abnormal transform operations
				transformSettings = {
					delay: delay,
					duration: duration,
					easing: easing,
					callback: callback
				};

				// set for normal transform operations
				if (!(/^scale/.test(property) && this._bugs.scale))
					this._animatedProperties[MAL.Environment.getCSSPropertyName('transform', true)] = transformSettings;

			} else if (!options.start || MAL.isUndefined(options.start[property])) {
				// add to elements transition list if they are not start options
				this._animatedProperties[propertyName] = {
					delay: delay,
					duration: duration,
					easing: easing,
					callback: callback
				};
			} else if (options.start && options.start[property]) {
				// remove element from _animatedProperties list
				delete this._animatedProperties[propertyName];
			}

			if (property === 'transformOrigin') {
				this.setProperty('transformOrigin', info, false);
			}

		}

		// determine if we overrode scale, and if we did we should be using w,h,x,y to simulate scale
		if (!MAL.isUndefined(this._transformations._scaleX)) this._animatedProperties.left = this._animatedProperties.width = transformSettings;
		if (!MAL.isUndefined(this._transformations._scaleY)) this._animatedProperties.top = this._animatedProperties.height = transformSettings;


		// compute elements transition list
		for (var name in this._animatedProperties) {
			transition.property += name + ',';
			transition.duration += this._animatedProperties[name].duration + 's,';
			transition.delay += this._animatedProperties[name].delay + 's,';
			transition.easing += (MAL.isArray(this._animatedProperties[name].easing) ? 'cubic-bezier(' + this._animatedProperties[name].easing.join(',') + ')' : this._animatedProperties[name].easing) + ',';
		}

		return {
			property: transition.property.slice(0, -1),
			delay: transition.delay.slice(0, -1),
			duration: transition.duration.slice(0, -1),
			easing: transition.easing.slice(0, -1)
		};
	},

	/**
	 * fired everytime a property transition ends
	 *
	 * @private
	 * @param {Event} e
	 */
	_onTransitionEnd: function (e) {
		// prevent events from bubbling
		e.stopPropagation();

		// we dont care about start animations
		if (!this._animating) return false;

		// fire property callback
		if (this._animatedProperties[e.propertyName] && this._animatedProperties[e.propertyName].callback) this._animatedProperties[e.propertyName].callback(this);

		// record that we are no longer animating on the property
		delete this._animatedProperties[e.propertyName];

		// cycle through uncompleted transitions
		var animations = 0;
		for (var name in this._animatedProperties) if (this._animatedProperties[name]) animations++;

		// fire transition completion event
		if (!animations) this._onTransitionCompleted();
	},

	/**
	 * fired when the first transition ends
	 *
	 * @private
	 */
	_onTransitionCompleted:  function () {
		if (!this._animating) return;

		// clear
		this._css({
			'transition-delay': null,
			'transition-duration': null,
			'transition-timing-function': null
		});

		// apply changes
		this._setCSSTextFromBuffer();

		// set layers animation state
		this._animating = false;

		// fire transition callback
		if (this._transitionCallback) {
			this._transitionCallback(this);
			this._transitionCallback = false;
		}

		// fire persistant callback
		if (this._persistantCallback) this._persistantCallback(this);

		// dispatch transition completion event
		this.dispatchEvent('animation-end');

		// start next animation
		this._startNextAnimation();
	},



	/**
	 * sets and gets css property values
	 *
	 * @private
	 * @param {String} property name of the property
	 * @param {String/Number} value value of property
	 * @param {Object} properties set of properties
	 */
	_css: function () {
		if (!MAL.isUndefined(arguments[0]) && !MAL.isUndefined(arguments[1])) {
			// set specific value
			this._cssSetProperty(arguments[0], arguments[1]);
		} else if (MAL.isString(arguments[0])) {
			// return specific value
			return this._cssGetProperty(arguments[0]);
		} else if (MAL.isObject(arguments[0])) {
			// set multiple properties
			this._cssSetProperties(arguments[0]);
		}
	},

	/**
	 * sets the cssText of the layer (this is very IMPORTANT)
	 *
	 * @private
	 */
	_setCSSTextFromBuffer: function () {
		var string = '';

		for (var property in this._bufferedCSSProperties) {
			// do not add things that are strictly used for computing (filters)
			if (property.substr(0,1) !== '_') {
				string += property + ': ' + this._bufferedCSSProperties[property] + '; ';
			}
		}

		this.element.style.cssText = string;
	},

	/**
	 * sets a single css property
	 *
	 * @private
	 * @param {String} property name of the property
	 * @param {String/Number} value value of property
	 */
	_cssSetProperty: function (property, value) {
		// method is used to detect if we need any additional css overrides
		var method = MAL.toCamelCaseWithPrefix('_cssSet', property);

		// add browser prefix to properties that start with a dash
		var propertyName = MAL.Environment.getCSSPropertyName(property, true);

		if (this[method]) {
			// use override
			var override = this[method](value);

			// do we need to save it
			if (override) this._bufferedCSSProperties[override.name] = override.value;
		} else if (MAL.isNull(value)) {
			delete this._bufferedCSSProperties[propertyName];
		} else {
			// add suffix
			if (MAL.isNumber(value) && this._suffixes[property]) value = value + this._suffixes[property];

			this._bufferedCSSProperties[propertyName] = value;
		}
	},

	/**
	 * sets a the _backface boolean if set to visible, otherwise the layer may use it for performance gains
	 *
	 * @private
	 * @param {Object} properties set of properties
	 */
	_cssSetBackfaceVisibility: function (value) {
		if (value === 'visible') this._backface = true;

		return {
			name: MAL.Environment.prefix + 'backface-visibility',
			value: value
		};
	},

	/**
	 * sets a set of css properties
	 *
	 * @private
	 * @param {Object} properties set of properties
	 */
	_cssSetProperties: function (properties) {
		for (var property in properties) {
			this._cssSetProperty(property, properties[property]);
		}
	},

	/**
	 * gets the value of a css property
	 *
	 * @private
	 * @param {String} property name of css property
	 */
	_cssGetProperty: function (property, raw) {
		// method is used to detect if we need any additional css overrides
		var method = MAL.toCamelCaseWithPrefix('_cssGet', property);

		// add browser prefix to properties that start with a dash
		var propertyName = MAL.Environment.getCSSPropertyName(property, true);

		if (this[method]) {
			// use override
			return this[method](property);
		} else {
			// get basic style property
			return this.element.style[propertyName] || this.cssGetComputedProperty(propertyName, raw);
		}
	},

	/**
	 * attempts to retrieve the current transformations from the elements transformation matrix
	 *
	 * @private
	 */
	_getComputedTransformations: function () {
		var x = 0,
			y = 0,
			z = 0,
			matrix = new window.WebKitCSSMatrix(window.getComputedStyle(this.element).webkitTransform);

		if ('m33' in matrix) {
			x = matrix.e;
			y = matrix.f;
		} else {
			x = matrix.m41;
			y = matrix.m42;
			z = matrix.m43;
		}

		return {x: x, y: y};
	}
}));
MAL.tween.CSSLayerProxy = new MAL.tween.LayerProxy(MAL.tween.CSSLayer);

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * **WARNING: this has yet to be finalized**
 *
 * @class MAL.tween.CSSJSLayer
 * @extends MAL.tween.CSSLayer
 */
MAL.define('tween.CSSJSLayer', MAL.tween.CSSLayer.extend({
	// standard easing dictionary list
	_eases: {
		ease: [0.25, 0.1, 0.25, 1],
		linear: [0, 0, 1, 1],
		'ease-in': [0.42, 0, 1, 1],
		'ease-out': [0, 0, 0.58, 1],
		'ease-in-out': [0.42, 0, 0.58, 1]
	},

	/**
	 * updates the layers cssText based on the current tween thats being applied
	 *
	 * @private
	 * @param {Number} time
	 */
	_update: function (time) {
		if (!this._currentTween) return;

		var tween = this._currentTween,
			finished = time > (tween.startTime + tween.duration),
			percent = (time - tween.startTime) / tween.duration;

		// in case we are late
		if (percent > 1) percent = 1;

		// apply ease if provided
		if (MAL.isArray(tween.easing)) {
			percent = MAL.cubicBezierWithPercent(percent, tween.easing[0], tween.easing[1], tween.easing[2], tween.easing[3]);
		} else {
			percent = tween.easing(percent);
		}

		// apply changes to properties
		for (var property in tween.values.start) {
			var value = MAL.Util.shortenFloat(tween.values.start[property] + (tween.values.delta[property] * percent), 5);
			this.setProperty(property, value, true);
		}

		// apply everything to the DOM
		this._css('transform', this._constructTransformationString());
		this._setCSSTextFromBuffer();

		// check if animation is complete
		if (finished) {
			this._currentTween = null;
			//this._clearTemporaryCachedProperties();

			// fire persistant callback
			if (this._persistantCallback) this._persistantCallback(this);

			// fire callback
			if (tween.callback) tween.callback(this);

			this.dispatchEvent('animation-end');
			this._startNextAnimation();
		}
	},

	/**
	 * constructs and adds an animation to the tween stack, also handles the delay
	 *
	 * @private
	 * @param {Object} options
	 */
	_applyAnimation: function (signature, options) {
		// if signature missmatch, animation has been overridden
		if (signature !== this._signature) return;

		// we don't really need to do it this way but to keep the core logic consistant...
		if (options.delay) {
			this._stage.timeouts.addTimeout((function (self, signature, options) {
				return function () {
					// remove delay
					options.delay = 0;

					// apply animation with start values
					self._applyAnimation.apply(self, [signature, options]);
				};
			}(this, signature, options)), options.delay * 1000);

			return;
		}

		// checks if we have transformation support
		this._transformationSupport = !!MAL.Environment.getCSSPropertyName('transform') && !this._bugs.transformations;

		// force hardware acceleration (webkit is the only browser that supports a Z axis)
		if (MAL.Environment.has3d && !this._bugs.transform3d && this._features.backfacePerformanceFix) this._transformations.translateZ = '0px';

		// set start with props
		if (options.start) this._setStartProperties(options.start);

		// generate deltas
		var tween = {};
		tween.duration = options.duration * 1000 || 1000;
		tween.startTime = new Date().getTime();
		if (options.easing) {
			tween.easing = MAL.isString(options.easing) && this._eases[options.easing] ? this._eases[options.easing] : options.easing;
		} else {
			tween.easing = this._eases.linear;
		}

		tween.values = {
			start: {},
			delta: {},
			finish: options.finish
		};
		tween.callback = options.callback;

		for (var property in options.finish) {
			tween.values.start[property] = MAL.parseNumber(this.getProperty(property));
			tween.values.delta[property] = MAL.delta(tween.values.start[property], options.finish[property]);
		}

		this._currentTween = tween;
		this._animating = true;

		// increment iteration count
		this.iteration++;

		this.dispatchEvent('animation-start');
	}
}));
MAL.tween.CSSJSLayerProxy = new MAL.tween.LayerProxy(MAL.tween.CSSJSLayer);

/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * @class MAL.tween.JSLayer
 * @extends MAL.tween.Layer
 *
 * **You should not have to instantiate this class directly, it is managed by {@link MAL.tween.Stage}.**
 *
 * # Layers
 *
 * we have a standard of aliasing the layer getter with $
 *     var $ = stage.getLayers;
 *
 * obtaining a layer (this actually creates a Layer object if one does not already exist)
 *     $('box');
 *
 * obtaining multiple layers (this will create a layer proxy object)
 *     $('box', 'box2');
 *
 * setting a property on a layer
 *     $('box').setProperty('opacity', 0);
 *
 * setting a property on multiple layers
 *     $('box', 'box2').setProperty('opacity', 0);
 *
 * if for some reason you cant to find out if a layer has went through a proxy you can do this
 *     document.getElementById('mal-ad-box').MALTweenLayer; // this property would reference the layer instance
 *
 * # Helpers
 *
 * you can also string things
 *
 *     $('box', 'box2').clearAll().addTween({duration: 1, finish: {x: 100}});
 *
 * # Animating on a layer
 *
 * we only want to do one animation with this layer so we use {@link #addTween}
 *     layer.addTween({duration: 1, finish: {x: 100}}); // animate layer to {x:100} in a second
 *
 * but if we want to do a set of animations in a specific order we would use {@link #addTweenSet}
 *
 *     layer.addTweenSet(
 *      {duration: 1, finish: {x: 100}},
 *      {duration: 1, finish: {x: 0}},
 *      {duration: 1, finish: {y: 100}},
 *      {duration: 1, finish: {y: 0}}
 *     );
 *
 * here is a runnable example of adding a tween set
 *
 *     @example
 *     // add a layer (there is an element already in the dom with an ID of mal-ad-box)
 *     // we are also setting some initial css properties
 *     var layer = stage.addLayer('box', {width: 25, height: 25, 'background-color': '#085083'});
 *
 *     // you can also add multiple tweens like this
 *     layer.addTweenSet(
 *
 *         // wait 500 ms then animate to {x:500} with an ease of 'ease-in-out'
 *         {delay: .5, duration: .5, finish: {x: 500}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {x: 0, y: 100}
 *         {duration: .5, finish: {x: 0, y: 100}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {rotate: 45}
 *         {duration: .5, finish: {rotate: 45}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {x: 500, y: 0}
 *         {duration: .5, finish: {x: 500, y: 0}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {x: 0} then disappear
 *         {duration: .5, finish: {x: 0, y: 0}, easing: 'ease-in-out', callback: function (layer) {
 *          layer.setProperty('visibility', 'hidden');
 *         }}
 *
 *     );
 *
 * # Dealing with bugs and features
 *
 * all layers will inherit the bugs/features settings that are set through the {@link MAL.Environment}, but you can override them on a per layer basis with {@link #setFeatures} and {@link #setBugs}.
 *
 * because a lot of our animations depend on the bleeding edge
 */

/**
 * dispatched when an animation has been added
 *
 * @event animation-add
 * @param {Event} this
 */

/**
 * dispatched when an animation has started
 *
 * @event animation-start
 * @param {Event} this
 */

/**
 * dispatched when an animation has ended
 *
 * @event animation-end
 * @param {Event} this
 */
MAL.define('tween.JSLayer', MAL.tween.Layer.extend({
	/**
	 * initializes MALTweenJSLayer
	 *
	 * @method constructor
	 * @param {Object} config
	 */

	/**
	 * @cfg {MAL.tween.Stage} stage that the layer will be added to
	 */

	/**
	 * @cfg {HTMLElement} element direct element reference
	 */

	/**
	 * @cfg {Object} begin properties that the layer should begin with
	 */
	initialize: function ($config) {
		this._super($config);

		// type of layer
		this.layerType = 'JS';

		// establish element association
		this.element = $config.element;
		this.element.MALTweenLayer = this;

		// layer name / ref
		this.name = this.element.id ? this.element.id : this.element;

		// features and bugs
		this._features = {};
		this._bugs = {};
		this._initializeBugsAndFeatures();

		// checks if we have transformation support
		this._transformationSupport = !!MAL.Environment.getCSSPropertyName('transform') && !this._bugs.transformations;

		// sets a reference to the parent stage
		this._stage = $config.stage;

		// animation state
		this._animating = false;

		// signature for override verification
		this._signature = null;

		// animation iteration count
		this.iteration = 0;

		// persistant callback
		this._persistantCallback = null;

		// holds all tweens
		this._tweens = [];

		// started
		this._started = false;

		// current tween
		this._currentTween = null;

		// css property buffer
		this._bufferedCSSProperties = {};

		// current applied transformations
		this._transformations = {};

		// current applied filters
		this._filters = {};
		this._filtersCache = {};

		// properties that have been animated on
		this._animatedProperties = {};

		/** @private empty transition callback */
		this._transitionCallback = function () {};

		// cached properties
		this._initialProperties = {
			left: this.cssGetComputedProperty('left'),
			top: this.cssGetComputedProperty('top'),
			width: this.cssGetComputedProperty('width'),
			height: this.cssGetComputedProperty('height')
		};

		// startWithTimeout
		this._startWithTimeout = null;

		// default anchor point (used for manual position calculations)
		this._transformOrigin = {};
		this._setLayerTransformOrigin({x: '50%', y: '50%'});

		this._transparency = true;

		// adds a small amount of rotation to all elements to smooth animations

		//if (this._features.rotationPerformanceFix) this._transformations.rotate = '.01deg';

		// set begin properties
		if ($config.begin) this.setProperties($config.begin);

	},

	// ensures that the filters happen in the specified order
	_filterOrder: [
		'DXImageTransform.Microsoft.Matrix',
		'DXImageTransform.Microsoft.Alpha'
	],

	// ensures that the function operations happen in this order
	_transformationFunctionOrder: [
		'preRotate',

		'translateX',
		'translateY',

		'skew',
		'skewX',
		'skewY',

		'scale',
		'scaleX',
		'scaleY',

		'rotate',

		'postX',
		'postY'
	],

	// custom transformation function aliases
	_transformationFunctionAlias: {
		preRotate: 'rotate',

		postX: 'translateX',
		postY: 'translateY'
	},

	// property suffix dictionary list
	_suffixes: {
		perspective: 'px',
		width: 'px',
		height: 'px',
		left: 'px',
		top: 'px',
		skew: 'deg',
		preRotate: 'deg',
		rotate: 'deg',
		x: 'px',
		y: 'px',
		postX: 'px',
		postY: 'px'
	},

	// property dictionary (used for ensuring default values)
	_transformationPropertyDictionary: {
		'x': {name: 'translateX', value: 0},
		'y': {name: 'translateY', value: 0},
		'transformOrigin': {name: 'transformOrigin', value: {x: '50%', y: '50%', z: 0}},
		'transformPerspective': {name: 'transformPerspective', value: 0},
		'skew': {name: 'skew', value: 0},
		'skewX': {name: 'skewX', value: 0},
		'skewY': {name: 'skewY', value: 0},
		'scale': {name: 'scale', value: 1},
		'scaleX': {name: 'scaleX', value: 1},
		'scaleY': {name: 'scaleY', value: 1},
		'preRotate': {name: 'preRotate', value: 0},
		'rotate': {name: 'rotate', value: 0},
		'postX': {name: 'postX', value: 0},
		'postY': {name: 'postY', value: 0}
	},

	// standard easing dictionary list
	_eases: {
		ease: [0.25, 0.1, 0.25, 1],
		linear: [0, 0, 1, 1],
		'ease-in': [0.42, 0, 1, 1],
		'ease-out': [0, 0, 0.58, 1],
		'ease-in-out': [0.42, 0, 0.58, 1]
	},

	/**
	 * copies the enviorments default bugs and features
	 *
	 * @private
	 */
	_initializeBugsAndFeatures: function () {
		for (var feature in MAL.Environment.features) this._features[feature] = MAL.Environment.features[feature];
		for (var bug in MAL.Environment.bugs) this._bugs[bug] = MAL.Environment.bugs[bug];
	},

	/**
	 * enables or disables bugs for the layer
	 *
	 *     layer.setBugs({
	 *      transform3d: true,
	 *      scale: true
	 *     });
	 *
	 * @param {Object} bugs
	 */
	setBugs: function (bugs) {
		for (var bug in bugs) this._bugs[bug] = bugs[bug];
		return this;
	},

	/**
	 * enables or disables features for the layer
	 *
	 *     layer.setFeatures({
	 *      backfacePerformanceFix: false,
	 *      translateZPerformanceFix: false
	 *     });
	 *
	 * @param {Object} features
	 */
	setFeatures: function (features) {
		for (var feature in features) this._features[feature] = features[feature];
		return this;
	},

	/**
	 * add a tween to a layer
	 *
	 *     layer.addTween({duration: 1, finish: {x: 100}});
	 *
	 * @param {MAL.tween.TweenObject} tween
	 */
	addTween: function (options) {
		this._tweens.push(options);

		if (this._stage.started && !this._started) {
			this._startNextAnimation();
		}

		this.dispatchEvent('animation-add');
	},

	/**
	 * add tweens to a layer
	 *
	 *     layer.addTweenSet(
	 *      {duration: 1, finish: {x: 100}},
	 *      {duration: 1, finish: {x: 0}},
	 *      {duration: 1, finish: {y: 100}},
	 *      {duration: 1, finish: {y: 0}}
	 *     );
	 *
	 * @param {MAL.tween.TweenObject...} tweens
	 */
	addTweenSet: function (tweens) {
		// convert arguments into an array
		tweens = MAL.argumentsToArray(arguments);

		// add tweens
		tweens.forEach(function(tween) {
			this.addTween(tween);
		}, this);
	},

	/**
	 * add a persistant callback, usually used to dynamic animations like this
	 *
	 *     layer.addTweenPersistantCallback(function (layer) {
	 *      // this get fired after every tween
	 *      if (layer.iteration <= 5) {
	 *
	 *       // add a tween with an alternating finish animation based on the next iteration
	 *       layer.addTween({
	 *        duration: .2,
	 *        finish: {
	 *         x: this.iteration % 2 ? 50 : 0
	 *        }
	 *       });
	 *
	 *      }
	 *     });
	 *
	 * @param {Function} callback
	 */
	addTweenPersistantCallback: function (callback) {
		callback(this);
		this._persistantCallback = callback;
	},

	/**
	 * add a tween that overrides all other tweens
	 *
	 * @param {MAL.tween.TweenObject} tween
	 */
	addTweenOverride: function (options) {
		this.clearTweens();
		this.addTween(options);
		this._startNextAnimation();
	},

	/**
	 * add a tween set overrides all other tweens
	 *
	 * @param {MAL.tween.TweenObject...} tweens
	 */
	addTweenSetOverride: function (tweens) {
		// convert arguments into an array
		tweens = MAL.argumentsToArray(arguments);

		this.clearTweens();
		this.addTweenSet(tweens);
		this._startNextAnimation();
	},

	/**
	 * add a persistant callback that overrides all other tweens
	 *
	 * @param {Function} callback
	 */
	addTweenPersistantCallbackOverride: function (callback) {
		this.clearTweens();
		this.addTweenPersistantCallback(callback);
	},

	/**
	 * remove all remaining animations from the layer
	 *
	 *     layer.clearTweens();
	 *
	 */
	clearTweens: function () {
		this._clearTemporaryCachedProperties();
		this._started = true;
		this._animating = false;
		this._signature = null;
		this._tweens = [];
		this._currentTween = null;
	},

	/**
	 * remove all animations from the layer, and returns current layer
	 *
	 * @return {MAL.tween.JSLayer} layer
	 */
	clearAll: function () {
		this._persistantCallback = null;
		this.clearTweens();
		return this;
	},

	/**
	 * flag layer as transparent (this is mainly used for some of our hacks in IE >9)
	 *
	 * @private
	 * @param {Boolean} transparent
	 */
	_setLayerTransparency: function (transparent) {
		this._transparency = transparent;
	},

	/**
	 * set layer transform origin
	 *
	 * @private
	 * @param {Object} origin X and Y origin
	 */
	_setLayerTransformOrigin: function (origin) {
		if (!this._transformationSupport || this._bugs.scale) {
			// turn pixels into percents
			if (!MAL.isUndefined(origin.x)) origin.x = typeof origin.x === 'number' ? (origin.x / this._initialProperties.width) : parseFloat(origin.x) / 100;
			if (!MAL.isUndefined(origin.y)) origin.y = typeof origin.y === 'number' ? (origin.y / this._initialProperties.height) : parseFloat(origin.y) / 100;
		} else {
			if (!MAL.isUndefined(origin.x)) origin.x = origin.x + (typeof origin.x === 'number' ? 'px' : '');
			if (!MAL.isUndefined(origin.y)) origin.y = origin.y + (typeof origin.y === 'number' ? 'px' : '');

			// set it to the CSS
			this._css('transform-origin', origin.x + ' ' + origin.y);
		}

		// set it in the layer, so that we can do manual transforms
		this._transformOrigin = MAL.mergeOptions(this._transformOrigin, origin);
	},

	/**
	 * get layer transform origin
	 *
	 * @private
	 * @return {Object} origin
	 */
	_getLayerTransformOrigin: function () {
		return this._transformOrigin;
	},

	/**
	 * transform positioning
	 *
	 * @private
	 * @param {Number} value
	 */
	_setLayerX: function (value) {
		if (this._transformationSupport) {
			// browsers that can handle translate

			// TODO: add support for % values
			this._transformations.translateX = (value - this._initialProperties.left) + this._suffixes.x;
		} else {
			// browsers that cant handle translate
			this._bufferedCSSProperties.left = value + this._suffixes.x;
		}
	},
	_setLayerY: function (value) {
		if (this._transformationSupport) {
			// browsers that can handle translate

			// TODO: add support for % values
			this._transformations.translateY = (value - this._initialProperties.top) + this._suffixes.y;
		} else {
			// browsers that cant handle translate
			this._bufferedCSSProperties.top = value + this._suffixes.y;
		}
	},

	_getLayerX: function () {
		if (this._transformationSupport) {
			// browsers that can handle translate
			return this._transformations.translateX ? this._initialProperties.left + MAL.parseNumber(this._transformations.translateX) : this._initialProperties.left;
		} else {
			// browsers that cant handle translate
			return this._bufferedCSSProperties.left ? MAL.parseNumber(this._bufferedCSSProperties.left) : this._initialProperties.left;
		}
	},
	_getLayerY: function () {
		if (this._transformationSupport) {
			// browsers that can handle translate
			return this._transformations.translateY ? this._initialProperties.top + MAL.parseNumber(this._transformations.translateY) : this._initialProperties.top;
		} else {
			// browsers that cant handle translate
			return this._bufferedCSSProperties.top ? MAL.parseNumber(this._bufferedCSSProperties.top) : this._initialProperties.top;
		}
	},

	/**
	 * transform scaling
	 *
	 * @private
	 * @param {Number} value
	 */
	_setLayerScale: function (value) {
		// check if browser doesn't support scale or has a scale bug
		if (!this._transparency && MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			this._setFilter('DXImageTransform.Microsoft.Matrix', 'M11=' + value + ', M21=0, M12=0, M22=' + value + ', SizingMethod="auto expand"', [value,value]);

			// apply everything to the DOM so that our offsets are correct
			this._setCSSTextFromCache();

			// buffer actual inital width/height
			if (!this._bufferedCSSProperties._initialProperties) this._bufferedCSSProperties._initialProperties = {width: this._parsePropertyValue(this.element.style.width), height: this._parsePropertyValue(this.element.style.height)};

			// set left according to scale
			this._bufferedCSSProperties.left = (this._initialProperties.left - (this.element.offsetWidth - this._bufferedCSSProperties._initialProperties.width) * this._transformOrigin.x) + 'px';

			// set top according to scale
			this._bufferedCSSProperties.top = (this._initialProperties.top - (this.element.offsetHeight - this._bufferedCSSProperties._initialProperties.height) * this._transformOrigin.y) + 'px';

		} else if (!this._transformationSupport || this._bugs.scale) {
			this._setLayerScaleX(value, true);
			this._setLayerScaleY(value, true);

			// cache value
			this._transformations._scale = value;
		} else {
			this._transformations.scale = value;
		}
	},
	_setLayerScaleX: function (value, doNotSetTransform) {
		if (!this._transparency && MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			// browser can use filters

			// matrix scaling
			var matrix = this._getFilter('DXImageTransform.Microsoft.Matrix'),
				scaleY = matrix ? matrix[1] : 1;

			this._setFilter('DXImageTransform.Microsoft.Matrix', 'M11=' + value + ', M21=0, M12=0, M22=' + scaleY + ', SizingMethod="auto expand"', [value,scaleY]);

			// apply everything to the DOM so that our offsets are correct
			this._setCSSTextFromCache();

			// buffer actual inital width/height
			if (!this._bufferedCSSProperties._initialProperties) this._bufferedCSSProperties._initialProperties = {width: this._parsePropertyValue(this.element.style.width), height: this._parsePropertyValue(this.element.style.height)};

			// set left according to scale
			this._bufferedCSSProperties.left = (this._initialProperties.left - (this.element.offsetWidth - this._bufferedCSSProperties._initialProperties.width) * this._transformOrigin.x) + 'px';
		} else if (!this._transformationSupport || this._bugs.scale) {
			// browser doesn't support scale or has a scale bug

			// not that awesome but its a middleground
			var width = this._initialProperties.width * value,
				left = this._initialProperties.left - (MAL.delta(this._initialProperties.width, width) * this._transformOrigin.x);

			// apply changed properties to cache
			this._bufferedCSSProperties.left = left + 'px';
			this._bufferedCSSProperties.width = width + 'px';

			// cache value
			this._transformations._scaleX = value;
		} else {
			this._transformations.scaleX = value;
		}
	},
	_setLayerScaleY: function (value, doNotSetTransform) {
		if (!this._transparency && MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			// browser can use filters

			// matrix scaling
			var matrix = this._getFilter('DXImageTransform.Microsoft.Matrix'),
				scaleX = matrix ? matrix[0] : 1;

			this._setFilter('DXImageTransform.Microsoft.Matrix', 'M11=' + scaleX + ', M21=0, M12=0, M22=' + value + ', SizingMethod="auto expand"', [scaleX,value]);

			// apply everything to the DOM so that our offsets are correct
			this._setCSSTextFromCache();

			// buffer actual inital width/height
			if (!this._bufferedCSSProperties._initialProperties) this._bufferedCSSProperties._initialProperties = {width: this._parsePropertyValue(this.element.style.width), height: this._parsePropertyValue(this.element.style.height)};

			// set top according to scale
			this._bufferedCSSProperties.top = (this._initialProperties.top - (this.element.offsetHeight - this._bufferedCSSProperties._initialProperties.height) * this._transformOrigin.y) + 'px';
		} else if (!this._transformationSupport || this._bugs.scale) {
			// browser doesn't support scale or has a scale bug

			// not that awesome but its a middleground
			var height = this._initialProperties.height * value,
				top = this._initialProperties.top - (MAL.delta(this._initialProperties.height, height) * this._transformOrigin.y);

			// apply changed properties to cache
			this._bufferedCSSProperties.top = top + 'px';
			this._bufferedCSSProperties.height = height + 'px';

			// cache value
			this._transformations._scaleY = value;
		} else {
			this._transformations.scaleY = value;
		}
	},

	_getLayerScale: function () {
		if (!this._transparency && MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			var matrix = this._getFilter('DXImageTransform.Microsoft.Matrix');
			return matrix ? matrix[0] : 1;
		} else {
			// check for native support
			var value = !this._transformationSupport || this._bugs.scale ? this._transformations._scale : this._transformations.scale;

			return MAL.defaultValue(1, value);
		}
	},
	_getLayerScaleX: function () {
		if (!this._transparency && MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			var matrix = this._getFilter('DXImageTransform.Microsoft.Matrix');
			return matrix ? matrix[0] : 1;
		} else {
			// check for native support
			var value = !this._transformationSupport || this._bugs.scale ? this._transformations._scaleX : this._transformations.scaleX;

			return MAL.defaultValue(1, value);
		}
	},
	_getLayerScaleY: function () {
		if (!this._transparency && MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			var matrix = this._getFilter('DXImageTransform.Microsoft.Matrix');
			return matrix ? matrix[1] : 1;
		} else {
			// check for native support
			var value = !this._transformationSupport || this._bugs.scale ? this._transformations._scaleY : this._transformations.scaleY;

			return MAL.defaultValue(1, value);
		}
	},

	/**
	 * transform rotation setter
	 *
	 * @private
	 * @param {Number} value
	 */
	_setLayerPreRotate: function (value) {
		this._transformations.preRotate = value + this._suffixes.preRotate;
	},

	_setLayerRotate: function (value) {
		if (this._transformationSupport) {
			// browsers that can handle translate
			this._transformations.rotate = value + this._suffixes.rotate;
		} else if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			// store xywh info
			if (!this._bufferedCSSProperties._rotationInfo) this._bufferedCSSProperties._rotationInfo = {
				x: this.element.offsetLeft,
				y: this.element.offsetTop,
				w: this.element.offsetWidth,
				h: this.element.offsetHeight
			};

			var radian = 0.0174532925,
				angle = value * radian,
				costheta = Math.cos(angle),
				sintheta = Math.sin(angle),
				a = costheta,
				b = sintheta,
				c = -sintheta,
				d = costheta,
				e = 0,
				f = 0;

			// set linear transformation via Matrix Filter
			this._setFilter('DXImageTransform.Microsoft.Matrix', 'M11=' + a + ', M21=' + b + ', M12=' + c + ', M22=' + d + ', SizingMethod="auto expand"', value);

			// apply everything to the DOM so that our offsets are correct
			this._setCSSTextFromCache();

			// bounding box dimensions
			// IE has updated these values based on transform set above
			var wb = this.element.offsetWidth;
			var hb = this.element.offsetHeight;

			// determine how far origin has shifted
			var sx = (wb - this._bufferedCSSProperties._rotationInfo.w) / 2;
			var sy = (hb - this._bufferedCSSProperties._rotationInfo.h) / 2;

			// translation, corrected for origin shift
			// rounding helps, but doesn't eliminate, integer jittering
			this._bufferedCSSProperties.left = Math.round(this._bufferedCSSProperties._rotationInfo.x + e - sx) + 'px';
			this._bufferedCSSProperties.top = Math.round(this._bufferedCSSProperties._rotationInfo.y + f - sy) + 'px';
		}
	},

	/**
	 * transform rotation getter
	 *
	 * @private
	 */
	_getLayerRotate: function () {
		if (this._transformationSupport) {
			// browsers that can handle translate
			return this._transformations.rotate;
		} else if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			var value = this._getFilter('DXImageTransform.Microsoft.Matrix');
			return MAL.isUndefined(value) ? 0 : value;
		}
	},

	/**
	 * transform skew
	 *
	 * @private
	 * @param {Number} value
	 */
	_setLayerSkew: function (value) {
		this._transformations.skew = value + this._suffixes.skew;
	},
	_setLayerSkewX: function (value) {
		this._transformations.skewX = value + this._suffixes.skew;
	},
	_setLayerSkewY: function (value) {
		this._transformations.skewY = value + this._suffixes.skew;
	},

	/**
	 * steps to the next animation in the animation stack
	 *
	 * @private
	 */
	_startNextAnimation: function () {
		var tween = this._tweens.shift();
		if (tween) {
			// we have to do this incase someone runs an addTween right after an addLayer with start transforms
			this._stage.timeouts.addTimeout((function (self, tween) {
				return function () {
					self._signature = new Date().getTime() * Math.random();
					self._applyAnimation.apply(self, [self._signature, tween]);
				};
			}(this, tween)), 0);

			this._started = true;
		} else {
			this._started = false;
		}
	},

	/**
	 * updates the layers cssText based on the current tween thats being applied
	 *
	 * @private
	 * @param {Number} time
	 */
	_update: function (time) {
		if (!this._currentTween) return;

		var tween = this._currentTween,
			finished = time > (tween.startTime + tween.duration),
			percent = (time - tween.startTime) / tween.duration;

		// in case we are late
		if (percent > 1) percent = 1;

		// apply ease if provided
		if (MAL.isArray(tween.easing)) {
			percent = MAL.cubicBezierWithPercent(percent, tween.easing[0], tween.easing[1], tween.easing[2], tween.easing[3]);
		} else {
			percent = tween.easing(percent);
		}

		// apply changes to properties
		for (var property in tween.values.start) {
			var value = MAL.Util.shortenFloat(tween.values.start[property] + (tween.values.delta[property] * percent), 5);
			this.setProperty(property, value, true);
		}

		// apply everything to the DOM
		this._setCSSTextFromCache();

		// check if animation is complete
		if (finished) {
			this._currentTween = null;
			this._clearTemporaryCachedProperties();

			// fire persistant callback
			if (this._persistantCallback) this._persistantCallback(this);

			// fire callback
			if (tween.callback) tween.callback(this);

			this.dispatchEvent('animation-end');
			this._startNextAnimation();
		}

	},

	/**
	 * constructs and adds an animation to the tween stack, also handles the delay
	 *
	 * @private
	 * @param {Object} options
	 */
	_applyAnimation: function (signature, options) {
		// if signature missmatch, animation has been overridden
		if (signature !== this._signature) return;

		// we don't really need to do it this way but to keep the core logic consistant...
		if (options.delay) {
			this._stage.timeouts.addTimeout((function (self, signature, options) {
				return function () {
					// remove delay
					options.delay = 0;

					// apply animation with start values
					self._applyAnimation.apply(self, [signature, options]);
				};
			}(this, signature, options)), options.delay * 1000);

			return;
		}

		// checks if we have transformation support
		this._transformationSupport = !!MAL.Environment.getCSSPropertyName('transform') && !this._bugs.transformations;

		// force hardware acceleration (webkit is the only browser that supports a Z axis)
		if (MAL.Environment.has3d && !this._bugs.transform3d && this._features.backfacePerformanceFix) this._transformations.translateZ = '0px';

		// set start with props
		if (options.start) this._setStartProperties(options.start);

		// generate deltas
		var tween = {};
		tween.duration = options.duration * 1000 || 1000;
		tween.startTime = new Date().getTime();
		if (options.easing) {
			tween.easing = MAL.isString(options.easing) && this._eases[options.easing] ? this._eases[options.easing] : options.easing;
		} else {
			tween.easing = this._eases.linear;
		}

		tween.values = {
			start: {},
			delta: {},
			finish: options.finish
		};
		tween.callback = options.callback;

		for (var property in options.finish) {
			tween.values.start[property] = MAL.parseNumber(this.getProperty(property));
			tween.values.delta[property] = MAL.delta(tween.values.start[property], options.finish[property]);
		}

		this._currentTween = tween;
		this._animating = true;

		// increment iteration count
		this.iteration++;

		this.dispatchEvent('animation-start');
	},

	/**
	 * sets the layers start properties and updates the cssText
	 *
	 * @private
	 */
	_setStartProperties: function (properties) {
		this.setProperties(properties);
		this._setCSSTextFromCache();
	},

	/**
	 * set multiple properties on a layer
	 *
	 *     layer.setProperties({
	 *      opacity: .5,
	 *      x: 100,
	 *      y: 100,
	 *      'background-color': '#000000'
	 *     });
	 *
	 * refer to {@link MAL.tween.TransformObject} for possible properties
	 *
	 * @param {Object} properties
	 */
	setProperties: function (properties, dontApplyToDOM) {
		for (var property in properties)
			this.setProperty(property, properties[property], true);

		// regenerate the transform string
		if (!dontApplyToDOM) {
			if (this._transformationSupport) this._css('transform', this._constructTransformationString());
			this._setCSSTextFromCache();
		}

		return this;
	},

	/**
	 * set a layers property
	 *
	 *     layer.setProperty('rotate', 45);
	 *
	 * refer to {@link MAL.tween.TransformObject} for possible properties
	 *
	 * @param {String} name
	 * @param {String/Number} value
	 */
	setProperty: function (name, value, _dontApplyToDOM) {
		// method is used to detect if we need any additional setProperty overrides
		var method = MAL.toCamelCaseWithPrefix('_setLayer', name);

		if (this._transformationPropertyDictionary[name]) {
			// set advanced property
			if (this[method]) {
				this[method](value);
			} else {
				var transform = this._transformationPropertyDictionary[name];
				this._transformations[transform.name] = MAL.defaultValue(transform.value, value) + (this._suffixes[name] || '');
			}
		} else {
			// set basic styling property
			this._css(name, value);
		}

		// regenerate the transform string
		if (!_dontApplyToDOM) {
			if (this._transformationSupport) this._css('transform', this._constructTransformationString());
			this._setCSSTextFromCache();
		}

		return this;
	},

	/**
	 * get a layers property value
	 *
	 *     layer.getProperty('rotate');
	 *
	 * @param {String} name
	 */
	getProperty: function (name) {
		var method = MAL.toCamelCaseWithPrefix('_getLayer', name),
			value;

		if (this._transformationPropertyDictionary[name]) {
			// get transform property
			value = this[method] ? this[method]() : this._transformations[this._transformationPropertyDictionary[name].name];
		} else {
			// get basic styling property
			value = this._css(name);
		}

		return this._parsePropertyValue(value);
	},

	/**
	 * get layers current position after transformations and positioning
	 *
	 *     layer.getPosition();
	 *
	 * @return {Object} {x: Float, y: Float}
	 */
	getPosition: function () {
		return {
			x: this.cssGetComputedProperty('left') + (this._transformationSupport ? parseFloat(this._getLayerX()) : 0),
			y: this.cssGetComputedProperty('top') + (this._transformationSupport ? parseFloat(this._getLayerY()) : 0)
		};
	},

	/**
	 * sets and gets css property values
	 *
	 * @private
	 * @param {String} property name of the property
	 * @param {String/Number} value value of property
	 * @param {Object} properties set of properties
	 */
	_css: function () {
		if (!MAL.isUndefined(arguments[0]) && !MAL.isUndefined(arguments[1])) {
			// set specific value
			this._cssSetProperty(arguments[0], arguments[1]);
		} else if (MAL.isString(arguments[0])) {
			// return specific value
			return this._cssGetProperty(arguments[0]);
		} else if (MAL.isObject(arguments[0])) {
			// set multiple properties
			this._cssSetProperties(arguments[0]);
		}
	},

	/**
	 * clears temporary cached properties (tween buffer properties)
	 *
	 * @private
	 */
	_clearTemporaryCachedProperties: function () {
		for (var property in this._bufferedCSSProperties) {
			if (property.substr(0,1) === '_') delete this._bufferedCSSProperties[property];
		}
	},

	/**
	 * sets the cssText of the layer (this is very IMPORTANT)
	 *
	 * @private
	 */
	_setCSSTextFromCache: function () {
		var string = '';

		// apply filters or transforms
		if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			this._setFilterString();
		} else if (this._transformationSupport) {
			this._css('transform', this._constructTransformationString());
		}

		for (var property in this._bufferedCSSProperties) {
			// do not add things that are strictly used for computing (filters)
			if (property.substr(0,1) !== '_') {
				string += property + ': ' + this._bufferedCSSProperties[property] + '; ';
			}
		}

		this.element.style.cssText = string;
	},

	/**
	 * produces a transformation property string from the _transformations object
	 *
	 * @private
	 */
	_constructTransformationString: function () {
		var string = '';

		this._transformationFunctionOrder.forEach(function (transformation) {
			if (transformation.substr(0,1) === '_' || MAL.isUndefined(this._transformations[transformation])) return;

			// look up alias name
			var name = this._transformationFunctionAlias[transformation] ? this._transformationFunctionAlias[transformation] : transformation;

			string += ' ' + name + '(' + this._transformations[transformation] + ')';
		}, this);

		return string.substr(1);
	},

	/**
	 * produces a transformation property object from a transform string
	 *
	 * @private
	 */
	_constructTransformationObject: function (string) {
		var match,
			transformation = {},
			pattern = new RegExp(/([a-z0-9]+)\((.+?)\)/gi);

		while ((match = pattern.exec(string))) transformation[match[1]] = match[2];

		return transformation;
	},

	/**
	 * sets a filter (IE <9 only method)
	 *
	 * @private
	 * @param {String} name filter name
	 * @param {String} value filter value string
	 * @param {Number} raw filter value float
	 */
	_setFilter: function (name, value, raw) {
		// we cache the raw filter so we don't have to keep reconverting it
		this._filtersCache[name] = raw;

		// destination value string
		this._filters[name] = value;
	},

	/**
	 * returns a filters value float (IE <9 only method)
	 *
	 * @private
	 * @param {String} name filter name
	 */
	_getFilter: function (name) {
		// returns actual integer that you would put into cssSet...
		return this._filtersCache[name];
	},

	/**
	 * constructs and sets the css filter property string out of the applied filters (IE <9 only method)
	 *
	 * @private
	 */
	_setFilterString: function () {
		var string = '';

		// for performance sake use a normal for loop
		for (var i = 0; i < this._filterOrder.length; i++)
			if (!MAL.isUndefined(this._filters[this._filterOrder[i]])) string += 'progid:' + this._filterOrder[i] + '(' + this._filters[this._filterOrder[i]] + ') ';

		if (string) this._bufferedCSSProperties.filter = string.slice(0,-1);
	},

	/**
	 * sets a single css property
	 *
	 * @private
	 * @param {String} property name of the property
	 * @param {String/Number} value value of property
	 */
	_cssSetProperty: function (property, value) {
		// method is used to detect if we need any additional css overrides
		var method = MAL.toCamelCaseWithPrefix('_cssSet', property);

		// add browser prefix to properties that start with a dash
		var propertyName = MAL.Environment.getCSSPropertyName(property, true);

		if (this[method]) {
			// use override
			var override = this[method](value);

			// do we need to save it
			if (override) this._bufferedCSSProperties[override.name] = override.value;
		} else if (MAL.isNull(value)) {
			delete this._bufferedCSSProperties[propertyName];
		} else {
			// add suffix
			if (MAL.isNumber(value) && this._suffixes[property]) value = value + this._suffixes[property];

			this._bufferedCSSProperties[propertyName] = value;
		}
	},

	/**
	 * sets a set of css properties
	 *
	 * @private
	 * @param {Object} properties set of properties
	 */
	_cssSetProperties: function (properties) {
		for (var property in properties) {
			this._cssSetProperty(property, properties[property]);
		}
	},

	/**
	 * gets the value of a css property
	 *
	 * @private
	 * @param {String} property name of css property
	 */
	_cssGetProperty: function (property, raw) {
		// method is used to detect if we need any additional css overrides
		var method = MAL.toCamelCaseWithPrefix('_cssGet', property);

		// add browser prefix to properties that start with a dash
		var propertyName = MAL.Environment.getCSSPropertyName(property, true);

		if (this[method]) {
			// use override
			return this[method](property);
		} else {
			// get basic style property
			return this.element.style[propertyName] || this.cssGetComputedProperty(propertyName, raw);
		}
	},

	/**
	 * overridden opacity setter
	 *
	 * @private
	 * @param {Number} value
	 */
	_cssSetOpacity: function (value) {
		if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			// custom alpha filter for older IE versions
			this._setFilter('DXImageTransform.Microsoft.Alpha', 'opacity=' + (value * 100), value);
		} else {
			// browser supports standard alpha
			return {name: 'opacity', value: value};
		}
	},

	/**
	 * overridden opacity getter
	 *
	 * @private
	 */
	_cssGetOpacity: function () {
		if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			// custom alpha filter for older IE versions
			var value = this._getFilter('DXImageTransform.Microsoft.Alpha');
			return MAL.isUndefined(value) ? 1 : value;
		} else {
			// browser supports standard alpha
			return this._bufferedCSSProperties.opacity || this.element.style.opacity || 1;
		}
	},

	/**
	 * helper for some of our string values
	 *
	 * @private
	 */
	_scalePercentOrAdd: function (current, value) {
		if (value.indexOf('%') !== -1) {
			current *= parseFloat(value)/100;
		} else {
			current += parseFloat(value);
		}
		return current;
	}
}));
MAL.tween.JSLayerProxy = new MAL.tween.LayerProxy(MAL.tween.JSLayer);

MAL.define('services.ImageLoader', MAL.Object.extend({

	initialize: function($config) {
		this._super();

		var defaultOptions = {
			maxTime: 5,
			startTime: null,
			manualLoad: false
		};

		$config = MAL.mergeOptions(defaultOptions, $config || {});

		this._maxTime = $config.maxTime;
		this._startTime = $config.startTime;
		this._manualLoad = $config.manualLoad;

		this._timeout = null;
		this._hasTimedOut = false;
	},

	load: function() {
		var self = this,
			controllersSets = (arguments.length > 1) ? MAL.argumentsToArray(arguments) : (arguments.length === 1) ? [arguments[0]] : null;

		this._timeout = setTimeout(this._timedout, this._maxTime * 1000);
		MAL.Async.serial(
			controllersSets.map(function(controllerSet) {
				return function(callback) {
					if (self._hasTimedOut || self._checkTimedOut()) {
						return self.dispatchEvent('timeout');
					}
					if (self._manualLoad) {
						self._loadManualImages(controllerSet, callback);
					} else {
						self._loadControllers(controllerSet, callback);
					}

				};
			}),
			function() {
				if (!self._hasTimedOut) self.dispatchEvent('loaded');
				else self.dispatchEvent('timeout');
			}
		);
	},

	_loadControllers: function(controllers, callback) {
		var images = controllers.map(function(controller) {
			return controller.view.getImages();
		});
		this._preloadImages(images, callback);
	},

	_loadManualImages: function(imagelist, callback) {
		var images = imagelist;
		this._preloadImages(images, callback);
	},

	_timedout: function() {
		this._hasTimedOut = true;
	},

	_checkTimedOut: function() {
		return Date.now() - this._startTime > this._maxTime * 1000;
	},

	_preloadImages: function(images, callback) {
		var preloader = new MAL.Preloader({
			images: images
		});
		preloader.addEventListener('load', callback, false);
		preloader.load();
	}
}));

MAL.define('services.MarcomProxy', MAL.Object.extend({
	initialize: function (RMV) {
		this._super();

		this.startTime = new Date().getTime();
		this.userDetails = MAL.Environment.os+'_'+MAL.Environment.browser+'_'+MAL.Environment.browserVersion;
		var _scope = this;
		this.RMV = RMV;
		this.RMVReady = false;
		this.queue = [];
		this.uniqueEvents = [];

		// copy some stuff to this object
		this.paths = RMV.paths;
		this.filePathProxy = RMV.filePathProxy;
		this.siteAnimations = RMV.siteAnimations || false;
	
		var self = this;		
		if (RMV.events) {
			MAL.each(RMV.events,function (value, key, ref) {				
				var shortName = key.slice(2).toLowerCase();
				self.addEventListener(shortName, function(o,e,c) {
					if (!self.RMVReady) {
						self.queue.push({eventKey:key,eventArgs:c});
						return;
					}

					self.runEventCall(key,c);
				}.bind(self), true);
			});
		} else {
			MAL.log('RMV does not implement any events');
		}

		this.RMVReadyCheck();
	},

	runEventCall: function (eventKey, args) {
		
		if (this.RMV.events[eventKey].unique && this.uniqueEvents.indexOf(eventKey) !== -1) return;

		if (this.RMV.events[eventKey].unique) {
			this.uniqueEvents.push(eventKey);
		}

		if (MAL.ad && MAL.isFunction(MAL.ad.site[eventKey])) {
			MAL.ad.site[eventKey].call(this.RMV,args);
		}

		if (MAL.isFunction(this.RMV.events[eventKey])) {
			this.RMV.events[eventKey].call(this.RMV,args);	
		} else if (this.RMV.events[eventKey].handler) {
			this.RMV.events[eventKey].handler.call(this.RMV,args);
		} else {
			this.log('Event does not implement a handler');
		}
		
	},

	RMVReadyCheck: function () {
		if (this.RMV.isRMVReady) {
			if (this.RMV.isRMVReady()) {
				this.RMVReady = true;
				this.queue.forEach(function (value) {
					this.runEventCall(value.eventKey,value.eventArgs);					
				}, this);
			} else {
				setTimeout(this.RMVReadyCheck, 50);
			}
		} else {
			this.RMVReady = true;
		}
	}
}));

MAL.define('services.InViewService', MAL.Object.extend({

	initialize: function($config){
		this._super();

		var defaultOptions = {
			width: 0,
			height: 0,
			top: 0,
			left: 0
		};

		$config = MAL.mergeOptions(defaultOptions, $config || {});

		this._width = $config.width;
		this._height = $config.height;
		this._top = $config.top;
		this._left = $config.left;
		this._windowSize = this._getWindowSize();
		this._windowLocation = this._getScrollLocation();
		this._visible = this.focus = this._checkInFullView();
		this._focussed = true;
		this.hidden = null;
		this._setupWindowEvents();

		this._setCheckForVisibility();
	},

	_setCheckForVisibility: function(){
		this.hidden = this._addVisibilityListener(function(hidden){
			var eventObject =  {
				visible: this._checkInFullView(),
				focused: hidden
			};
			MAL.log(eventObject);
			this.hidden = hidden;
			this.dispatchEvent('in-view', eventObject);
		}.bind(this));
	},

	_setupWindowEvents: function() {
		MAL.addEvents(window, {
			scroll: function () {
				this._windowLocation = this._getScrollLocation();

				// dispatch event when scrolled back into view
				if(this._visible !== this._checkInFullView()) {
					var inFullView = this._checkInFullView(),
					str = '';

					switch(inFullView){
						case true: str = 'in'; break;
						case 'partial': str = 'in partial'; break;
						case false: str = 'not in'; break;
					}
					MAL.log('InViewService: ' + str + ' view (focussed)');
					this.dispatchEvent('in-view', {visible: inFullView});
				}

				this._visible = this._checkInFullView();

			}.bind(this),

			resize: function () {
				this._windowSize = this._getWindowSize();
			}.bind(this)
		});
	},

	_getScrollLocation: function () {
		var scrollY = (window.pageYOffset !== undefined) ? window.pageYOffset : (document.documentElement || document.body.parentNode || document.body).scrollTop;
		var scrollX = (window.pageXOffset !== undefined) ? window.pageXOffset : (document.documentElement || document.body.parentNode || document.body).scrollLeft;

		return {x:scrollX,y:scrollY};
	},

	_getWindowSize: function () {
		var size = {};
		if(!window.innerWidth){
			if(document.documentElement.clientWidth !== 0){
				//strict mode
				size.width = document.documentElement.clientWidth;
				size.height = document.documentElement.clientHeight;
			} else {
				//quirks mode
				size.width = document.body.clientWidth;
				size.height = document.body.clientHeight;
			}
		} else {
			size.width = window.innerWidth;
			size.height = window.innerHeight;
		}

		return size;
	},

	_checkInFullView: function () {
		var inFullView = this._top >= this._windowLocation.y && (this._top + this._height) <= this._windowSize.height;

		if(!inFullView && this._windowLocation.y < (this._top + this._height)){
			inFullView = 'partial';
		}

		this.focus = inFullView;

		return inFullView;
	},

	_addVisibilityListener: function (callback) {
		var propertyName = MAL.isUndefined(document.hidden) ? MAL.Environment.engine + 'Hidden' : 'hidden',
			eventName = MAL.isUndefined(document.hidden) ? MAL.Environment.engine + 'visibilitychange' : 'visibilitychange';

		/*document.addEventListener(eventName, function () {
			callback(document[propertyName]);
		});*/

		if (document.addEventListener) {
			document.addEventListener(eventName, function () {
				callback(document[propertyName]);
			}); 
		} else if (document.attachEvent) {
			document.attachEvent(eventName, function () {
				callback(document[propertyName]);
			});
		}

		return document[propertyName];
	}

}));

/*global MAL*/
MAL.define('services.MouseHotSpot', MAL.Object.extend({
	initialize: function ($config) {
		$config = $config || {};
		this._super();

		this._working = true;

		this._bounds = [];
		this._inBounds = false;
		this._deActivateTimer = null;
		this._activateTimer = null;

		this._mouseObj = {};

		this._deActivateDelay = 0;
		this._activateDelay = 0;

		this._zeroObj = null;

		this._autoDeactivate = false;
		this._active = false;
		this.name = "";
		this._instantActive = false;

		if ($config.bounds) this.addBounds($config.bounds);
		if ($config.zero) this.addZero($config.zero);
		if ($config.deActivateDelay) this.setDeactivateDelay($config.deActivateDelay);
		if ($config.activateDelay) this.setActivateDelay($config.activateDelay);

		MAL.addEvent(window,'mousemove', this._onMouseMove.bind(this));
		MAL.addEvent(window,'click', this._onMouseMove.bind(this));
	},

	setDisabled: function () {
		//MAL.log('DISABLING HOTSPOT');
		this._working = false;
	},

	setEnabled: function() {
		MAL.log('ENABLING HOTSPOT');
		this._working = true;
	},

	isWorking: function() {
		return this._working;
	},

	setDeactivateDelay: function (delay) {
		this._deActivateDelay = delay*1000;
		this._autoDeactivate = true;
	},

	setActivateDelay: function (delay) {
		this._activateDelay = delay;
	},

	addBounds: function (bounds) {

		bounds.x = bounds.x || 0;
		bounds.y = bounds.y || 0;

		this._bounds.push(bounds);


	},

	addZero: function (element) {
		if (typeof element === 'string') {
			this._zeroObj = document.getElementById(element);
		} else {
			this._zeroObj = element;
		}

	},

	_clearTimers: function () {
		clearTimeout(this._activateTimer);
		clearTimeout(this._deActivateTimer);
	},

	_deActivate: function () {

		if (!this._instantActive) return;

		this._instantActive = false;
		this._clearTimers();

		this.dispatchEvent('deactivate');
		this._active = false;
	},


	_activate: function() {


		if (this._instantActive) return;

		this._instantActive = true;

		if (this._autoDeactivate) {
			this._setDeactiveTimer();
		}
		clearTimeout(this._activateTimer);
		this._activateTimer = setTimeout(this._delayedActivate.bind(this), this._activateDelay);

	},

	_currInBounds: function () {
		if (!this._active && !this._instantActive) this._activate();
		this.dispatchEvent('active');

		if (this._autoDeactivate) {
			this._setDeactiveTimer();
		}

	},

	_currOutBounds: function () {
		if (this._active) return;
		this.dispatchEvent('inactive');
	},

	_delayedActivate: function() {

		if (this._active) return;

		this._active = true;
		clearTimeout(this._activateTimer);

		this.dispatchEvent('activate');
	},

	_setDeactiveTimer: function () {
		clearTimeout(this._deActivateTimer);
		this._deActivateTimer = setTimeout(function () {
			this._deActivate();
		}.bind(this), this._deActivateDelay);
	},

	_checkBounds: function (x, y) {
		var inBounds = false,
			cX,
			cY;
		for (var i = 0, l = this._bounds.length; i < l; i++) {

			var bounds = this._bounds[i];


			if (this._zeroObj != null) {
				var zeroPosition = MAL.findElementPosition(this._zeroObj);
				cX = this._bounds[i].x + zeroPosition.x;
				cY = this._bounds[i].y + zeroPosition.y;

			} else {
				cX = this._bounds[i].x;
				cY = this._bounds[i].y;
			}


			if (cX <= x && (cX+bounds.width) >= x && cY <= y && (cY+bounds.height) >= y) inBounds = true;
		}

		if (inBounds) {
			this.dispatchEvent('inbound');
		}
		if ((inBounds && !this._inBounds) || (!this._active && inBounds)) {
			this._currInBounds();

			if (inBounds && !this._inBounds)
				this._activate();

		} else if (inBounds && this._inBounds && this._autoDeactivate) {
			this._setDeactiveTimer();


		} else if (!inBounds && this._instantActive) {
			this._currOutBounds();
			this._deActivate();
			this._clearTimers();
		}



		this._inBounds = inBounds;
	},

	_setMouse: function(x,y) {

		var scrollY = (window.pageYOffset !== undefined) ? window.pageYOffset : (document.documentElement || document.body.parentNode || document.body).scrollTop;


		if (this._zeroObj != null) {

			var zeroPosition = MAL.findElementPosition(this._zeroObj);


			this._mouseObj.x = x - zeroPosition.x;
			this._mouseObj.y = (y - zeroPosition.y) + scrollY;
		} else {
			this._mouseObj.x = x;
			this._mouseObj.y = (y + scrollY);
		}


	},

	getMouse: function() {

		return this._mouseObj;

	},

	_onMouseMove: function (e) {

		if (!this._working) return;

		var scrollY = (window.pageYOffset !== undefined) ? window.pageYOffset : (document.documentElement || document.body.parentNode || document.body).scrollTop;
		var scrollX = (window.pageXOffset !== undefined) ? window.pageXOffset : (document.documentElement || document.body.parentNode || document.body).scrollLeft;

		this._setMouse(e.clientX,e.clientY);
		this._checkBounds((e.clientX+scrollX), (e.clientY+scrollY));
	}
}));

MAL.define('services.FrequencyCap', MAL.Object.extend({

	initialize: function($config){
		$config = $config || {};
		this._super();

		this._capAmount = $config.amount || 0;
		this._placementName = "MAL." + $config.placement + "." || "MAL.";

	},

	check: function() {

		if (this._capAmount === 0) return false;

		var date = new Date(),
			dateString = date.getFullYear().toString() + (date.getMonth() + 1).toString() + date.getDate().toString(),
			storageKey = this._placementName + dateString + '.views', // eg MAL.20121214.views
			views;

		if (MAL.Environment.hasLocalStorage) {
			views = localStorage.getItem(storageKey);			
			localStorage.setItem(storageKey, views === null ? 1 : ++views);
			
			if (views > this._capAmount) {
				return true;
			}
		}

		return false;
	}

}));

MAL.define('services.Keyline', MAL.Object.extend({

    /**
     * For rendering the keyline.
     *
     * @type HTMLCanvasElement
     * @private
     */
    _canvas: null,

    /**
     * CanvasContext2D
     */
    ctx: null,

    /**
     * For displaying the rendered keyline
     *
     * @type HTMLImageElement
     */
    element: null,

    /**
     * Color of the keyline
     *
     * @type string
     */
    color: '',

    /**
     * Width of the keyline,
     */
    width: -1,


    /**
     * Height of the keyline,
     */
    height: -1,

    /**
     *
     * @param $config
     * @returns {*|Error}
     */
    initialize: function ($config) {
        $config = $config || {};
        this._super();

        this.color = $config.color || '#000000';
        this.width = $config.width || MAL.error('keyline requires a width');
        this.height = $config.height || MAL.error('keyline requires a height');

        this._canvas = document.createElement('canvas');
        this._canvas.width = this.width;
        this._canvas.height = this.height;

        this.element = $config.drawImg ? document.createElement('img') : this._canvas;

        if (!this._canvas.getContext && !this.element.getContext('2d')) {
            return MAL.error('keyline only supports canvas');
        }

        this.ctx = this._canvas.getContext('2d');

        if( $config.drawImg ) {
            this.drawImg();
            this.drawImg();
            this.drawImg();
        } else {
            this.drawCanvas();
            this.drawCanvas();
            this.drawCanvas();
        }
 

        if ($config.container) {
            $config.container.appendChild(this.element);
        }
    },

    /**
     * Draws keyline and puts the result in a image.
     */
    drawImg: function () {
        this.ctx.strokeStyle = this.color;
        this.ctx.rect(0, 0, this.width, this.height);
        this.ctx.stroke();

        // set rendered data from canvas to image src. this fixes ie10,11 subpixel problems
        this.element.src = this._canvas.toDataURL('image/png');
    },

    drawCanvas: function() {
        this.ctx.strokeStyle = this.color;
        this.ctx.moveTo(0,0);
        this.ctx.lineTo(this.width,0);
        this.ctx.lineTo(this.width,this.height);
        this.ctx.lineTo(0,this.height);
        this.ctx.lineTo(0,0);
        this.ctx.stroke();
    }

}));

MAL.define('services.Supertag', MAL.Object.extend({
	initialize: function($config, collapse){
		this._super();
		this.config = $config;
		this.connection = this.config.connection;
		this.$collapse = collapse || this.config.collapse;
		this.setup();
		this.size = null;
		return this;
	},

	setup: function(){

		if(this.$collapse){
			this.collapse();
			return;
		}

		this.breakpoints = this.config.breakpoints || null;

		this.config.placementId = typeof prPlacementId !== 'undefined' ? prPlacementId : 0;

		if(this.breakpoints === null){
			MAL.error('No breakpoints set.');
		}

		/*this.connection.setDebugMode(true);
		if(MAL.Environment.browser === 'msie' && !window.console){
			this.connection.setDebugMode(false);
		}*/

		var sizes = this.sizes = MAL.clone(this.breakpoints);
		sizes.width = [];
		if(this.breakpoints && this.breakpoints.width){
			for (var width in this.breakpoints.width){
				if(width === 'max'){
					sizes.width.push(200000);
				} else{
					sizes.width.push(parseInt(width, 10));
				}
			}
		}
		sizes.height = [];
		if(this.breakpoints && this.breakpoints.height){
			for (var height in this.breakpoints.height){
				if(height === 'max'){
					sizes.height.push(200000);
				} else{
					sizes.height.push(parseInt(height, 10));
				}
			}
		}

		this.connection.updateAdSize = this.update;
		this.connection.updatePageScroll = this.updatePageScroll;

		this.connectionInstance = this.connection.createConnection(this.config.container, sizes, this.config.placementId, this.config.nested);


		this.waitForPageInfo(10);
	},

	waitForPageInfo: function(attempts) {

		var self = this;
		var pageInfo = this.connection.getPubPageInfo();

		if (attempts > 0) {
			attempts--;
			setTimeout(function() {
				if (MAL.Util.isObject(pageInfo) && !MAL.Util.isUndefined(pageInfo.window)) {
					
					self.dispatchEvent('page-ready', {
						windowSize: pageInfo.window,
						pageScroll: pageInfo.scroll,
						bannerRect: pageInfo.iframe
					});
				} else {
					self.waitForPageInfo(attempts);
				}
			}, 50);
		} else {
			self.dispatchEvent('page-failed', {});
		}
	},

	update: function(sizeIndex, type){
		console.log(arguments);

		var size = this.sizes[type][sizeIndex];

		if(size === 200000){
			size = 'max';
		}

		if(this.breakpoints[type] && this.breakpoints[type][size]){
			this.connection.sendMessageToPub(this.breakpoints[type][size]);
			this.dispatchEvent('size-update', {change: type, current:size, prior: this.size});
			this.size = size;
		}
	},

	addPageListener: function(target, eventName, callback){
		this.connection.addPageListener(target, eventName, callback);
	},

	setDebug: function(debug){
		// this.connection.setDebugMode(debug);
	},

	message: function(message){
		this.connection.sendMessageToPub(message);
	},

	updatePageScroll: function(position){
		MAL.log('updated scroll position: '+position);
	},

	setAdSlotTarget: function(slot){
		this.connection.sendMessageToPub({setAdSlotTarget: slot});
	},

	addCustomStyles: function(message){
		this.connection.sendMessageToPub({addDirectStyles: message});
	},

	addImportantStyles: function(message){
		this.connection.sendMessageToPub({overwriteImportantStyles: message});
	},

	collapse: function(){
		this.connection.closeAd();
	},

	hide: function(){
		this.connection.sendMessageToPub({hide: null});
	},

	reveal: function(){
		this.connection.sendMessageToPub({reveal: null});
	},

	page: function(){
		return this.connection.getPubPageInfo();
	}
}));

MAL.define('services.ScrollRange', MAL.Object.extend({

	initialize: function($config){
		this._super();

		var defaultOptions = {
			top: 0.0,
			bottom: 1.0
		};

		$config = MAL.mergeOptions(defaultOptions, $config || {});

		this.top = $config.top;
		this.bottom = $config.bottom;
	},

	map: function(input, min, max) {

		var value = (input < this.top) ? this.top :
					(input > this.bottom) ? this.bottom :
					input;

		return min + ((max - min) * (value - this.top) / (this.bottom - this.top));
	},

	isInRange: function(value) {

		if (value >= this.top && value <= this.bottom && !MAL.Util.isUndefined(this.top) && !MAL.Util.isUndefined(this.bottom)) {
			return true;
		}

		return false;
	},

	disable: function() {
		this.top = undefined;
		this.bottom = undefined;
	}

}));

MAL.define('views.Main', MAL.View.extend({

    styles: function() {
        return {};

    },

    html: ''

}));

MAL.define('views.RichMedia.Main', MAL.View.extend({

    initialize: function(){
        this.html = this.renderHtml();
        this._super();
    },

    styles: function () {
        var styles = {
                'view': {
                    'position': 'relative',
                    'z-index': 0,
                    'width': 1300,
                    'height': 265,
                    'margin': 'auto',
                    'backface-visibility' : 'hidden',
                    'cursor': 'auto',
                    'overflow': 'hidden'
                },

                '#clickthrough': {
                    'z-index': 999999,
                    'display': 'block',
                    'position': 'absolute',
                    'width': '100%',
                    'height': '100%',
                    'top': 0,
                    'left': 0,
                    'cursor': 'pointer'
                },

                '#frames':{
                    'position': 'absolute',
                    'overflow': 'hidden',
                    'width': '100%',
                    'height': '100%',
                    'top': 0,
                    'left': 0,
                    'bottom': 0,
                    'right': 0,
                    'transform-origin': '0 0 0',
                    '-webkit-transform-origin': '0 0 0'
                },

                '#frame': {
                    'position': 'absolute',
                    'overflow': 'hidden',
                    'display': 'inline-block',
                    'width': '1300px',
                    'height': 'auto',
                    'top': 0,
                    'left': 0,
                    'bottom': 0,
                    'right': 0,
                    'margin': 'auto',
                    'background-repeat': 'no-repeat',
                    'background': 'url(sprites-' + MAL.placement.geo + '.jpg) no-repeat'
                },

                '#supers':{
                    'position': 'absolute',
                    'overflow': 'hidden',
                    'width': '100%',
                    'height': '265px',
                    'top': 0,
                    'left': 0,
                    'bottom': 0,
                    'right': 0,
                    'transform-origin': '0 0 0' ,
                    '-webkit-transform-origin': '0 0 0'
                },

                '.super':{
                    'position': 'absolute',
                    'overflow': 'hidden',
                    'width': '100%',
                    'height': '265px',
                    'background-repeat': 'no-repeat',
                    'opacity': 0,
                },

                '#super-1':{
                    'background-image': 'url(super_1-1300-' + MAL.placement.geo + '.svg)'
                },

                '#super-2':{
                    'background-image': 'url(super_2-1300-' + MAL.placement.geo + '.svg)'
                },


                '#loader': {
                    'position': 'absolute',
                    'width': '100%',
                    'height': '100%',
                    'top': 0,
                    'left': 0,
                    'background-color': '#ffffff',
                    'opacity': 0.999
                },

                '#logo':{
                    'width': 35,
                    'height': 43,
                    'position': 'absolute',
                    'top': '50%',
                    'left': '50%',
                    //'display': 'none',
                    'margin-left': -17.5,
                    'margin-top' : -40,
                    'z-index': 100,
                    'background': 'url(data:image/gif;base64,R0lGODlhIwArAMQAALa2ttjY2MXFxfr6+vX19d3d3efn57u7u87OzsDAwPDw8OLi4uvr69PT08nJyf///7GxsQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAjACsAAAXx4COOZGk+RHNAzOm+ZQDNLGybg0DPw+2Ljh3k8PMVhJBA8QYQAnrL1wJpiMIaQqX1paMttjAdAEEAEwoNAaJQxhnSAkGgCmN0hQm2yIBoIgEBUCVYSHh+hTQHCiUIiI6PEACLIjKQlkhEDwqXnDQJk3edkAlQBKKXTyOVp49fI6GsfyWxjwiztIhaI7iILSS8ha67wFm3xDQCJYfHEG0isMS2qsw7vg8G1DSSw9mRBUDdyNfhM5kr4VpH3akPA+fUug9T1Oyv1MIjBMu4Di7YwAcEmVBHaxsMgjvGBFiYoJBBGwwaznBAhwSBABIhkDERAgA7)'
                },

                '#spinner': {
                    'position': 'absolute',
                    'width': 16,
                    'height':16,
                    'top': '50%',
                    //'display': 'none',
                    'left': '50%',
                    'margin-left': -8,
                    'margin-top': 20,
                    'background': 'url(data:image/gif;base64,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)',
                    'z-index': 100
                },

                '#overlay':{
                    'position': 'absolute',
                    'overflow': 'hidden',
                    'width': '100%',
                    'height': '100%',
                    'cursor': 'pointer',
                    'background-color': '#ff0000',
                    'opacity': 0
                }
            };

            return styles; 
    },

    renderHtml : function(){

        var clickthrough = '<div id="clickthrough"></div>';

        var frames =    '<div id="frames">'+
                            '<div id="frame" ></div>' +
                        '</div>';

        var supers =    '<div id="supers">' + 
                            '<div id="super-1" class="super"></div>' +
                            '<div id="super-2" class="super"></div>' +
                        '</div>';

        var loader =    '<div id="loader">' + 
                            '<div id="logo"></div>' +
                            '<div id="spinner"></div>' +
                        '</div>';

        var overlay =   '<div id="overlay"></div>'

        return clickthrough + frames + supers + loader + overlay;
    },

    html: ''

}));

MAL.define('controllers.Main', MAL.ViewController.extend({
	initialize: function(){
		this._super();

		var self = this;

		// rev up to make sure library change gets pushed through
		if (location.search.match(/removecap/)){
			delete localStorage['MALBWTRK'];
		}

		this.startTime = Date.now();

		this.controllers = [];

		this.imageQueue = new Array();
		this.imageLoader = new MAL.services.ImageLoader({maxTime: 10, startTime: this.startTime, manualLoad: true});

		this.responsiveRMV = MAL.responsiveRMV;

		MAL.rmv.dispatchEvent('init', {creative:window.MALCreative});

		this.site = new MAL.Site(this);

		// this is just for demonstration, and will be done in a much cleaner way
		this.stylesheet = new MAL.Stylesheet({
			callback: this._writeMainViewToDomAndLoad.bind(this)
		});

		this.domContainer = document.getElementById('mal-ad-container');


		this._animationFramePolyfill();
	},

	_writeMainViewToDomAndLoad: function () {
		this.stylesheet.addView(this);
		this.stylesheet.writeStyles();

		// attempt to center the loader
		this.ui.loader.setProperties({'width': this.currentWidth * this.currentScale + 'px', 'height': this.maxHeight * this.currentScale / 2 + 'px'});

		this.domContainer.appendChild(this.view.element);

		var self = this;

		this.dispatchEvent('stylesheet-ready');
	},

	_writeViewsToDOM: function () {
		var controllers = MAL.argumentsToArray(arguments);

		// write styles
		this.stylesheet.addViews(controllers);
		this.stylesheet.writeStyles();

		// add views to DOM
		this.view.appendElementsTo(controllers);
	},

	_clickThrough: function () {

		MAL.rmv.dispatchEvent('clickthrough');
	},




	// polyfill for standardizing the requestAnimationFrame API
	// courtesy of paul irish
	_animationFramePolyfill: function() {
		var lastTime = 0;
		var vendors = ['webkit', 'moz'];
		for (var x = 0; x < vendors.length && !window.requestAnimationFrame; ++x) {
		window.requestAnimationFrame = window[vendors[x]+'RequestAnimationFrame'];
			window.cancelAnimationFrame =
			window[vendors[x]+'CancelAnimationFrame'] || window[vendors[x]+'CancelRequestAnimationFrame'];
		}
		if (!window.requestAnimationFrame) {
			window.requestAnimationFrame = function(callback, element) {
				var currTime = new Date().getTime();
				var timeToCall = Math.max(0, 16 - (currTime - lastTime));
				var id = window.setTimeout(function() { callback(currTime + timeToCall); },
				timeToCall);
				lastTime = currTime + timeToCall;
				return id;
			};
		}
		if (!window.cancelAnimationFrame) {
			window.cancelAnimationFrame = function(id) {
				clearTimeout(id);
			};
		}
	}

}));



MAL.define('controllers.RichMedia.Main', MAL.controllers.Main.extend({
	initialize: function(config) {
		this._super();

		var self = this;


		// width / height of the window
		this.windowSize = config.windowSize;

		// x / y scroll position of page
		this.pageScroll = config.pageScroll;

		// bounding box of iframe
		this.bannerRect = config.bannerRect;

		// flag to disable events when hidden
		this.isHidden = false;



		// listen to size-update and make any necessary changes here.
		MAL.supertag.addEventListener('size-update', function(object, details) {
			console.log('size update', details);
			if (details.change === 'width') {
				if (details.current === 699) {
					self.isHidden = true;
					setTimeout(function() {

						// hide ad unit and reset page styles
						MAL.supertag.connection.sendMessageToPub({
							'addDirectStyles': {
								'mal-ad-wrapper': {
									'display': 'none'
								},
								'#mal-outer-wrapper': {
									'margin-bottom': '0px'
								},
								'#mal-upper-wrapper': {
									'transform': 'translateY(0)',
									'-webkit-transform': 'translateY(0)',
								},
								'#mal-lower-wrapper': {
									'margin-top': '0px',
									'transform': 'translateY(0)',
									'-webkit-transform': 'translateY(0)',
								}
							}
						});		
					}, 100);
					MAL.rmv.dispatchEvent('hide');
				}

				if (details.current === 'max') {
					self.isHidden = false;
					setTimeout(function() {

						// show ad unit and reset page styles
						MAL.supertag.connection.sendMessageToPub({
							'addDirectStyles': {
								'mal-ad-wrapper': {
									'display': 'block'
								},
								'#mal-outer-wrapper': {
									'margin-bottom': '-120px'
								},
								'#mal-lower-wrapper': {
									'margin-top': '-150px'
								}
							}
						});	
					}, 100);
					MAL.rmv.dispatchEvent('devicesize', 'large');
				}
			}
				
		});


		// instantiate all controllers (this is the only one for now)
		this.controllers.push(this);


		// cache references to elements (refer to views.RichMedia.Main)
		this.ui = {
			frames: this.view.queryLayer('#frames'),
			frame: this.view.queryLayer('#frame'),
			supers: this.view.queryLayer('#supers'),
			super1: this.view.queryLayer('#super-1'),
			super2: this.view.queryLayer('#super-2'),
			loader: this.view.queryLayer('#loader'),
			spinner: this.view.queryLayer('#spinner'),
			overlay: this.view.queryLayer('#overlay')
		};



		// max / min settings for mapped properties

		this.smallWindowHeight = 700;
		this.mediumWindowHeight = 1000;

		this.minWidth = 680;
		this.maxWidth = 1300;
		this.currentWidth = 1300;

		this.currentOffset = 0;

		this.maxScale = MAL.placement.dimensions.width / this.maxWidth;
		this.currentScale = (this.windowSize.width < MAL.placement.dimensions.width) ? this.windowSize.width / this.maxWidth : this.maxScale;
		this.minSuperScale = 0.7

		this.currentScroll = this.bannerRect.top / this.windowSize.height;

		this.minHeight = 115;
		this.maxHeight = 265;
		this.currentHeight = this.minHeight;

		this.prevFrame;
		this.prevFrames = [];
		this.currentFrame = 0;
		this.minFrame = 0;
		this.maxFrame = 199;

		this.minOpacity = 0;
		this.maxOpacity = 1;
		this.currentOpacity1 = this.minOpacity;
		this.currentOpacity2 = this.minOpacity;

		this.minTranslate = 0;
		this.maxTranslate = this.minHeight - this.maxHeight;
		this.currentTranslate = this.minTranslate;


		// scroll ranges - refer to MAL.services.ScrollRange for details

		// main ranges where laptop opens and closes
		this.openingRange = new MAL.services.ScrollRange();
		this.closingRange = new MAL.services.ScrollRange();

		// used to translate page content upwards
		this.translateRange = this.openingRange;

		// ranges for showing / hiding supers
		this.showSuper1Range = new MAL.services.ScrollRange();
		this.hideSuper1Range = new MAL.services.ScrollRange();
		this.showSuper2Range = new MAL.services.ScrollRange();
		this.hideSuper2Range = new MAL.services.ScrollRange();

		// range where clickthrough is active
		this.clickableRange = new MAL.services.ScrollRange();


		// hard-coded height values for each frame
		this.frames = [
			115,
			115,
			115,
			115,
			116,
			116,
			116,
			116,
			116,
			116,
			116,
			116,
			117,
			117,
			118,
			118,
			119,
			119,
			119,
			119,
			119,
			120,
			120,
			121,
			121,
			122,
			122,
			123,
			123,
			124,
			124,
			125,
			125,
			126,
			127,
			127,
			128,
			129,
			130,
			130,
			131,
			132,
			133,
			134,
			134,
			135,
			136,
			137,
			137,
			138,
			139,
			140,
			141,
			141,
			142,
			143,
			144,
			145,
			146,
			147,
			148,
			149,
			150,
			151,
			152,
			153,
			154,
			155,
			156,
			157,
			158,
			159,
			160,
			161,
			162,
			163,
			164,
			165,
			166,
			167,
			168,
			169,
			170,
			171,
			172,
			174,
			175,
			176,
			177,
			179,
			180,
			181,
			182,
			183,
			184,
			185,
			186,
			188,
			189,
			190,
			191,
			192,
			193,
			195,
			196,
			197,
			198,
			199,
			200,
			201,
			202,
			203,
			204,
			206,
			207,
			208,
			209,
			210,
			211,
			212,
			213,
			214,
			215,
			216,
			218,
			219,
			220,
			221,
			222,
			223,
			224,
			225,
			226,
			227,
			228,
			229,
			231,
			232,
			232,
			233,
			234,
			235,
			236,
			236,
			237,
			238,
			239,
			240,
			241,
			242,
			243,
			244,
			244,
			245,
			246,
			247,
			247,
			248,
			249,
			250,
			251,
			252,
			252,
			253,
			253,
			254,
			255,
			256,
			256,
			257,
			258,
			258,
			259,
			260,
			260,
			260,
			261,
			261,
			261,
			261,
			262,
			262,
			262,
			262,
			263,
			263,
			263,
			263,
			264,
			264,
			264,
			264,
			264,
			265,
			265,
			265,
			265,
			265,
			265,
			265
		];


		// kick things off by loading assets
		this._loadImages();
	},

	_updateRanges: function() {
		if (this.windowSize.height < this.smallWindowHeight) {
			this._setSmallRanges();
		} else if (this.windowSize.height < this.mediumWindowHeight) {
			this._setMediumRanges();
		} else {
			this._setLargeRanges();
		}
	},


	_setSmallRanges: function() {
		
		this.openingRange.top = 0.10;
		this.openingRange.bottom = 0.80;

		this.closingRange.disable();

		this.translateRange.top = this.openingRange.top;
		this.translateRange.bottom = this.openingRange.bottom;

		
		this.showSuper1Range.disable();
		this.hideSuper1Range.disable();
		
		this.showSuper2Range.top = 0.10;
		this.showSuper2Range.bottom = 0.40;
		
		this.hideSuper2Range.disable();


		this.clickableRange.top = -0.20;
		this.clickableRange.bottom = 0.20;
	},

	_setMediumRanges: function() {
		
		this.openingRange.top = 0.75 - this.pixelsToPercentage(this.maxHeight * this.currentScale);
		this.openingRange.bottom = 1.00 - this.pixelsToPercentage(this.minHeight * this.currentScale);

		this.closingRange.top = 0.00;
		this.closingRange.bottom = 0.25;

		this.translateRange.top = this.openingRange.top;
		this.translateRange.bottom = this.openingRange.bottom;

		
		this.showSuper1Range.top = 0.60;
		this.showSuper1Range.bottom = 0.65;

		this.hideSuper1Range.top = 0.35;
		this.hideSuper1Range.bottom = 0.40;
		
		this.showSuper2Range.top = 0.30;
		this.showSuper2Range.bottom = 0.35;
		
		this.hideSuper2Range.top = 0.10;
		this.hideSuper2Range.bottom = 0.15;


		this.clickableRange.top = 0.20;
		this.clickableRange.bottom = 0.75 - this.pixelsToPercentage(this.maxHeight * this.currentScale);
	},

	_setLargeRanges: function() {

		this.openingRange.top = 0.70 - this.pixelsToPercentage(this.maxHeight * this.currentScale);
		this.openingRange.bottom = 0.90 - this.pixelsToPercentage(this.minHeight * this.currentScale);

		this.closingRange.top = 0.05;
		this.closingRange.bottom = 0.25;

		this.translateRange.top = this.openingRange.top;
		this.translateRange.bottom = this.openingRange.bottom;

		
		this.showSuper1Range.top = 0.58;
		this.showSuper1Range.bottom = 0.65;

		this.hideSuper1Range.top = 0.40;
		this.hideSuper1Range.bottom = 0.45;
		
		this.showSuper2Range.top = 0.35;
		this.showSuper2Range.bottom = 0.40;
		
		this.hideSuper2Range.top = 0.15;
		this.hideSuper2Range.bottom = 0.20;


		this.clickableRange.top = 0.20;
		this.clickableRange.bottom = 0.70 - this.pixelsToPercentage(this.maxHeight * this.currentScale);
	},




	_loadImages: function() {

		var self = this;

		// add the sprite sheet to the image queue and load it
		self.imageQueue.push('sprites-' + MAL.placement.geo + '.jpg');
		self.imageLoader.load(this.imageQueue);

		// listen for images to finish loading...
		MAL.Util.addEvent(this.imageLoader, 'loaded', function() {
			MAL.rmv.dispatchEvent('load');
			self.dispatchEvent('dom-ready');
			MAL.log('controllers.RichMedia.Main 	|	images finished loading');

			// ... and continue the setup process
			setTimeout(function() {
				self._setup();
			}, 100);

		});

		// listen for images failing to complete loading in time
		MAL.Util.addEvent(this.imageLoader, 'timeout', function() {
			MAL.log('controllers.RichMedia.Main 	|	image loading timed out');
			// TODO: add static fallback scenario
		});
	},



	_setup: function() {
		this._hideLoader();

		this._updateScroll();
		this._updateScale();

		this._updateFrame();
		this._updateHeight();
		this._updateTranslate();
		this._updateSprite();
		this._updateLoader();
		this._updateOverlay();

		this._addListeners();
		this._render();
	},

	pixelsToPercentage: function(pixels) {
		return pixels / this.windowSize.height;
	},

	percentageToPixels: function(percentage) {
		return percentage * this.windowSize.height;
	},



	_updateScroll: function() {
		// calculate top of banner as percentage of window height
		this.currentScroll = this.bannerRect.top / this.windowSize.height;
	},

	_updateScale: function() {
		// called whenever the window is resized
		// updates all properties whose values are dependent on window size or banner scale

		this.currentScale = (this.windowSize.width < MAL.placement.dimensions.width) ? this.windowSize.width / this.maxWidth : this.maxScale;
		this.currentWidth = this.maxWidth;
		this.currentOffset = Math.floor(this.maxHeight * (1 - this.currentScale) * -0.5);

		this._updateRanges();

		this.ui.frames.setProperties({
			'scale': this.currentScale,
			'transform-origin': '0 0 0'
		});
		this.ui.supers.setProperties({
			'scale': (this.currentScale < this.minSuperScale) ? this.minSuperScale : this.currentScale,
			'transform-origin': '0 0 0',
			'margin-left': (this.currentScale < this.minSuperScale) ? (((this.maxWidth * this.currentScale) - (this.maxWidth * this.minSuperScale)) / 2) + 'px' : '0px',
			'margin-top': (this.currentScale < this.minSuperScale) ? (((this.maxHeight * this.currentScale) - (this.maxHeight * this.minSuperScale)) / 2) + 'px' : '0px'
		});

		this._updateLoader();
		this._updateOverlay();

		MAL.supertag.connection.sendMessageToPub({
			'addCustomStyles': {
				'width': (this.currentWidth * this.currentScale) + 'px'
			}
		});

		this._updateFrame(true);
	},


	_addListeners: function() {

		// listen to resize and scroll
		MAL.supertag.connection.addPageListener('window', 'resize', this._onResize);
		MAL.supertag.connection.addPageListener('window', 'scroll', this._onScroll);

		// standard clickthrough (see base controller)
		MAL.Util.addEvent(this.view.element, 'click', this._clickThrough);
	},


	// handler for resize event
	_onResize: function(event) {

		if (this.isHidden) return;

		this.windowSize.width = event.innerWidth;
		this.windowSize.height = event.innerHeight;

		this.bannerRect = event.rect;

		this._updateScroll();
		this._updateScale();
	},

	// handler for scroll event
	_onScroll: function(event) {

		if (this.isHidden) return;

		this.pageScroll.x = event.scrollX;
		this.pageScroll.y = event.scrollY;

		this.bannerRect = event.rect;

		this._updateScroll();
	},


	// render loop using requestAnimationFrame
	// note that renders will only happen when required
	_render: function() {

		this._updateFrame();
		this._updateSuper1();
		this._updateSuper2();

		requestAnimationFrame(this._render);
	},



	_updateFrame: function(force) {

		var prevFrame = this.currentFrame;

		
		if (this.openingRange.isInRange(this.currentScroll)) {

			// opening
			this.currentFrame = Math.ceil(this.openingRange.map(this.currentScroll, this.maxFrame, this.minFrame));

		} else if (this.closingRange.isInRange(this.currentScroll)) {

			// closing
			this.currentFrame = Math.ceil(this.closingRange.map(this.currentScroll, this.minFrame, this.maxFrame));

		} else {

			if (this.currentScroll > this.closingRange.bottom && this.currentScroll < this.openingRange.top) {

				// open (middle of page)
				this.currentFrame = this.maxFrame;

			} else if (this.currentScroll > this.openingRange.bottom || this.currentScroll < this.closingRange.top) {

				// closed (top or bottom of page)
				this.currentFrame = this.minFrame;

			}
		}


		if (this.currentFrame !== prevFrame || force == true) {

			this.prevFrame = prevFrame;

			this._updateHeight();
			this._updateSprite();
			this._updateTranslate();
			this._doEffect();
		}
	},


	_updateHeight: function() {
		// lookup frames hash to find height value corresponding to current frame index
		this.currentHeight = this.frames[this.currentFrame];
	},

	_updateSprite: function() {
		// set the image that is displayed by changing background-position of the frame element
		var row = Math.floor(this.currentFrame / 10)
		var column = this.currentFrame % 10;
		this.ui.frame.setProperty('background-position', (this.maxWidth * column * -1) + 'px ' + (this.maxHeight * row * -1) + 'px');
	},

	_updateTranslate: function() {

		if (this.translateRange.isInRange(this.currentScroll)) {

			// opening
			this.currentTranslate = Math.floor((this.minHeight - this.currentHeight) * this.currentScale);

		} else if (this.currentScroll + this.pixelsToPercentage(this.currentHeight * this.currentScale) > this.translateRange.bottom) {

			// closed (bottom of page)
			this.currentTranslate = this.minTranslate;

		} else if (this.currentScroll < this.translateRange.top) {

			// open (top of page)
			this.currentTranslate = Math.floor(this.maxTranslate * this.currentScale);

		}
	},

	_doEffect: function() {
		// gives the impression of the ad unit expanding by translating the page content above whilst offsetting page content below

		var delta = Math.ceil((this.currentHeight - this.minHeight) * this.currentScale);

		MAL.supertag.message({
			'addDirectStyles': {
				'#mal-upper-wrapper': {
					'transform': 'translate3d(0, ' + this.currentTranslate + 'px, 0)',
					'-webkit-transform': 'translate3d(0, ' + this.currentTranslate + 'px, 0)'
				},
				'#mal-lower-wrapper': {
					'transform': 'translate3d(0, ' + delta + 'px, 0)',
					'-webkit-transform': 'translate3d(0, ' + delta + 'px, 0)',
					'margin-top': (this.currentOffset - 150) + 'px'
				}
			}
		});

	},


	_updateSuper1: function() {

		if (this.showSuper1Range.isInRange(this.currentScroll)) {

			// showing
			this.currentOpacity1 = this.showSuper1Range.map(this.currentScroll, this.maxOpacity, this.minOpacity);

		} else if (this.hideSuper1Range.isInRange(this.currentScroll)) {

			// hiding
			this.currentOpacity1 = this.hideSuper1Range.map(this.currentScroll, this.minOpacity, this.maxOpacity);

		} else {
			if (this.currentScroll < this.hideSuper1Range.top || this.currentScroll > this.showSuper1Range.bottom) {

				// hidden
				this.currentOpacity1 = this.minOpacity;

			} else if (this.currentScroll > this.hideSuper1Range.bottom && this.currentScroll < this.showSuper1Range.top) {

				// shown
				this.currentOpacity1 = this.maxOpacity;

			}
		}

		this.ui.super1.setProperties({
			'opacity': this.currentOpacity1
		});
	},

	_updateSuper2: function() {

		if (this.showSuper2Range.isInRange(this.currentScroll)) {

			// showing
			this.currentOpacity2 = this.showSuper2Range.map(this.currentScroll, this.maxOpacity, this.minOpacity);

		} else if (this.hideSuper2Range.isInRange(this.currentScroll)) {

			// hiding
			this.currentOpacity2 = this.hideSuper2Range.map(this.currentScroll, this.minOpacity, this.maxOpacity);

		} else {
			if (this.currentScroll < this.hideSuper2Range.top || this.currentScroll > this.showSuper2Range.bottom) {

				// hidden
				this.currentOpacity2 = this.minOpacity;

			} else if (this.currentScroll > this.hideSuper2Range.bottom && this.currentScroll < this.showSuper2Range.top) {

				// shown
				this.currentOpacity2 = this.maxOpacity;

			}
		}

		this.ui.super2.setProperties({
			'opacity': this.currentOpacity2
		});

		// Safari hack: Had to set the 'top' style property directly instead of the library's 'setProperty' method
		this.ui.supers.element.style.top = (((this.maxHeight - this.currentHeight + 12) * this.currentScale) / -1.8) + 'px';
	},



	_hideLoader: function() {
		this.isLoaded = true;
		this.ui.spinner.setProperty('display', 'none');
		this.ui.loader.addTween({
			duration: 0.24,
			finish: {
				opacity: 0
			},
			callback: function(layer) {
				layer.setProperties({
					display: 'none'
				});
			}
		});
	},

	_updateLoader: function() {
		// sets the dimensions of the loader

		if (this.isLoaded) return;

		this.ui.loader.setProperties({
			'width': (this.currentWidth * this.currentScale) + 'px',
			'height': (this.currentHeight * this.currentScale) + 'px',
			'transform-origin': '0 0 0'
		});
	},

	_updateOverlay: function() {
		// sets the dimensions of the clickable overlay

		this.ui.overlay.setProperties({
			'width': (this.currentWidth * this.currentScale) + 'px',
			'height': (this.currentHeight * this.currentScale) + 'px',
			'display': (this.clickableRange.isInRange(this.currentScroll)) ? 'block' : 'none',
			'transform-origin': '0 0 0'
		});
	},


	// NOT CURRENTLY USED
	// method of decoding multiple spritesheets sequentially during the loading process
	/*_decodeImages: function() {
		
		var self = this;

		var timer = setInterval(function() {

			if (self.currentSheet <= self.numSheets) {
				self.ui.decoder.setProperty('background-image', 'url(images/' + MAL.placement.backgroundColor + '/sprites-' + self.currentSheet + '.jpg)');
				self.currentSheet++;
			} else {
				clearInterval(timer);
				setTimeout(function() {
					self._hideLoader();
				}, 1000);
			}

		}, 100);
	},*/

}));

function MALAdReadyCheck () {

	// do we have everything we need?
	if (
		!window.MALSite ||
		!window.MALRMV ||
		!document.getElementById('mal-ad-container') ||
		!window.MALCreative ||
		!window.MAL.connections ||

		(MAL.Environment.browser === 'firefox' && !(window.document.defaultView && window.document.defaultView.getComputedStyle && window.document.defaultView.getComputedStyle(document.getElementById('mal-ad-container'), '')))
	) {
		
		setTimeout(MALAdReadyCheck, 50);
		return;
	}

	// define MAL.placement and MAL.site
	if (!MAL.placement) window.MALSite();


	// supertag setup
	if (MAL.placement && MAL.placement.supertag) {

		MAL.supertag = new MAL.services.Supertag(MAL.placement.supertag);

		// wait until the page is ready
		MAL.supertag.addEventListener('page-ready', function(object, details) {
			console.log('PAGE IS READY');

			console.log('WINDOW SIZE:', details.windowSize);
			console.log('PAGE SCROLL:', details.pageScroll);
			console.log('BANNER RECT:', details.bannerRect);

			
			// set global collapse conditions
			var collapse = (details.windowSize.width < 700);

			if (collapse || MAL.placement.supertag.collapse) {
				
				MAL.supertag.collapse();

			} else {

				window.MALRMV();

				var stageConfig = {dualLayer: true};

				if(
					(MAL.Environment.browser === 'msie' && MAL.Environment.documentMode>9) ||
					((MAL.Environment.osVersion === 5.1 || MAL.Environment.osVersion === 5.2) && MAL.Environment.os === 'windows' && MAL.Environment.browser === 'chrome') ||
					(MAL.Environment.browser==='safari')
				){
					stageConfig.layerType = 'JS';
				}

				MAL.stage = new MAL.tween.Stage(stageConfig);
				MAL.stage.start();

				// debug settings
				MAL.DEBUG = true;
				MAL.Object.prototype.logging = true;
				MAL.Object.prototype.logEvents = false;
				MAL.Object.prototype.logEventsCollapsed = false;
				MAL.Object.prototype.logEventsNoListeners = false;

				// noisey objects
				MAL.video.Base.prototype.logging = false;
				MAL.video.Advanced.prototype.logging = false;
				MAL.tween.Layer.prototype.logging = false;
				MAL.tween.Stage.prototype.logging = false;
				MAL.Preloader.prototype.logging = false;
				MAL.services.MouseHotSpot.prototype.logging = false;


				// safari-specific performance hacks
				if (MAL.Environment.browser === 'safari') {
					MAL.Environment.features.perspectivePerformanceFix = false;
					MAL.Environment.features.backfacePerformanceFix = false;
					MAL.Environment.features.translateZPerformanceFix = false;
				}

				var desktopMinimumBrowserCheck = function() {
					// Check if minimum desktop browser requirements are set.
					var browsers = {
						'safari' : 7,
						'firefox' : 35,
						'chrome' : 36,
						'msie' :  9
					};

					return (((MAL.Environment.os === 'windows' && !navigator.userAgent.match('Tablet')) || MAL.Environment.os === 'mac') && browsers[MAL.Environment.browser] && MAL.Environment.browserVersion >= browsers[MAL.Environment.browser]);
				};


				if (desktopMinimumBrowserCheck()) {
					// load the standard rich media experience
					MAL.ad = new MAL.controllers.RichMedia.Main(details);
				} else {
					// TODO: need add static backup in case we need to bail
				}

			}

		});
	}

}

if (!window.MALAdManualReadyCheck) MALAdReadyCheck();

MAL.define('connections.Default', MAL.Object.extend({
	initialize: function(){
		return this;
	},

	createConnection: function(element, breakpoints, id, index){
		window.addEventListener('resize', this.resize.bind(this));
		this._adContainer = element;
		this._breakpoints = breakpoints;
		this._id = id;
		this._currentState = {
			width: null,
			height: null
		};

		this.resize();
	},

	resize: function(){
		var data = {
			width: document.body.offsetWidth,
			height: document.body.offsetHeight
		};

		for(var type in this._breakpoints){
			var breakpoints = this._breakpoints[type];
			if(breakpoints.length > 0){
				var state = breakpoints.length;
				var points = breakpoints.forEach(function(breakpoint, index){
					if(data[type] <= breakpoint && index < state){
						state = index;
					}
				});
				if(this.updateAdSize && typeof this._currentState[type] !== 'undefined' && this._currentState[type] !== state){
					this._currentState[type] = state;
					this.updateAdSize(state, type);
				}
			}
		}

	},

	setDebugMode: function(mode){
		if(mode === true || mode === false){
			MAL.connections.Default.prototype.logging = mode;
		}
	},

	sendMessageToPub: function(message){
		for (var type in message){
			if(this[type]){
				this[type](message[type]);
			}
		}
	},

	addPageListener: function(target, type, callback){
		var targetName = null;
		if(typeof target === 'string' && window[target]){
			targetName = target;
			target = window[target];
		} else if (typeof target === 'string'){
			targetName = target;
			target = document.querySelector(target);
		} else {
			try{
				targetName = target.toString().match(/^\[object (.+)\]$/)[1].toLowerCase();
			} catch(e){
				MAL.log('Target doesnt exist.');
				return;
			}
		}
		window.addEventListener('custom' + targetName + type, callback, false);
		target.addEventListener(type, this.customEvent.bind(this), false);
	},

	customEvent: function(event){
		var _event = {};
		var _element;
		var _target;
		var _type;
		var _rect = {};

		for(var key in event){
			_event[key] = event[key];
		}

		if(_event['currentTarget']['window']){
			_element = window;
			_target = 'window';
		} else if(_event['currentTarget']['write']){
			_element = document;
			_target = 'document';
		} else{
			_element = document.getElementById(_event.target.id);
			_target = _event.target.id;
		}

		_type = _event['type'];
		_event['currentTarget'] = null;
		_event['srcElement'] = null;
		_event['target'] = null;
		_event['toElement'] = null;
		_event['view'] = null;

		switch(_type){
			case 'resize':
				_event['innerWidth'] = window.innerWidth;
				_event['innerHeight'] = window.innerHeight;
			break;

			case 'scroll':
				_event['scrollX'] = _element.scrollX;
				_event['scrollY'] = _element.scrollY;
			break;
		}

		try{
			for(var key in this._adContainer.getBoundingClientRect()){
				_rect[key] = this._adContainer.getBoundingClientRect()[key];
			}

			_event['rect'] = _rect;
		}catch(error){
			MAL.log(this._id, 'getBoundingClientRect is not supported by this browser');
		}

		try{

			var _custom_event = document.createEvent('Event');
 			_custom_event.initEvent('custom' + _target + _type, true, false);

			for(var key in _event){
				_custom_event[key] = _event[key];
			}

			window.dispatchEvent(_custom_event);
		}catch(error){
			MAL.error(this._id, 'cyclic structure found in listener event object, cannot send event to ad');
			MAL.log(this._id, _custom_event);
		}
	},

	setAdSize: function(message){
		for(var property in message){
			if (property.match(/width|height/)){
				this._adContainer.style[property] = message[property];
			}
		}
	},

	addCustomStyles: function(message){
		for(var key in message){
			this._adContainer.style[key] = message[key];
		}
	},

	addDirectStyles: function(object){
		for(var key in object){
			MAL.toArray(document.querySelectorAll(key)).forEach(function(element){
				for(var property in object[key]){
					element.style[property] = object[key][property];
				}
			});
		}
	},

	getPubPageInfo: function(){
		var rect = this._adContainer.getBoundingClientRect();
		var info = {
			window: {height: window.innerHeight, width: window.innerWidth},
			body: {width: document.body.offsetWidth, height: document.body.offsetHeight},
			scroll: {x: window.pageXOffset || document.documentElement.scrollLeft, y: window.pageYOffset || document.documentElement.scrollTop},
			rect: {
				top: rect.top,
				left: rect.left,
				right: rect.right,
				bottom: rect.bottom,
				width: rect.width,
				height: rect.height
			}
		};

		info.iframe = info.rect || {};

		return info;
	},

	closeAd: function(){
		window.removeEventListener('resize', this.handleResize);
		this._adContainer.parentNode.removeChild(this._adContainer);
	}


}));