
/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * @class MAL.Environment
 *
 * User Environment class. Provides an easy way to do browser and feature detection.
 *
 * # Browser detection
 *
 * To detect a users browser you would do this
 *
 *     if (MAL.Environment.browser == 'firefox') {
 *      // browser is firefox
 *     } else {
 *      // browser is not firefox
 *     }
 *
 * A common thing you may need to check for is if the browser is IE and is not running in a lower compatibility mode
 *
 *     if (MAL.Environment.browser == 'msie' && MAL.Environment.documentMode < 9) {
 *      // browser is IE and can not do newer IE features
 *     } else {
 *      // can do ecma5 related stuff
 *     }
 *
 * # Global bugs and features
 *
 * we have a list of {@link #bugs} and {@link #features} that you can toggle
 *
 * for example if you wanted to disable all 3d transformations
 *     MAL.Environment.bugs.transform3d = true; // disabling all 3d transformations
 *
 * or if backfaces on all transition elements are causing issues you can do this
 *     MAL.Environment.features.backfacePerformanceFix = false; // disable auto backface
 *
 * @singleton
 */
MAL.define('Environment', (function () {
	function create (vendor, platform, userAgent) {
		function browserSearchString (data) {
			for (var i=0;i<data.length;i++)	{
				var string = data[i].string,
					property = data[i].property;

				versionSearchString = data[i].versionSearch || data[i].identity;

				if (string && string.indexOf(data[i].subString) !== -1 || property)
					return data[i].identity;
			}
			return '';
		}

		function browserSearchVersion (dataString) {
			var index = dataString.indexOf(versionSearchString);
			if (index === -1) return;
			return parseFloat(dataString.substring(index+versionSearchString.length+1));
		}

		function osSearchVersion (dataString, os) {
			for (var i = 0; i < data.os.length; i++) {
				if (os.indexOf(data.os[i].subString) === -1) continue;
				if (data.os[i].version) {

					return data.os[i].version(dataString);
				}
			}

			return '';
		}

		//<exclude=localStorage>
		function checkLocalStorage(browser) {
			/*
			 * Temporaray hack for IE on Surface. We needed a quick solution. It looks like in
			 * Metro mode, the browser has a bug that doesn't allow localStorage to be modified
			 * or read. A more permanent solution is to include the if statement in the try/catch.
			 */
			if (browser && browser=== 'msie' && navigator.userAgent.match(/Touch/)) {
				return false;
			}
			if ('localStorage' in window && window['localStorage'] !== null) {
				try {
					var test = 'localtest-' + Math.round(Math.random()*1000);
					localStorage.setItem(test,1);
					localStorage.removeItem(test);
					return true;
				} catch(e) {}
			}
			return false;
		}
		//</exclude=localStorage>
		
		function getVendorPrefix() {
			var regex = new RegExp('^(' + data.prefixes.join('|') + ')(?=[A-Z])');

			// latest browsers should support this
			for(var property in element.style)
				if (regex.test(property))
					return property.match(regex)[0];

			// fallback incase we need to check a specific property
			for (var i in data.prefixes) {
				var prefix = data.prefixes[i];
				if (prefix + 'Opacity' in element.style)
					return prefix;
			}

			return '';
		}

		var CSSPropertyNameCache = {normal: {}, camel: {}};

		function getCSSPropertyName (property, doNotCamelCase) {
			var cache = doNotCamelCase ? 'normal' : 'camel';

			// return if we have already cached the property
			if (CSSPropertyNameCache[cache][property] !== undefined) {
				return CSSPropertyNameCache[cache][property];
			}

			// HACK Firefox: float returns false
			if (property === 'float') return CSSPropertyNameCache[cache][property] = property;

			// HACK Firefox: src returns false
			if (property === 'src') return CSSPropertyNameCache[cache][property] = property;

			// HACK Webkit: issues with non prefixed filter returning a false positive
			if (property === 'filter' && engine === 'webkit') {
				return CSSPropertyNameCache[cache][property] = getCSSPropertyName('-' + property, doNotCamelCase);
			}

			var propertyCamelCase = MAL.toCamelCase(property);

			if (property.substr(0, 1) !== '-' && propertyCamelCase in element.style) {
				return CSSPropertyNameCache[cache][property] = property;
			} else {
				var propertyName = property;

				if (propertyName.substr(0, 1) === '-') {
					propertyName = propertyName.substr(1);
				}

				propertyCamelCase = MAL.toCamelCaseWithPrefix(prefix, propertyName);

				if (propertyCamelCase in element.style) {
					return CSSPropertyNameCache[cache][property] = doNotCamelCase ? '-' + prefix.toLowerCase() + '-' + propertyName : propertyCamelCase;
				}
			}

			return CSSPropertyNameCache[cache][property] = false;
		}

		function appleOSVersionMatch (string) {
			var matches = string.match(/(?:OS|Mac[^;]+) ([0-9_.]+)/);
			return matches ? parseFloat(matches[1].split('_').slice(0,2).join('.')) : '';
		}

		var versionSearchString = '',
			element = document.createElement('div'),
			data = {
				prefixes: 'Webkit Moz O ms'.split(' '),
				engines: {
					applewebkit: 'webkit',
					safari: 'webkit',
					chrome: 'webkit',
					msie: 'trident',
					firefox: 'gecko',
					opera: 'webkit',
					opera_legacy: 'presto',
					adobeair: 'webkit'
				},
				browser: [
					{
						string: userAgent,
						subString: 'MSIE',
						identity: 'MSIE',
						versionSearch: 'MSIE'
					},
					{
						string: userAgent,
						subString: 'Trident',
						identity: 'Trident',
						versionSearch: 'Trident'
					},
					{
						string: userAgent,
						subString: 'OPR/',
						identity: 'Opera',
						versionSearch: 'OPR'
					},
					{
						string: userAgent,
						subString: 'Chrome',
						identity: 'Chrome'
					},
					{
						string: userAgent,
						subString: 'Opera',
						identity: 'opera_legacy',
						versionSearch: 'Version'
					},
					{
						string: userAgent,
						subString: 'AdobeAIR',
						identity: 'AdobeAIR'
					},
					{
						string: vendor,
						subString: 'Apple',
						identity: 'Safari',
						versionSearch: 'Version'
					},
					{
						string: userAgent,
						subString: 'Firefox',
						identity: 'Firefox'
					},
					{
						string: userAgent,
						subString: 'AppleWebKit',
						identity: 'AppleWebKit'
					},
					{
						string: userAgent,
						subString: 'AppleWebkit',
						identity: 'AppleWebkit'
					}
				],
				os: [
					{
						string: platform,
						subString: 'Win',
						identity: 'Windows',
						version: function (string) {
							var matches = string.match(/Windows NT ([0-9.]+)/i);
							return matches ? parseFloat(matches[1]) : '';
						}
					},
					{
						string: platform,
						subString: 'Mac',
						identity: 'Mac',
						version: appleOSVersionMatch
					},
					{
						string: platform,
						subString: 'Android',
						identity: 'Android',
						version: function (string) {
							var matches = string.match(/Android ([0-9.]+)/i);
							return matches ? matches[1] : '';
						}
					},
					{
						string: userAgent,
						subString: 'iPod',
						identity: 'iPod',
						version: appleOSVersionMatch
					},
					{
						string: userAgent,
						subString: 'iPhone',
						identity: 'iPhone',
						version: appleOSVersionMatch
					},
					{
						string: userAgent,
						subString: 'iPad',
						identity: 'iPad',
						version: appleOSVersionMatch
					},
					{
						string: platform,
						subString: 'Linux',
						identity: 'Linux'
					}
				],
				transitionEndEventNames: {
					webkit: 'webkitTransitionEnd',
					gecko: 'transitionend',
					presto: 'oTransitionEnd',
					trident: 'transitionend'
				},
				animationEndEventNames: {
					webkit: 'webkitAnimationEnd',
					gecko: 'animationend',
					presto: 'oAnimationEnd',
					trident: 'animationend'
				}
			},
			browser = browserSearchString(data.browser).toLowerCase(),
			browserVersion = browserSearchVersion(userAgent);

		// IE11+ pretends not to be IE
		if (browser === 'trident') {
			browser = 'msie';

			// assumes that they will bump the trident version every time there is a new IE version (this has worked since IE7)
			browserVersion += 4;
		}

		var	os = browserSearchString(data.os),
			osVersion = osSearchVersion(userAgent, os),
			prefix = getVendorPrefix(),
			engine = data.engines[browser],
			transitionEnd = data.transitionEndEventNames[engine],
			animationEnd = data.animationEndEventNames[engine],
			documentMode,
			mobile = false;

		// mobile os check
		if (userAgent.match(/\bMobile\b/)) mobile = true;
		if (os === 'Android') mobile = true;
		if (browser === 'opera_legacy') browser = 'opera';

		var result = {
			/**
			* @property {String} browser
			*
			* - safari
			* - chrome
			* - msie
			* - firefox
			* - opera
			*/
			browser: browser,

			/**
			* @property {String} engine
			*
			* - webkit
			* - trident
			* - gecko
			* - presto
			*/
			engine: engine,

			/**
			* @property {Boolean} mobile
			*/
			mobile: mobile,

			/**
			* @property {Number} browserVersion
			*
			* float value of the browser version
			*/
			browserVersion: browserVersion,

			/**
			* @property {String} os
			*
			* - windows
			* - mac
			* - ipod
			* - iphone
			* - ipad
			* - linux
			*/
			os: os.toLowerCase(),

			/**
			* @property {String} osVersion
			*
			* usually a numerical representation of the os version
			*/
			osVersion: osVersion,

			/**
			* @property {String} prefix
			*
			* - -webkit-
			* - -moz-
			* - -o-
			* - -ms-
			*/
			prefix: '-' + prefix.toLowerCase() + '-',

			/**
			* @property {String} domPrefix
			*
			* - Webkit
			* - Moz
			* - O
			* - ms
			*/
			domPrefix: prefix,

			/**
			* returns the correct property name if the browser supports it
			*
			*     if (MAL.Environment.getCSSPropertyName('transform')) {
			*      // logs out the browser prefixed transform css property (if its still prefixed)
			*      console.log(MAL.Environment.getCSSPropertyName('transform'));
			*     } else {
			*      // browser does not support css transformations
			*     }
			*
			* @param {String} property
			* @param {String} [doNotCamelCase] use if you want dashed based properties
			* @return {String/Boolean} false or the property
			*/
			getCSSPropertyName: getCSSPropertyName,

			/**
			* @property {String/Boolean} requestAnimationFrame
			*
			* provides the correct name of the browser's requestAnimationFrame method, or false if the browser doesn't support the method
			*/
			requestAnimationFrame: window[prefix.toLowerCase() + 'RequestAnimationFrame'] ? prefix.toLowerCase() + 'RequestAnimationFrame' : false,

			/**
			* @property {Boolean} has3d
			*
			* informs you if the browser supports 3d transformations
			*/
			has3d: getCSSPropertyName('perspective') && (engine !== 'webkit' || !window.matchMedia || window.matchMedia && window.matchMedia('(transform-3d),(-' + prefix.toLowerCase() + '-transform-3d)').matches),

			/**
			* @property {Boolean} hasCanvas
			*
			* informs you if the browser supports canvas
			*/
			hasCanvas: !!document.createElement('canvas').getContext,

			//<exclude=localStorage>
			/**
			* @property {Boolean} hasLocalStorage
			*
			* informs you if the browser supports localStorage
			*/
			hasLocalStorage: checkLocalStorage(browser),
			//</exclude=localStorage>
			
			/**
			* @property {Boolean} hasKeyframeAnimations
			*
			* informs you if the browser supports css keyframe animations
			*/
			hasKeyframeAnimations: !!getCSSPropertyName('animation-name'),

			/**
			* @property {Object} bugs namespace helper for maintaining browser bugs
			*
			* - transformations
			* - transform3d
			* - scale
			*/
			bugs: {
				transformations: false,
				transform3d: !!(browser === 'safari' && userAgent.match(/Version\/(4.0 Safari|4.0.1|4.0.2|4.0.5)/)),
				forceWebm: browser === 'firefox',
				scale: false // (browser == 'firefox' && browserVersion < 7) || (browser == 'msie' && browserVersion == 9)
			},

			/**
			* @property {Object} features namespace allows you to enable/disable features
			*
			* - backfacePerformanceFix
			* - perspectivePerformanceFix
			* - translateZPerformanceFix
			*/
			features: {
				// adds rotation to elements in firefox and ie9 to smooth animation
				//rotationPerformanceFix: false,

				// adds performance fixes relating to css backfaces, you need to disable this when you need to use backfaces and tweening
				backfacePerformanceFix: true,

				// adds performance fixes relating to css aliasing, you need to disable this when you need to use 3d perspective and tweening
				perspectivePerformanceFix: true,

				// adds translateZ to all css animations
				translateZPerformanceFix: true
			}
		};

		/**
		* @property {String} transitionEnd
		*
		* provides the correct name of the browser's transitionEnd event
		*/
		if (getCSSPropertyName('transition')) result.transitionEnd = transitionEnd;

		/**
		* @property {String} animationEnd
		*
		* provides the correct name of the browser's animationEnd event
		*/
		if (getCSSPropertyName('animation')) result.animationEnd = animationEnd;

		/**
		* @property {String} documentMode
		*
		* provides you with a consistent way to figure out what documentMode IE is in
		*/
		if (browser === 'msie') {
			if (document.documentMode) {
				// IE8 or later
				result.documentMode = document.documentMode;
			} else {
				// IE 5-7
				result.documentMode = document.compatMode && document.compatMode === 'CSS1Compat' ? 7 : 5;
			}
		}
		result.create = create;

		return result;
	}

	return create(navigator.vendor, navigator.platform, navigator.userAgent);
})());