/**
 * @author Chad Scira <chad@mediaartslab.com>, Martijn Korteweg <martijn@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Keyframe Animation class. Provides an really useful timer based tool that is usually used for keyframe animations, but can be used for other things.
 *
 * # Usage
 *
 * auto calculated timing based on time/frames
 *
 *     new MAL.KeyframeAnimation({
 *       delay: 1,
 *       time: 10,
 *       frames: 20
 *      },
 *      function () {
 *       // this.frame
 *      }
 *     );
 *
 * Specific timing on frame 1:
 *
 *     new MAL.KeyframeAnimation({
 *       time: {defaultTime: .5, '1': .75},
 *       frames: 20
 *      },
 *      function () {
 *       // this.frame
 *      }
 *     );
 *
 * @class MAL.KeyframeAnimation
 * @extends MAL.Object
 */
MAL.define('KeyframeAnimation', MAL.Object.extend({
	/**
	 * initialize
	 *
	 *     new MAL.KeyframeAnimation({
	 *       delay: 1,
	 *       time: 10,
	 *       frames: 20
	 *      },
	 *      function () {
	 *       // this.frame
	 *      }
	 *     );
	 *
	 * @method constructor
	 * @param {Object} config
	 * @param {Function} callback
	 */

	/**
	* @cfg {Object/Array} time
	*
	* controls the frame timing
	*
	* default time
	*     {defaultTime: .5}
	* default time and specific frame timing
	*     {defaultTime: .5, '23': .2}
	* specific timing for every frame (array length must not be less than the amount of frames)
	*     [.1, .2., .3, .4]
	*
	*/
	/**
	* @cfg {Number} delay
	*
	* event loop delay in milliseconds
	*/
	/**
	* @cfg {Number} frames
	*
	* amount of frames
	*/
	/**
	* @cfg {Number} startFrame
	*
	* frame to start animation on
	*/
	/**
	* @cfg {Boolean} autoplay
	*
	* autoplay the animation upon initialization
	*/
	/**
	* @cfg {Boolean} loop
	*
	* loops animation
	*/
	initialize: function ($config, $callback) {
		this._super();

		// set private vars
		this.options = $config;
		this.callback = $callback;
		this.frame = 0;
		this.totalTime = 0;
		this.nextTime = 0;
		this.frameTime = 0;
		this.autoplay = true;
		this._timeout = null;

		// throw error if we do not have the required options set
		if (!this.options || !this.callback || !this.options.frames || !this.options.time) MAL.error('[MAL.KeyframeAnimation] InvalidInitialization');

		// if there is a start frame and set the nessary info
		if (!MAL.isUndefined(this.options.startFrame)) {
			this.frame = this.options.startFrame;
			this.options.frames = this.options.frames+this.options.startFrame;
		}

		// make frames 1 based
		this.options.frames--;

		// stop here if its not suppose to play
		if (this.options.autoplay === false) return;

		this.play();

	},

	/**
	 * start animation
	 *
	 *     keyframeAnimation.start();
	 *
	 */
	play: function () {
		// fire delayed loop
		if (this.options.delay) {
			setTimeout(this._delayedStart, this.options.delay * 1000);
		} else {
			this._delayedStart();
		}
	},

	/**
	 * stops animation
	 *
	 *     keyframeAnimation.stop();
	 *
	 */
	stop: function () {
		clearTimeout(this._timeout);
	},

	/**
	 * replay the animation
	 *
	 *     keyframeAnimation.replay({delay: 1});
	 *
	 */
	replay: function ($config) {

		if (typeof $config !== 'undefined')
			this.options.delay = MAL.isUndefined($config.delay) ? this.options.delay : $config.delay;

		// reset frame
		this.frame = 0;

		// start the animation loop
		this.play();
	},

	/**
	 * delays the start of the loop
	 *
	 * @private
	 */
	_delayedStart: function () {
		// set the nessary timestamps
		this.startTime = this.timeSpend = new Date().getTime();

		// start the animation loop
		this._loop();
	},

	/**
	 * recursive loop function that adjusts the delay based on real time spent
	 *
	 * @private
	 */
	_loop: function () {
		// check if we are finished
        if (this.frame > this.options.frames) {
          if (this.options.loop) {
            this.frame = 0;
          } else {
            return;
          }
        }

		// get the current time
		var dateNow = new Date().getTime();

		// fire callback
		this.callback.apply(this);

		var time = this.options.time,
			frameTime = this._getFrameTime();

		// time that should be spent
		this.frameTime += frameTime;

		// time actually spent
		this.timeSpend = (dateNow-this.timeSpend)/1000;

		// keep track of the total time spent
		this.totalTime += this.timeSpend;

		// incriment frame
		this.frame++;

		// calculate when the next frame should be
		this.nextTime = this.frameTime;
		// figure out the difference
		var tempNext = this.nextTime-this.totalTime;

		// reset timeSpend
		this.timeSpend = dateNow;

		// fire recursive loop
		this._timeout = setTimeout(this._loop, tempNext * 1000);
	},

	/**
	 * figures out what the next delay should be
	 *
	 * @private
	 */
	_getFrameTime: function() {
		var frameTime;
		if (MAL.isNumber(this.options.time)) {
			// time will be divided upon the total keyframes
			frameTime = this.options.time/this.options.frames;
		} else if (MAL.isArray(this.options.time)) {
			// time is explicitly set for each and every keyframe
			frameTime = this.options.time[this.frame-1];
		} else if (MAL.isObject(this.options.time)) {
			// time is an object that provides a default time
			frameTime = !MAL.isUndefined(this.options.time[this.frame]) ? this.options.time[this.frame] : this.options.time.defaultTime;
		}

		return frameTime;
	}
}));