/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Keyframe Incremental Interval class. Provides a easy way to implement forward and backwards keyframe animation.
 *
 * # Usage
 *
 * first implement a keyframe incremental instance (it will automatically start its timer)
 *
 *     var keyframeIncrementalInterval = new MAL.KeyframeIncrementalInterval({
 *     	interval: 0.04170,
 *     	min: 0,
 *     	max: 28,
 *     	frame: 28,
 *     	incremental: 1,
 *     	callback: function () {
 *     		$('Sprite').setProperty('background-position', (this.frame * -55) + 'px 0px');
 *     	}
 *     });
 *
 * it will automatically start heading towards the max or min based on the incremental, you can change the incremental at any time like this
 *
 *     keyframeIncrementalInterval.incremental = -1;
 *
 * now it will go backwards to the min, this is usually used in the mouseover/mouseout events but can be used for other things
 *
 * @class MAL.KeyframeIncrementalInterval
 * @extends MAL.Object
 */
MAL.define('KeyframeIncrementalInterval', MAL.Object.extend({
	/**
	 * initialize
	 *
	 *     var keyframeIncrementalInterval = new MAL.KeyframeIncrementalInterval({
	 *     	interval: 0.04170,
	 *     	min: 0,
	 *     	max: 28,
	 *     	frame: 28,
	 *     	incremental: 1,
	 *     	callback: function () {
	 *     		$('Sprite').setProperty('background-position', (this.frame * -55) + 'px 0px');
	 *     	}
	 *     });
	 *
	 * @method constructor
	 * @param {Object} config
	 */

	/**
	* @cfg {Number} frame
	*
	* frame to start on
	*/

	/**
	* @cfg {Number} incremental
	*
	* amount to increment
	*/

	/**
	* @cfg {Number} min
	*
	* min keyframe
	*/

	/**
	* @cfg {Number} max
	*
	* max keyframe
	*/

	/**
	* @cfg {Number} delay
	*
	* amount of time to delay before the animation starts
	*/

	/**
	* @cfg {Function} callback
	*
	* iteration callback function
	*/
	initialize: function ($options) {
		this._super();

		// setup defaults
		$options = MAL.mergeOptions({
			frame: 0,
			incremental: 1,
			interval: 1,
			min: 0,
			max: 0,
			delay: 0,
			callback: null
		}, $options);

		// frame to start on
		this.frame = $options.frame;

		// amount to increment
		this.incremental = $options.incremental;

		// interval between callbacks
		this._interval = $options.interval * 1000;

		// min keyframe
		this._min = $options.min;

		// max keyframe
		this._max = $options.max;

		// animation callback
		this._callback = $options.callback;

		// amount of time to delay before the animation starts
		this._delay = $options.delay * 1000;

		// start loop
		if (this._delay) {
			setTimeout(this._loop, this._delay);
		} else {
			this._loop();
		}
	},

	/**
	 * recursive loop
	 *
	 * @private
	 */
	_loop: function () {
		// take a snapshot of the previous frame
		var previousFrame = this.frame;

		// increment frame by incremental
		this.frame += this.incremental;

		// normalize the frame number
		if (this.frame < this._min) {
			this.frame = this._min;
		} else if (this._max && this.frame > this._max) {
			this.frame = this._max;
		}

		// fire callback if we have a new frame
		if (this.frame != previousFrame && this._callback) this._callback.apply(this);

		// continue with loop
		setTimeout(this._loop, this._interval);
	}
}));