/**
 * @author Chad Scira <chad@mediaartslab.com>, Steve Cox <Steve.Cox@eu.mediaartslab.com>, Martijn Korteweg <martijn@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Object class. Provides a standard base object with custom event support.
 *
 * # Event Usage
 *
 *     // instantiate a extended Object
 *     var Person = MAL.Object.extend({
 *      initialize: function () {
 *       // call supers initialization method
 *       this._super();
 *      },
 *
 *      poke: function () {
 *       // dispatch poke event to listeners
 *       this.dispatchEvent('poke');
 *      }
 *     });
 *
 *     // instantiate person
 *     var person = new Person();
 *
 *     // subscribe to poke events
 *     person.addEventListener('poke', function () {
 *      MAL.log('poked');
 *     });
 *
 *     // poke the person
 *     person.poke();
 *
 * @class MAL.Object
 */
MAL.define('Object', MAL.Util.create({
	/**
	 * initialize
	 *
	 *     var Person = MAL.Object.extend({
	 *         // requires an initialize method
	 *         initialize: function () {
	 *             // call supers initialization method
	 *             this._super();
	 *         }
	 *     };
	 *
	 * @method constructor
	 */
	initialize: function () {
		this._listeners = {};
	},

	/**
	 * enable/disable all log calls to the object
	 */
	logging: false,

	/**
	 * enable/disable on any object to show event logging
	 */
	logEvents: false,

	/**
	 * enable/disable on any object to show event logging
	 */
	logEventsCollapsed: true,

	/**
	 * array of events to not log
	 */
	logEventsIgnore: [],

	/**
	 * enable/disable on any object event logging for events with no listeners
	 */
	logEventsNoListeners: true,

	/**
	 * object log method
	 * @return {Object...} message
	 */
	log: function () {
		if (!this.logging) return;

		if (MAL.Environment.browser === 'chrome') {
			console.log.apply(console, ['%c' + this.getFileAndLineNumberAtStackIndex(2) + '%c ' + this.objectClassPathToken + '%c', 'color: #de7400;', 'font-weight: bold', 'font-weight: normal'].concat(MAL.argumentsToArray(arguments)));
		} else {
			MAL.log.apply(MAL, ['Controller', '"' + this.objectClassPathToken + '"'].concat(MAL.argumentsToArray(arguments)));
		}
	},

	/**
	 * add an event listener to the object
	 *
	 *     this.addEventListener('poke', pokeFunction);
	 *
	 * @param {String} type
	 * @param {Function} listener
	 * @inheritable
	 */
	addEventListener: function(type, listener) {
		if (!MAL.isFunction(listener)) throw new Error ('the listener you tried to add for "' + (this.objectClassPathToken + ':' + type) + '" is not a function');

		type = this._prefixTypeWithPathToken(type);

		if (MAL.isUndefined(this._listeners[type])) {
			this._listeners[type] = [];
		}

		this._listeners[type].push(listener);
	},

	/**
	 * add multiple event listeners to the object
	 *
	 *     this.addEventListeners({poke: pokeFunction, punch: punchFunction});
	 *
	 * @param {Object} listeners
	 * @inheritable
	 */
	addEventListeners: function(listeners) {
		for (var type in listeners) this.addEventListener(type, listeners[type]);
	},

	/**
	 * remove an event listener from the object. if the function needs the scope binded this is a pain, and you may want to use {@link #removeAllEventListeners}
	 *
	 *     // you must pass in the original function (bind makes this a little less ideal)
	 *     this.removeEventListener('poke', originalFunction);
	 *
	 * all events are prefixed with their objects objectClassPathToken
	 *
	 * @param {String} type
	 * @param {Function} listener
	 * @inheritable
	 */
	removeEventListener: function(type, listener) {
		type = this._prefixTypeWithPathToken(type);

		if (MAL.isArray(this._listeners[type])){
			var listeners = this._listeners[type];
			for (var i=0, l=listeners.length; i < l; i++){

				if (listeners[i] === listener){
					listeners.splice(i, 1);
					break;
				}
			}
		}
	},

	/**
	 * remove an event listener of a specific type or all listeners in general from the object
	 *
	 *     // remove all subscribers of the 'poke' event
	 *     person.removeAllEventListeners('poke');
	 *
	 *     // remove all subscribers for all events
	 *     person.removeAllEventListeners();
	 *
	 * @param {String} type
	 * @inheritable
	 */
	removeAllEventListeners: function(type) {
		if (type) {
			delete this._listeners[type];
		} else {
			this._listeners = {};
		}
	},

	/**
	 * dispatches event to all listeners
	 *
	 *     // dispatch a basic event
	 *     this.dispatchEvent('poke');
	 *
	 *     // dispatch an event with details merged into the event object
	 *     this.dispatchEvent('poke', {enabled: true});
	 *
	 *     // dispatch an event with details and a original event object
	 *     this.dispatchEvent('poke', {enabled: true}, e);
	 *
	 * @param {String} event
	 * @param {Object} details
	 * @param {Object} originalEvent
	 * @inheritable
	 */
	dispatchEvent: function(event, details, originalEvent, proxy) {
		var loggingGroup;

		event = typeof event === 'string' ? { type: event } : event;
		event.type = this._prefixTypeWithPathToken(event.type);

		if (this.logging && this.logEvents && this.logEventsIgnore.indexOf(event.type) === -1 && (!(event.type.indexOf(':') && (this.logEventsIgnore.indexOf(event.type.split(':')[1]) !== -1)))) {
			if (proxy) {
				if (MAL.Environment.browser === 'chrome') {
					console.log('%cProxiedEvent%c ' + proxy.replace('controllers.', '') + ' %csent%c ' + event.type + ' %cto%c ' + this.objectClassPathToken.replace('controllers.', ''), 'color: #00993d;', 'font-weight: bold', 'font-weight: normal', 'font-weight: bold', 'font-weight: normal', 'font-weight: bold', details || '');
				} else {
					MAL.log('ProxiedEvent (' + proxy.replace('controllers.', '') + ') sent (' + event.type + ') to (' + this.objectClassPathToken.replace('controllers.', '') + ')');
				}
			} else {

				if (MAL.Environment.browser === 'chrome') {
					loggingGroup = ['%c' + (this.getFileAndLineNumberAtStackIndex(2) || 'Event') + '%c dispatched%c ' + event.type , 'color: #00993d;', 'color: #000; font-weight: normal;', 'color: #000; font-weight: bold;', details || ''];
				} else {
					MAL.log('Event (' + event.type + ')');
				}
			}
		}

		event.target = (!event.target) ? this : event.target;
		event.timestamp = new Date().getTime();
		event.details = details;
		if (!event.type){
			throw new Error('missing \'type\' property');
		}

		// attach original event
		if (typeof originalEvent !== 'undefined') event.originalEvent = originalEvent;

		// add details to event
		if (typeof details !== 'undefined') for (var detail in details) event[detail] = details[detail];

		this._callListeners(event, loggingGroup);
	},

	getFileAndLineNumberAtStackIndex: function (index) {
		var stack = new Error().stack,
			paths = stack.match(/\([^()]+\)/g),
			path;

		for (var i = 0; i < paths.length; i++) {
			path = paths[i];
			path = path.replace(/\(.*\/(.*)\)$/, '$1');
			path = path.replace(/\:\d*$/, '');

			if (i === index) {
				return path;
			}
		}
	},

	/**
	 * choose events to proxy to a specific object or array of objects. proxied events will trigger listeners on objects you are proxying to.
	 *
	 *     // proxy event to object
	 *     this.proxyEvents('poke', otherObject);
	 *     otherObject.addEventListener('poke', function);
	 *
	 * @param {Array/Object} events
	 * @param {Array/Object} targets
	 */
	proxyEvents: function (events, targets) {
		events = MAL.isArray(events) ? events : [events];
		targets = MAL.isArray(targets) ? targets : [targets];

		var self = this;

		events.forEach(function (event) {
			if (event.type) {
				event.type = self._prefixTypeWithPathToken(event.type);
			} else {
				event = self._prefixTypeWithPathToken(event);
			}

			this.addEventListener(event.type || event, function (object, event, details) {
				targets.forEach(function (target) {
					target.dispatchEvent.call(target, event.type, details, event, self.objectClassPathToken);
				});
			});
		}, this);
	},

	/**
	 * @private
	 */
	_prefixTypeWithPathToken: function (type) {
		type = type.match(':') ? type : this.objectClassPathToken + ':' + type;
		type = type.replace('controllers.', '');
		return type;
	},

	/**
	 * @private
	 */
	_callListeners: function(event, loggingGroup) {
		if (this._listeners[event.type] instanceof Array){

			if (loggingGroup) console[this.logEventsCollapsed ? 'groupCollapsed' : 'group'].apply(console, loggingGroup);

			var listeners = this._listeners[event.type];
			for (var i=0, l=listeners.length; i < l; i++){
				var details = event.details;
				delete event.details;

				try {
					listeners[i](this, event, details);
				} catch (error) {
					if (MAL.Environment.browser === 'chrome') {
						console.warn('%cerror encountered when throwing (' + event.type + ')\n\n%c' + error.stack, 'color: #ff0000;', 'color: #bcbab3');
					} else {
						if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
							MAL.log('encountered (' + error.message + ') when throwing (' + event.type + ')');
						} else {
							MAL.error('error encountered when throwing (' + event.type + ')\n\n' + error.stack);
						}
					}
				}
			}

			if (loggingGroup) console.groupEnd.apply(console, loggingGroup);
		} else {
			if (loggingGroup && this.logEventsNoListeners) {
				loggingGroup[1] = 'color: #e10000';
				console.log.apply(console, loggingGroup);
			}
		}
	}
}));