/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * @class MAL.Preloader
 * @extends MAL.Object
 *
 * Preloader class. Provides a way to preload a set of images.
 *
 * # Usage Example
 *
 * initialize preloader with an array of images
 *
 *     var preloader = new MAL.Preloader({
 *      // array of initial images
 *      images: ['one.png', 'two.png', 'three.png'],
 *
 *      // path for all images
 *      imagePath: 'images/',
 *
 *      // file path proxy method (required for google)
 *      filePathProxy: function (filepath, filename) {
 *       // every file request goes through this method, this is where you would add additional filepath logic
 *       return filepath + filename;
 *      }
 *     });
 *
 * add additional images after the preloader has been initialized (but before it has been instructed to start loading)
 *
 *     preloader.addImages(['four.png', 'five.png', 'six.png']);
 *
 * subscribe to the preloaders completion event
 *
 *     preloader.addEventListener('load', function (o,e) {
 *      console.log('all images have been loaded');
 *     });
 *
 * start preloading
 *
 *     preloader.load();
 *
 */

/**
 * dispatched when loading starts
 *
 * @event start
 * @param {MAL.Preloader} this
 */

/**
 * dispatched when an image has loaded starts
 *
 * @event image-loaded
 * @param {MAL.Preloader} this
 * @param {Object} image
 */

/**
 * dispatched when all the images are loaded
 *
 * @event load
 * @param {MAL.Preloader} this
 */
MAL.define('Preloader', MAL.Object.extend({
	/**
	 * initializes MAL.Preloader
	 *
	 *     var preloader = new MAL.Preloader({
	 *      // array of initial images
	 *      images: ['one.png', 'two.png', 'three.png'],
	 *
	 *      // path for all images
	 *      imagePath: 'images/',
	 *
	 *      // file path proxy method (required for google)
	 *      filePathProxy: function (filepath, filename) {
	 *       // every file request goes through this method, this is where you would add additional filepath logic
	 *       return filepath + filename;
	 *      }
	 *     });
	 *
	 * @method constructor
	 * @param {Object} $config
	 */

	/**
	 * @cfg {Function} filePathProxy
	 * event loop delay in milliseconds
	 */

	/**
	 * @cfg {String} imagePath
	 * event loop delay in milliseconds
	 */

	/**
	 * @cfg {Function} callback
	 *
	 * callback function that should be called upon completion
	 */

	/**
	 * @cfg {Boolean} autoload
	 *
	 * autoload the images upon initialization
	 */
	initialize: function ($config) {
		$config = $config || {};

		this._super();

		this._images = new MAL.Set();

		if ($config.imagePath) this._imagePath = $config.imagePath;
		if ($config.filePathProxy) this._filePathProxy = $config.filePathProxy;

		if (typeof MAL.rmv !== 'undefined') {
			this._imagePath = MAL.rmv.paths.images;
			this._filePathProxy = MAL.rmv.filePathProxy;
		}

		this._callback = $config.callback;

		// garbage collection fix for IE7?
		this.images = {};

		// add initial images if passed
		if ($config.images) this.addImages($config.images);
		if ($config.autoload) this.load();
	},

	/**
	 * image path
	 * @private
	 */
	_imagePath: '',

	/**
	 * image path resolution
	 *
	 * @private
	 * @param {String} filepath
	 * @param {String} filename
	 * @return {String} resolved filepath
	 */
	_filePathProxy: function (filepath, filename) {
		return filepath + filename;
	},

	/**
	 * Adds a set of images to the Preloader
	 *
	 *     preloader.addImages(['four.png', 'five.png', 'six.png']);
	 *
	 * @param {Array} images
	 */
	addImages: function (images) {
		images.forEach(function(value, key, ref) {
			this.addImage(value);
		}, this);
	},

	/**
	 * Adds a image to the Preloader
	 *
	 *     preloader.addImage('four.png');
	 *
	 * @param {String} image
	 */
	addImage: function (image) {
		this._images.add(image);
	},

	/**
	 * Starts the load process
	 *
	 *     preloader.load();
	 *
	 */
	load: function () {
		if (!this._images.length) {
			MAL.log('WARNING: empty image preloader');
			this._onImagesLoaded();
			return;
		}

		var self = this;

		this.dispatchEvent('start');

		MAL.Async.parallel(
			this._images.map(function(value, key, array) {
				var src = this._filePathProxy(this._imagePath, value);

				return function (callback) {
					var error,
						cache = new Image();

					if (cache.addEventListener) {
						// toss the image info in a closure
						error = (function (self, cache) {
							return function () {
								self._onImageError.apply(self, [cache]);
							};
						}(this, cache));
					} else {
						error = this._onImageError;
					}

					MAL.addEvents(cache, {
						load: function () {
							// DO NOT THROW LOAD EVENTS FOR DATA URI's
							if (cache.src.substr(0, 4) === 'data') return false;

							self.dispatchEvent('image-loaded', {image: cache});
							callback();
						},
						error: error,
						abort: error
					});

					// start loading
					cache.src = src;

					// hold image ref
					this.images[value] = cache;
				}.bind(this);
			}, this),
			this._onImagesLoaded
		);
	},

	/**
	 * @private
	 */
	_onImagesLoaded: function (e) {
		if (this._callback) this._callback();
		this.dispatchEvent('load');
	},

	/**
	 * @private
	 */
	_onImageError: function (image) {
		MAL.error('MALPreloader: can not load "' + image.src + '"');
	}
}));