/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Script loader class. Provides an easy way to preload javascript files.
 *
 * # Usage
 *
 * to load and ensure that a set of javascript files are loaded you may do the following
 *
 *     new MAL.ScriptLoader({
 *      path: 'this/is/a/path/',
 *      scripts: ['one.js', 'two.js', 'three.js'],
 *      callback: function () {
 *       // everything has been loaded
 *      }
 *     });
 *
 * if you want to build up a loader and start it later on you can do this
 *
 *     var loader = new MAL.ScriptLoader({
 *      autoload: false,
 *      path: 'this/is/a/path/',
 *      callback: function () {
 *       // everything has been loaded
 *      }
 *     });
 *
 *     loader.addScript('one.js');
 *
 *     loader.addScript('two.js');
 *
 *     loader.start();
 *
 *
 * @class MAL.ScriptLoader
 */
MAL.define('ScriptLoader', MAL.Util.create({
	/**
	 * initializes MAL.ScriptLoader
	 *
	 *     new MAL.ScriptLoader({
	 *      async: true,
	 *      path: 'this/is/a/path/',
	 *      scripts: ['one.js', 'two.js', 'three.js'],
	 *      callback: function () {
	 *       // everything has been loaded
	 *      }
	 *     });
	 *
	 * @method constructor
	 * @param {Object} config
	 */

	/**
	* @cfg {Function} callback
	*
	* callback function that should be called upon completion
	*/
	/**
	* @cfg {Boolean} async
	*
	* load scripts asynchronously instead of serially
	*/
	/**
	* @cfg {String} path
	*
	* path all the scripts should be prefixed with
	*/
	/**
	* @cfg {Array} scripts
	*
	* an array of the scripts you wish to load
	*/
	/**
	* @cfg {Boolean} autoload
	*
	* autoload the scripts upon initialization
	*/
	initialize: function ($config) {
		$config = $config || {};

		this.scripts = new MAL.Set();

		this.callback = $config.callback || false;
		this.async = $config.async || false;
		this.path = $config.path || '';

		// determine if we should be loading in library files normally
		this.packed = !MAL.isUndefined(MAL.CHANGESET);

		if ($config.scripts) this.addScripts($config.scripts);
		if ($config.scripts && $config.autoload !== false) this.start();
	},

	/**
	 * adds script to the load stack
	 *
	 *     loader.addScript('two.js');
	 *
	 * @param {String} src
	 */
	addScript: function (src) {
		// append .js file extension
		if (!/\.js$/.test(src)) src += '.js';

		// make special library related replacements
		if (this.packed && /^(.\/)?MALLibrary\/(.+)/.test(src)) {
			src = src.replace(/^(.\/)?MALLibrary\/(.+)/, function (match, prefix, path) {
				return ((prefix || '') + path).replace(/\//, '_');
			});
		}

		// prepend file path if relative
		if (!/^http\:\/\//.test(src)) src = this.path + src;

		this.scripts.add(src);
	},

	/**
	 * adds multiple scripts to the load stack
	 *
	 *     loader.addScript('two.js', 'three.js');
	 *
	 * @param {Array} scripts
	 */
	addScripts: function (scripts) {
		for (var script = 0; script < scripts.length; script++)
			this.addScript(scripts[script]);
	},

	/**
	 * starts script loader
	 *
	 *     preloader.start();
	 *
	 * @param {Array} scripts
	 */
	start: function () {
		MAL.Async[this.async ? 'parallel' : 'serial'](
			this.scripts.map(function (src) {

				return function (callback) {
					var self = this;

					var	script = window.document.createElement('script');
					script.setAttribute('type','text/javascript');
					script.setAttribute('src', src);

					if (script.addEventListener) {
						MAL.addEvent(script, 'load', callback);
					} else if (script.attachEvent) {
						// IE 6 7 8
						MAL.addEvent(script, 'readystatechange', function (event) {
							if(/loaded|complete/.test(script.readyState)) callback();
						});
					}

					window.document.getElementsByTagName('head')[0].appendChild(script);
				};
			}),
			this.callback
		);
	}
}));