/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Set class. Provides an array like object that ensures that all of its items are unique.
 *
 * # Usage
 *
 *     // adding and obtaining the layer
 *     var items = new MAL.Set();
 *     items.add('test');
 *     items.add('test');
 *     console.log(items.length); // should return 1 because a duplicate was added
 *
 * @class MAL.Set
 */
MAL.define('Set', MAL.Util.create({
	initialize: function () {
		// set length
		this.length = 0;

		// value cache
		this._values = [];

		// add items
		if (arguments.length) this.add.apply(this, arguments);
	},

	/**
	 * add items to set
	 *
	 *     items.add('test');
	 *
	 * @param {Object.../Number.../String...} items
	 */
	add: function (items) {
		if (!MAL.isArray(items)) items = Array.prototype.slice.call(arguments, 0);

		for (var i = 0; i < items.length; i++) {
			// item already exists
			if (this._values.indexOf(items[i]) !== -1) continue;

			// add to value cache
			this._values.push(items[i]);

			// update length
			this.length = this._values.length;
		}
	},

	/**
	 * remove item from set
	 *
	 *     items.remove('test');
	 *
	 * @param {Object.../Number.../String...} items
	 */
	remove: function (items) {
		if (!MAL.isArray(items)) items = Array.prototype.slice.call(arguments, 0);

		for (var i = 0, exists = -1; i < items.length; i++) {
			exists = this._values.indexOf(items[i]);

			// item doesn't exist
			if (exists === -1) continue;

			// remove item
			this._values.splice(exists, 1);

			// update length
			this.length = this._values.length;
		}
	},

	/**
	 * checks if set has item
	 *
	 *     items.has('test');
	 *
	 * @param {Object/Number/String} item
	 */
	has: function (item) {
		return this._values.indexOf(item) !== -1;
	},

	/**
	 * traverse set
	 *
	 *     items.forEach(function (value, key, array) {
	 *      console.log(value, key, array);
	 *     });
	 *
	 * @param {Function} callback
	 */
	forEach: function (callback, self) {
		this._values.forEach(callback, self);
	},

	/**
	 * map
	 *
	 *     var mapped = items.map(function (value) {
	 *      return value;
	 *     });
	 *
	 * @param {Function} callback
	 */
	map: function (callback, self) {
		return this._values.map(callback, self);
	},

	clone: function () {
		return this._values.slice(0);
	}
}));