/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Style instances are passed into {@link MAL.Stylesheet} via {@link MAL.Stylesheet#addStyle}
 *
 * ## example
 *
 *     var style = new MAL.Style({
 *      '.button': {
 *          width: 200,
 *          height: 200
 *      }
 *     });
 *
 * ## keyframe animations
 *
 *     var style = new MAL.Style({
 *       '@keyframes rotation-reverse': [
 *         {
 *             time: 'from',
 *             properties: {
 *                 transform: 'translateZ(0) rotate(0deg) scale(1)'
 *             }
 *         },
 *         {
 *             time: 'to',
 *             properties: {
 *                 transform: 'translateZ(0) rotate(-180deg) scale(0)'
 *             }
 *         }
 *       ]
 *     });
 *
 * @class MAL.Style
 * @extends MAL.Object
 */
MAL.define('Style', MAL.Object.extend({
	/**
	 * initializes MAL.Style
	 *
	 *     var style = new MAL.Style({
	 *       body: {
	 *         background: '#ccc'
	 *       }
	 *     });
	 *
	 * @method constructor
	 * @param {Object} config
	 */

	/**
	* @cfg {String} prefix
	* the className prefix (defaults to mal-ad-)
	*/

	/**
	* @cfg {Boolean} raw
	* if you set raw to true the rules will not have auto prefixed properties
	*/

	/**
	* @cfg {Object} defaults
	* an object of default rules
	*/

	/**
	* @cfg {Function} filePathProxy
	* event loop delay in milliseconds
	*/

	/**
	* @cfg {String} imagePath
	* event loop delay in milliseconds
	*/
	initialize: function (rules, $config) {
		this._super();

		$config = $config || {};

		this._images = new MAL.Set();
		this._prefix = $config.prefix || 'mal-ad-';
		this._selectorPrefix = $config.selectorPrefix || '';
		this._raw = $config.raw;
		this._defaults = $config.defaults || null;

		if ($config.imagePath) this._imagePath = $config.imagePath;
		if ($config.filePathProxy) this._filePathProxy = $config.filePathProxy;

		if (typeof MAL.rmv !== 'undefined') {
			this._imagePath = MAL.rmv.paths.images;
			this._filePathProxy = MAL.rmv.filePathProxy;
		}

		this.rules = {};
		this.animations = {};

		if (this._defaults) {
			this._addRulesWithDefaults(this._defaults, rules);
		} else {
			this._addRules(rules);
		}
	},

	INT_PROPERTIES: ['z-index', 'opacity'],

	/**
	 * image path
	 * @private
	 */
	_imagePath: '',

	/**
	 * image path resolution
	 *
	 * @private
	 * @param {String} filepath
	 * @param {String} filename
	 * @return {String} resolved filepath
	 */
	_filePathProxy: function (filepath, filename) {
		return filepath + filename;
	},

	/**
	 * add a rule to the rules array
	 *
	 * @private
	 * @param {String} selector
	 * @param {Object} properties
	 */
	_addRule: function (selector, properties, recursive) {
		if (selector.match(/^@keyframes/)) {
			return this._addKeyframeAnimation(selector, properties);
		}

		if (!this._raw && !recursive && this._prefix !== 'mal-' && !selector.match(/\:\:\:/) && selector.match(/[.#]mal\-/)) {
			return MAL.error('Style: bad selector (' + selector + ') you should never reference "mal-", use ::: instead');
		}

		if (selector.substr(0, 3) === ':::') {
			if (this._raw) return MAL.error('Style: bad selector (' + selector + ') you should not use ::: inside of a raw style object');

			return this._addRule(this._resolveStateWithSelector(selector), properties, true);
		} else if (selector.match(/^\:\:?[a-z]/)) {
			selector = this._selectorPrefix.slice(0, -1) + selector;

			return this._addRule(selector, properties, true);
		}

		// properly process the selector
		selector = selector.split(',').map(this._resolveSelector).join(', ');

		// track altered properties
		var _properties = this.rules[selector] || {};

		for (var property in properties) {
			if (MAL.isObject(properties[property])) {
		 		if (property.substr(0, 3) === ':::') {
					this._addRule(this._resolveStateWithSelector(property, selector), properties[property], true);
		 		} else {
		 			this._addRule(this._addParentSelectorToSelector(selector, property), properties[property], true);
		 		}
			} else {
				_properties[property] = this.resolveProperty(property, properties[property]) || undefined;
			}
		}

		// add to rules
		this.rules[selector] = _properties;
	},

	/**
	 * resolves a state selector (:::playing)
	 *
	 * @private
	 * @param {String} state
	 * @param {String} selector
	 */
	_resolveStateWithSelector: function (state, selector) {
		state = state.substr(3);
		selector = selector || '';

		var viewSelector = '.mal-view.' + this._prefix.substr(0, this._prefix.length-1);

		if (!state.match(/^mal/)) {
			// prepend with state
			state = state.split('.').map(function (state) {
				return 'state-' + state;
			}).join('.');

			if (!selector) {
				selector += this._resolveSelector(this._selectorPrefix.slice(0, -1) + '.' + this._prefix + state);
			} else {
				selector = selector.replace(viewSelector, viewSelector + '.' + this._prefix + state);
			}
		} else {
			if (!selector) {
				selector = this._resolveSelector('.' + state) + ' ' + viewSelector;
			} else {
				selector = this._resolveSelector('.' + state) + ' ' + selector;
			}
		}

		return selector;
	},

	/**
	 * resolves the original selector
	 *
	 * @private
	 * @param {String} selector
	 */
	_resolveSelector: function (selector) {
		if (this._prefix !== 'mal-') {
			if (/(^|\s|,)view/.test(selector)) {
				selector = selector.replace(/(^|\s|,)view/g, '$1.mal-view.' + this._prefix.substr(0, this._prefix.length-1));
			} else if (!selector.match(/.mal-/) && selector.match(/\.|\#/) && !selector.match(/\@instance/)) {
				selector = this._selectorPrefix + selector;
			}
		}

		if (MAL.isUndefined(this._raw)) selector = selector.replace(new RegExp('([#\\.])(?!mal-)([A-Z0-9\\-_]*)', 'ig'), '$1' + this._prefix + '$2');

		if (/\:\:-(\S+)/.test(selector)) {
			selector = selector.replace(/\:\:-(\S+)/, '::' + MAL.Environment.prefix + '$1');
		}

		return MAL.trim(selector);
	},

	/**
	 * adds the parent selectors to the selector
	 *
	 * @private
	 * @param {String} parent
	 * @param {String} selector
	 * @return {String} result
	 */
	_addParentSelectorToSelector: function (parent, selector) {
		return selector.split(',').map(function (item) {
			return parent + (item.substr(0, 1) === ':' ? '' : ' ') + item;
		}).join(',');
	},

	/**
	 * resolves a property so its crossbrowser
	 *
	 * @param {String} property
	 * @param {String} propertyValue
	 * @return {Object} resolved
	 */
	resolveProperty: function (property, propertyValue) {
		var methodName = MAL.toCamelCaseWithPrefix('_css', property);

		if (!MAL.isUndefined(this[methodName])) {
			return this[methodName](property, propertyValue);
		} else {
			var name = MAL.Environment.getCSSPropertyName(property, true),
				// auto append px if value is an int and does
				value = (MAL.Util.isNumber(propertyValue) && this.INT_PROPERTIES.indexOf(property) === -1) ? propertyValue + 'px' : propertyValue;

			// skip bugged or unsupported properties
			if (!name || (MAL.Environment.bugs.transform3d && (property === 'backface-visibility' || property === 'perspective'))) {
				return null;
			}

			// auto append px if value is an int and does
			return {
				name: name,
				value: value
			};
		}
	},

	/**
	 * add an animation to the animation object
	 *
	 * @private
	 * @param {String} selector
	 * @param {Object} list
	 */
	_addKeyframeAnimation: function (selector, list) {
		var identifier = selector.replace('@keyframes ', ''),
			keyframesName = (MAL.Environment.getCSSPropertyName('animation') === 'animation' ? '' : MAL.Environment.prefix) + 'keyframes',
			value = '';

		list.forEach(function (item) {
			value += item.time + ' {';
			for (var property in item.properties) {
				var resolved = MAL.Style.prototype.resolveProperty(property, item.properties[property]);
				if (resolved) value += resolved.name + ': ' + resolved.value + ';';
			}
			value += '}';
		}, this);

		delete this.rules[selector];
		this.animations['@' + keyframesName + ' ' + identifier] = value;
	},


	/**
	 * adds a set of rules to the rules array
	 *
	 * @private
	 * @param {Object} rules
	 */
	_addRules: function (rules) {
		// traverse arguments and run addRule on each item
		for (var rule in rules)
			this._addRule(rule, rules[rule]);
	},

	/**
	 * adds a set of rules with default rules to the rules array
	 *
	 * @private
	 * @param {Object} default rules
	 * @param {Object} rules
	 */
	_addRulesWithDefaults: function (defaults, rules) {
		// process rules with defaults
		for (var rule in rules)
			for (var defaultProperty in defaults)
				if (MAL.isUndefined(rules[rule][defaultProperty]))
					rules[rule][defaultProperty] = defaults[defaultProperty];

		this._addRules(rules);
	},

	/**
	 * background css property override
	 *
	 * @private
	 */
	_cssBackground: function (property, value) {
		// add no-repeat as default if it is not set
		if (!value.match(/repeat/)) value += ' no-repeat';
		return this._cssBackgroundImage(property, value);
	},

	/**
	 * background image property override
	 *
	 * @private
	 */
	_cssBackgroundImage: function (property, value) {
		var imageRegExp = /url\(['"]?(?!data:)([^'"]+)['"]?\)/,
			backgrounds = value.split(',');

		// multiple backgrounds
		backgrounds = backgrounds.map(function (background) {
			var image;

			if (imageRegExp.test(background)) {
				// fix src
				background = background.replace(imageRegExp, function (fullmatch, filename) {
					// high res / retina support
					if (!window.devicePixelRatio || window.devicePixelRatio === 1) {
						filename = image = filename.replace(/@[0-9.]+x/,'');
					} else {
						image = filename;
					}

					return 'url("' + this._filePathProxy(this._imagePath, filename) + '")';
				}.bind(this));

				// add to image set
				this._images.add(image);
			}

			return background;
		}, this);

		return {
			name: property,
			value: backgrounds.join(',')
		};
	},

	/**
	 * css opacity property override
	 *
	 * @private
	 */
	_cssOpacity: function (property, value) {
		if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			return {
				name: 'filter',
				value: 'Alpha(opacity=' + (value * 100) + ')'
			};
		} else {
			return {
				name: property,
				value: value
			};
		}
	},

	/**
	 * returns an array of all images used in the style
	 *
	 * @return {Array} images
	 */
	getImages: function () {
		return this._images.clone();
	}
}));