/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * @class MAL.ThirdPartyElements
 * @extends MAL.Object
 *
 * Provides the ability to modify third party site elements so that they can support our animations, and also maintains their initial state for restoration
 */
MAL.define('ThirdPartyElements', MAL.Object.extend({
	/**
	 * initializes ThirdPartyElements
	 *
	 * @method constructor
	 */
	initialize: function ($config) {
		this._super();

		this.stage = $config.stage;
		this.elements = [];
	},

	// properties that need to be overridden to the target element
	TARGET_PROPERTIES: {
		'z-index': 10000,
		'position': 'relative'
	},

	// properties that need to be overridden to the elements in the targets ancestors
	ANCESTRY_PROPERTIES: {
		'z-index': 'auto',
		'overflow': 'visible'
	},

	// property values that do not need to be overriden
	VALID_PROPERTY_DEFAULTS: {
		'z-index': ['auto'],
		'overflow': ['auto', 'visible'],
		'position': ['relative', 'absolute']
	},

	/**
	 * gets an elements computed property
	 *
	 * @private
	 * @param {Object} element
	 * @param {String} property
	 * @return {String} value
	 */
	_getComputedStyle: function (element, property) {
		return window.document.defaultView.getComputedStyle(element, '').getPropertyValue(property);
	},

	/**
	 * gets an elements computed property
	 *
	 * @private
	 * @param {Object} element
	 * @return {Array} ancestors
	 */
	_getAncestry: function (element) {
		var elements = [element],
			cursor = element;

		function getParent() {
			cursor = cursor.parentElement;
			elements.push(cursor);
			if (cursor.parentElement != null) getParent();
		}

		if (cursor.parentElement) getParent();
		return elements;
	},

	/**
	 * processes an elements ancestors for bad things and returns the flagged ones
	 *
	 * @private
	 * @param {Object} element
	 * @return {Array} dirtyElements
	 */
	_checkForDirtyElements: function (target) {
		var dirty = [];
			ancestry = this._getAncestry(target);

		ancestry.forEach(function (element) {
			if (element.parentElement == null || element == target) return;

			var properties = [];

			for (var property in this.ANCESTRY_PROPERTIES) {
				var value = this._checkElementProperty(element, property);
				if (value !== false) properties.push([property, value]);
			}

			if (properties.length) dirty.push({element: element, properties: properties});
		}, this);

		return dirty;
	},

	/**
	 * checks a properties value against the VALID_PROPERTY_DEFAULTS array returns false if its valid
	 *
	 * @private
	 * @param {Object} element
	 * @param {String} property
	 * @return {String/Boolean} value
	 */
	_checkElementProperty: function (element, property) {
		var value = this._getComputedStyle(element, property);

		if (this.VALID_PROPERTY_DEFAULTS[property].indexOf(value) != -1) {
			return false;
		} else {
			return value;
		}
	},

	/**
	 * modifies all elements that are required to be modified
	 *
	 * @private
	 * @param {Array} elements
	 */
	_modifyElements: function (elements) {
		elements.forEach(function (item) {
			var element = item.element,
				layer = element.MALTweenLayer;

			item.properties.forEach(function (property) {
				var name = property[0],
					value = property[1];

				layer.setProperty(name, this.ANCESTRY_PROPERTIES[name], true);
			}, this);
		}, this);
	},

	/**
	 * restores all elements that have been modified to their initial state
	 *
	 * @private
	 * @param {Array} elements
	 */
	_restoreElements: function (elements) {
		elements.forEach(function (item) {
			var element = item.element,
				layer = element.MALTweenLayer;

			item.properties.forEach(function (property) {
				var name = property[0],
					value = property[1];

				layer.setProperty(name, value, true);
			});
		});
	},

	/**
	 * adds elements
	 *
	 * @param {Array} elements
	 */
	addElements: function () {
		var elements = MAL.argumentsToArray(arguments);

		elements.forEach(function (element) {
			if (MAL.isString(element))
				element = document.getElementById(element);

			if (element instanceof MAL.tween.CSSLayer || element instanceof MAL.tween.JSLayer)
				element = element.element;

			if (!element.MALTweenLayer) this.stage.addLayer(element);

			var dirtyElements = this._checkForDirtyElements(element);

			if (dirtyElements.length) {

				var properties = {};
				for (var property in this.TARGET_PROPERTIES) {
					var value = element.style[property];
					if (this.VALID_PROPERTY_DEFAULTS[property].indexOf(value) != -1) properties[property] = value;
				}

				dirtyElements.forEach(function (item) {
					if (!item.element.MALTweenLayer) this.stage.addLayer(item.element);
				}, this);

				this.elements.push({
					target: {element: element, properties: properties},
					ancestry: dirtyElements
				});
			}
		}, this);
	},

	/**
	 * puts all of the elements in their modified state
	 */
	modify: function () {
		this.elements.forEach(function (item) {
			var element = item.target.element,
				layer = element.MALTweenLayer;

			for (var property in this.TARGET_PROPERTIES)
				layer.setProperty(property, this.TARGET_PROPERTIES[property], true);

			this._modifyElements(item.ancestry);
		}, this);

		this.stage._layers.forEach(function (layer) {
			layer._setCSSTextFromBuffer();
		});
	},

	/**
	 * puts all of the elements in their restored state
	 */
	restore: function () {
		this.elements.forEach(function (item) {
			var element = item.target.element,
				layer = element.MALTweenLayer;

			for (var property in item.target.properties)
				layer.setProperty(property, item.target.properties[property], true);

			if (layer) layer.clearAll().setProperties(layer._initialTransformations, true);

			this._restoreElements(item.ancestry);
		}, this);

		this.stage._layers.forEach(function (layer) {
			layer._setCSSTextFromBuffer();
		});
	}
}));