/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * Util namespace. Provides you with a set of common functions and helpers. **Majority of Util methods are also added directly to the MAL namespace, for example you can use MAL.log instead of MAL.Util.log**
 *
 * # Usage
 *
 * lets pretend that you have a function that accepts a string, array or object as the first param and does different things based on the type
 *
 *     function something (param) {
 *      if (MAL.Util.isString(param)) {
 *       // param is a string
 *      } else if (MAL.Util.isArray(param)) {
 *       // param is a array
 *      } else if (MAL.Util.isObject(param)) {
 *       // param is  object
 *      }
 *     }
 *
 * as you can see its very useful for type checking, another useful method is parseNumber it actually returns 0 instead of NaN
 *
 *     var ensuredNumber = MAL.Util.parseNumber(param);
 *
 * also the Util namespace is house of the create method which was used to create {@link MAL.Object} you can use it like this
 *
 *     var Person = MAL.Util.create({
 *      initialize: function () {
 *       // constructor
 *      }
 *     });
 *
 * this is an ever-growing list of methods, feel free to add to it
 *
 * @class MAL.Util
 * @singleton
 */

MAL.Util = function () {};

// Async helper
MAL.Util.Async = {
	parallel: function (tasks, callback) {
		var finished = 0;

		tasks.map(function (value) {
			return function () {
				value(function () {
					finished++;
					if (finished === tasks.length) callback();
				});
			};
		}).forEach(function (value) { value(); });
	},
	serial: function (tasks, callback) {
		tasks = tasks.map(function (value) {
			return function () {
				value(next);
			};
		});

		function next () {
			if (tasks.length) {
				tasks.shift()();
			} else {
				callback();
			}
		}
		next();
	}
};

/**
 * binds all methods to scope
 *
 *     MAL.bindAll(this, 'doThis', 'doThat');
 *
 * @param {Object} scope
 * @return {Function...} methods
 */
MAL.Util.bindAll = function () {
	var self = arguments[0];
	for (var i = 1; i < arguments.length; i++) self[arguments[i]] = self[arguments[i]].bind(self);
};

/**
 * check if the param is undefined
 *
 * @param {Object} o
 * @return {Boolean}
 */
MAL.Util.isUndefined = function (o) {
	return o === void 0;
};

/**
 * check if the param is a {@link Function}
 *
 * @param {Function} o
 * @return {Boolean}
 */
MAL.Util.isFunction = function (o) {
	return !!(o && o.constructor && o.call && o.apply);
};

/**
 * check if the param is a {@link Array}
 *
 * @param {Array} o
 * @return {Boolean}
 */
MAL.Util.isArray = function (o) {
	return Object.prototype.toString.call(o) === '[object Array]';
};

/**
 * check if the param is a {@link Date}
 *
 * @param {Date} o
 * @return {Boolean}
 */
MAL.Util.isDate = function (o) {
	return Object.prototype.toString.call(o) === '[object Date]';
};

/**
 * check if the param is a {@link RegExp}
 *
 * @param {RegExp} o
 * @return {Boolean}
 */
MAL.Util.isRegExp = function (o) {
	return Object.prototype.toString.call(o) === '[object RegExp]';
};

/**
 * check if the param is a {@link String}
 *
 * @param {String} o
 * @return {Boolean}
 */
MAL.Util.isString = function (o) {
	return !!(o === '' || o && o.charCodeAt && o.substr);
};

/**
 * check if the param is null
 *
 * @param {Object} o
 * @return {Boolean}
 */
MAL.Util.isNull = function (o) {
	return o === null;
};

/**
 * check if the param is NaN
 *
 * @param {Number} o
 * @return {Boolean}
 */
MAL.Util.isNaN = function (o) {
	return o !== o;
};

/**
 * check if the param is a {@link Number}
 *
 * @param {Number} o
 * @return {Boolean}
 */
MAL.Util.isNumber = function (o) {
	return typeof o === 'number';
};

/**
 * check if the param is a {@link Object}
 *
 * @param {Object} o
 * @return {Boolean}
 */
MAL.Util.isObject = function (o) {
	return !(MAL.isNumber(o) || MAL.isNaN(o) || MAL.isNull(o) || MAL.isString(o) || MAL.isArray(o) || MAL.isFunction(o) || MAL.isUndefined(o));
};

/**
 * check if the param is a {@link Boolean}
 *
 * @param {Boolean} o
 * @return {Boolean}
 */
MAL.Util.isBoolean = function (o) {
	return o === true || o === false || Object.prototype.toString.call(o) == '[object Boolean]';
};

/**
 * shorten a number to the specified decimal points
 *
 *     MAL.Util.shortenFloat(1.0011111, 3); // returns 1.001
 *
 * @param {Number} number
 * @param {Number} points
 * @return {Number}
 */
MAL.Util.shortenFloat = function (number, points) {
	return Math.round(number*Math.pow(10,points))/Math.pow(10,points);
};

/**
 * converts an array-like object into an array
 *
 * @param {Object} arrayLike
 * @return {Array}
 */
MAL.Util.toArray = function (arrayLike) {
	// HACK: issue with StaticNodeList
	if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
		return Array.prototype.map.call(arrayLike, function (item) {return item;});
	}

	return Array.prototype.slice.call(arrayLike, 0);
};

/**
 * retuns min number inside of the array
 *
 * @param {Array} numbers
 * @return {Number}
 */
MAL.Util.arrayMax = function (array) {
	return Math.max.apply(null, array);
};

/**
 * retuns max number inside of the array
 *
 * @param {Array} numbers
 * @return {Number}
 */
MAL.Util.arrayMin = function (array) {
	return Math.min.apply(null, array);
};

/**
 * parse the float out of a string, and if it is NaN return 0
 *
 *     MAL.Util.parseNumber(" 1.01 "); // returns 1.01 as a float
 *
 * @param {String} o
 * @return {Number}
 */
MAL.Util.parseNumber = function (string) {
	var number = parseFloat(string);

	if (MAL.isNaN(number)) {
		return 0;
	} else {
		return number;
	}
};

/**
 * clones an object or array of objects
 *
 *     MAL.Util.clone({name: 'joe', details: {age: 24}}, true); // returns a deep clone of the object
 *
 * @param {Object} src
 * @param {Boolean} deep
 * @return {Number}
 */
MAL.Util.clone = function (src, deep) {
    if(!src || MAL.isNumber(src) || MAL.isString(src) || MAL.isBoolean(src) || MAL.isNaN(src)){
        return src;
    }

	// Number
    if(MAL.isNumber(src)){
        return src;
    }

    //Honor native/custom clone methods
    if(src.clone && MAL.isFunction(src.clone)){
        return src.clone(deep);
    }

    //DOM Elements
    if(src.nodeType && MAL.isFunction(src.cloneNode)){
        return src.cloneNode(deep);
    }

    //Date
    if(MAL.isDate(src)){
        return new Date(src.getTime());
    }

    //RegExp
    if(MAL.isRegExp(src)){
        return new RegExp(src);
    }

    //Function
    if(MAL.isFunction(src)){
        //Wrap in another method to make sure == is not true;
        //Note: Huge performance issue due to closures, comment this :)
        return (function(){
            src.apply(this, arguments);
        });

    }

    var ret, index;
    //Array
    if(MAL.isArray(src)){
        //[].slice(0) would soft clone
        ret = src.slice();
        if(deep){
            index = ret.length;
            while(index--){
                ret[index] = MAL.clone(ret[index], true);
            }
        }
    } else {
        ret = src.constructor ? new src.constructor() : {};
        for (var prop in src) {
            ret[prop] = deep ? MAL.clone(src[prop], true) : src[prop];
        }
    }

    return ret;
};

/**
 * iterates objects and arrays
 *
 *     MAL.Util.each([1,2,3], function (value, key, list) {
 *
 *     });
 *
 *     MAL.Util.each({a: 'a', b: 'b', c: 'c'}, function (value, key, list) {
 *
 *     });
 *
 * @param {Object} o
 * @param {Object} callback
 */
MAL.Util.each = function (o, callback) {
	if (MAL.isArray(o)) {
		Array.prototype.forEach.call(o, callback);
	} else {
		for (var key in o) if (o.hasOwnProperty(key)) callback(o[key], key, o);
	}
};

/**
 * retuns trimmed string
 *
 * @param {String} string
 * @return {String}
 */
MAL.Util.trim = (function () {
	var trimRegExp = /^\s+|\s+$/g;
	return function (string) {
		return string.replace(trimRegExp,'');
	};
}());

/**
 * easy way to ensure a delta from strings/floats
 *
 *     MAL.Util.delta("1", "5"); // returns 4
 *     MAL.Util.delta(1, "5"); // returns 4
 *     MAL.Util.delta(1, 5); // returns 4
 *
 * @param {String/Number} start
 * @param {String/Number} finish
 * @return {String}
 */
MAL.Util.delta = function (start, finish) {
	return MAL.parseNumber(finish) - MAL.parseNumber(start);
};

/**
 * Prototype Inheritance
 *
 *     var Person = MAL.Util.create({
 *      initialize: function () {
 *       // constructor
 *      }
 *     });
 *
 * @param {Object} prototype
 * @return {Function} object
 */
MAL.Util.create = function (proto) {
	var Self = this,
		BaseClass = function (_MAL) { // only if the argument doesnt match MAL
			if (MAL !== _MAL) {
				for (var method in this) {
					if (MAL.isFunction(this[method])) {
						this[method] = this[method].bind(this);
					}
				}
				if (MAL.isFunction(this.initialize)) {
					this.initialize.apply(this, arguments);
				}
			}
		};

	BaseClass.prototype = new Self(MAL); // private method as magic cookie

	function makeMethod (key) {
		(function (func, superFunc) {
			BaseClass.prototype[key] = !MAL.isFunction(func) || !MAL.isFunction(superFunc) ? func : function () {this._super = superFunc;return func.apply(this, arguments);};
		}(proto[key], BaseClass.prototype[key]));
	}

	for (var key in proto) makeMethod(key);

	BaseClass.prototype.constructor = BaseClass;
	BaseClass.extend = this.extend || this.create;
	return BaseClass;
};

/**
 * ensures that you have a real array, this is mainly useful if you want to support multiple arguments and a single array arguemnt
 *
 *     var tweens = MAL.Util.argumentsToArray(arguments);
 *     tweens.forEach(function (tween) {});
 *
 * @param {Array} arguments
 * @return {Array} arguments
 */
MAL.Util.argumentsToArray = function (args) {
	if (args.length === 1 && MAL.isArray(args[0])) {
		return args[0];
	} else {
		return Array.prototype.slice.call(args, 0);
	}
};

/**
 * converts an object to an array of objects
 *
 * @param {Object} object
 * @return {Array} arguments
 */
MAL.Util.objectToArray = function (o) {
	var array = [];
	for (var property in o) array.push({name: property, value: o[property]});
	return array;
};

/**
 * easy way to camel case a string with a lowercase prefix
 *
 *     MAL.Util.toCamelCaseWithPrefix('webkit', 'some-random-css-property'); // returns "webkitSomeRandomCssProperty"
 *
 * @param {String} prefix
 * @param {String} string
 * @return {String}
 */
MAL.Util.toCamelCaseWithPrefix = function (prefix, string) {
	return MAL.toCamelCase(prefix + '-' + string);
};

/**
 * easy way to camel case a string
 *
 *     MAL.Util.toCamelCase('some-random-css-property'); // returns "someRandomCssProperty"
 *
 * @param {String} string
 * @return {String}
 */
MAL.Util.toCamelCase = function (string) {
	for (var i = 0, result = '', split = string.split('-'); i < split.length; i++) result += i ? split[i].charAt(0).toUpperCase() + split[i].slice(1) : split[i];
	return result;
};

/**
 * easy way to convert CamelCase to hyphen-delmited strings
 * @param  {String} string
 * @return {String} string
 */
MAL.Util.toHyphenDelimited = function (string) {
	return string.replace(/([A-Z])/g, function (string, match, index) {
		return (index > 0 ? '-' : '') + match.toLowerCase();
	});
};


/**
 * lets you specify a base object and extend upon it
 *
 *     MAL.mergeOptions({
 *      one: 1,
 *      three: 3
 *     }, {one: 'NOT ONE', two: 2}); // returns {one: 'NOT ONE', two: 2, three: 3}
 *
 * this is mainly used to providing default options to classes
 *
 * @param {Object} defaults
 * @param {Object...} objects to mixin
 * @return {Object} merged options
 */
MAL.Util.mergeOptions = function () {
	var options = arguments[0];

	for (var i = 1, l = arguments.length; i < l; i++)
		for (var name in arguments[i]) options[name] = arguments[i][name];

	return options;
};

/**
 * allows you to provide a default value, and if the new value specified is undefined it fallsback
 *
 *     MAL.defaultValue(1, undefined); // 1
 *     MAL.defaultValue(1, 2); // 2
 *
 * @param {Object} fallback value
 * @param {Object} new value
 * @return {Object} value
 */
MAL.Util.defaultValue = function (fallback, value) {
	return MAL.isUndefined(value) ? fallback : value;
};


/**
 * returns a percent based on a cubic bezier applied to a percent
 *
 * @param {Number} percent
 * @param {Number} p1x
 * @param {Number} p1y
 * @param {Number} p2x
 * @param {Number} p2y
 */
MAL.Util.cubicBezierWithPercent = (function () {
	var ax, bx, cx, ay, by, cy, t, duration;

	function _solve (x, epsilon) {
		return _sCY(_solveCX(x,epsilon));
	}

	function _solveCX (x, epsilon) {
		var t0,t1,t2,x2,d2,i;
			for(t2 = x, i = 0; i < 8; i++) {
				x2 = _sCX(t2)-x;
				if(_fabs(x2)<epsilon) {
					return t2;
				}
				d2 = _sCDX(t2);
				if(_fabs(d2)<1e-6) {
					break;
				}
				t2 = t2 - x2/d2;
			}

			t0 = 0.0;
			t1 = 1.0;
			t2 = x;

			if (t2 < t0) {
				return t0;
			}

			if (t2 > t1) {
				return t1;
			}

			while (t0 < t1) {
				x2 = _sCX(t2);
				if (_fabs(x2-x) < epsilon) {
					return t2;
				}
				if (x > x2) {
					t0 = t2;
				} else {
					t1 = t2;
				}
				t2 = (t1 - t0) * 0.5 + t0;
			}
			return t2; // Failure.
	}

	function _sCX (t) {
		return ((ax*t+bx)*t+cx)*t;
	}

	function _sCY (t) {
		return ((ay*t+by)*t+cy)*t;
	}

	function _sCDX (t) {
		return (3.0*ax*t+2.0*bx)*t+cx;
	}

	function _sE (duration) {
		return 1.0/(200.0*duration);
	}

	function _fabs (n) {
		if (n >= 0) {
			return n;
		} else {
			return 0 - n;
		}
	}

	return function (percent,p1x,p1y,p2x,p2y) {

		ax = bx = cx = ay = by = cy = 0;

		cx= 3.0 * p1x;
		bx= 3.0 * (p2x - p1x) - cx;
		ax= 1.0 - cx - bx;
		cy= 3.0 * p1y;
		by= 3.0 * (p2y - p1y) - cy;
		ay= 1.0 - cy - by;
		duration = 1;
		t = percent;

		return _solve(t, _sE(duration));
	};
}());

/**
 * protects html ID's and class names, it will also prefix images with the specified path and return a Set of the images that were used
 *
 *     var data = MAL.processHTML("<div id='Test'></div>", "images/");
 *     console.log(data.images);
 *     console.log(data.html);
 *
 * @param {String} html
 * @param {String} path path for images
 * @param {String} prefix prefix for class names and id's
 * @return {Object}
 */
MAL.Util.processHTML = function (html, path, prefix) {
	var self = this,
		srcRegExp = /src=['"](?!data:|http:|https:)([^'"]+)['"]/ig,
		matches,
		images = new MAL.Set();

	prefix = prefix || 'mal-ad-';

	// TODO: look into the reasoning behind this later
	html = String(html);

	// snag images that were used
	html = html.replace(srcRegExp, function () {
		var src = arguments[1];

		// high res / retina support
		if (!window.devicePixelRatio || window.devicePixelRatio === 1) src = src.replace(/@[0-9.]+x/,'');

		images.add(src);
		return "src='" + path + src + "'";
	});

	// prefix name spaces
	html = html.replace(/(id|class)=(['"])([^'"]+)['"]/ig, function () {
		var type = arguments[1],
			quote = arguments[2],
			items = arguments[3].split(' '),
			prefixRegexp = new RegExp('^(mal-|' + prefix + ')');

		for (var i = 0; i < items.length; i++) {
			if (!prefixRegexp.test(items[i])) items[i] = prefix + items[i];
		}

		return type + '=' + quote + items.join(' ') + quote;
	});

	// return html with namespaces
	return {
		html: html,
		images: images
	};
};

/**
 * gets the absolute elements position by adding up its ancestors offsets
 *
 * @param {HTMLElement} element
 * @return {Object} positioning object {x: 0, y: 0}
 */
MAL.Util.findElementPosition = function (element) {
	var x = 0,
		y = 0;

	if (element.offsetParent) {
		do {
			x += element.offsetLeft;
			y += element.offsetTop;
		} while (!!(element = element.offsetParent));
	}
	return {x: x, y: y};
};

/**
 * get the document size (cross browser)
 *
 * @return {Object} size {width: 0, height: 0}
 */
MAL.Util.getDocumentSize = function () {
	return {
		width: document.width || Math.max(document.documentElement.clientWidth, document.body.scrollWidth, document.documentElement.scrollWidth, document.body.offsetWidth, document.documentElement.offsetWidth),
		height: document.height || Math.max(document.documentElement.clientHeight, document.body.scrollHeight, document.documentElement.scrollHeight, document.body.offsetHeight, document.documentElement.offsetHeight)
	};
};

/**
 * attaches an object of events to the specified element
 *
 * @param {HTMLElement} element
 * @param {Object} events
 */
MAL.Util.addEvents = function (element, events) {
	for (var type in events) MAL.addEvent(element, type, events[type]);
};

/**
 * attaches an event specified element with a callback
 *
 * @param {HTMLElement} element
 * @param {String} event
 * @param {Function} callback
 */
MAL.Util.addEvent = function (element, type, callback) {
	if (element.addEventListener) {
		element.addEventListener(type, callback, false);
	} else {
		element.attachEvent('on' + type, callback);
	}
};

/**
 * removes an object of events to the specified element
 *
 * @param {HTMLElement} element
 * @param {Object} events
 */
MAL.Util.removeEvents = function (element, events) {
	for (var type in events) MAL.removeEvent(element, type, events[type]);
};


/**
 * removes an event specified element with a callback
 *
 * @param {HTMLElement} element
 * @param {String} event
 * @param {Function} callback
 */
MAL.Util.removeEvent = function (element, type, callback) {
	if (element.removeEventListener) {
		element.removeEventListener(type, callback, false);
	} else {
		element.detachEvent('on' + type, callback);
	}
};

/**
 * easy way to log to the console without worrying about its existence
 *
 *     MAL.Util.log('this', 'is', 'a', 'test'); // logs out "this is a test"
 *
 * @param {String...} message
 */
MAL.Util.log = function() {
	if (MAL.Environment.browser !== 'adobeair' && (typeof console === 'undefined' || typeof console.log === 'undefined' || typeof MAL.DEBUG === 'undefined' || !MAL.DEBUG) && !document.getElementById('mal-logs')) return;

	if (MAL.Environment.browser === 'adobeair' || document.getElementById('mal-logs')) {
		var message = document.createElement('div');
		message.className = 'mal-ad-log';
		message.innerHTML = Array.prototype.slice.call(arguments, 0).join(' ');
		if (document.getElementById('mal-logs')) {
			document.getElementById('mal-logs').appendChild(message);
		} else {
			document.body.appendChild(message);
		}
	} else if (typeof console.log.apply !== 'undefined' && MAL.Environment.os !== 'iphone' && MAL.Environment.os !== 'ipad') {
		console.log.apply(console, arguments);
	} else {
		console.log(Array.prototype.slice.call(arguments, 0).join(' '));
	}
};

/**
 * will stringify and log any object without traversing
 *
 * @param {Object} item
 */
MAL.Util.logObject = function (e) {
	var props = {};
	for (var prop in e) if (MAL.isNumber(e[prop]) || MAL.isString(e[prop])) props[prop] = e[prop];
	MAL.log(JSON.stringify(props));
};

/**
 * cross browser dir logging
 *
 * @param {Object} item
 */
MAL.Util.dir = function() {
	if (typeof console === 'undefined' || typeof console.dir === 'undefined' || typeof MAL.DEBUG === 'undefined' || !MAL.DEBUG) return;
	if (typeof console.dir.apply !== 'undefined') {
		console.dir.apply(console, arguments);
	} else {
		console.dir(arguments[0]);
	}
};

/**
 * cross browser error logging
 *
 * @param {String} message
 */
MAL.Util.error = function(message) {
	if (typeof console === 'undefined' || typeof MAL.DEBUG === 'undefined' || !MAL.DEBUG) return;
	if (MAL.Environment.browser === 'firefox' && typeof console !== 'undefined' && !MAL.Util.isUndefined(console.error)) {
		console.error(message);
	} else {
		throw new Error(message);
	}
};

// expose all methods to parent namespace for convenience
(function () {
	for (var method in MAL.Util) if (method !== 'create') MAL[method] = MAL.Util[method];
})();