/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * View class is made to be extended upon and should never be directly initialized, when extending upon it you need to specify the following properties
 *
 * * name
 * * styles
 * * html
 *
 * NOTE: MAL.rmv is required for imagePath and filePathProxy.
 *
 * ## example
 *
 *      MAL.ExampleView = MAL.View.extend({
 *          name: 'example',
 *
 *			className: 'card', // accepts an array too
 *
 *          styles: function () { return {
 *              view: {
 *                  width: 200,
 *                  height: 200
 *              },
 *
 *              '.button': {
 *                  width: 200,
 *                  height: 200
 *              }
 *          }},
 *
 *          html: '<div class="button"><div class="item mal-external-item"><img src="image2.png"/></div></div>'
 *      });
 *
 * ## styles
 *
 * use the 'view' selector to target the views container element
 *
 * ## html
 *
 * every class gets prefixed except for when you already prefix it so *class="something mal-something-else"* becomes *class="mal-{VIEW_NAME}-something mal-something-else"*
 *
 * @class MAL.View
 * @extends MAL.Object
 *
 */
MAL.define('View', MAL.Object.extend({
	/**
	 * initializes MAL.View
	 *
	 * @method constructor
	 * @param {Object} config
	 */
	initialize: function ($config) {
		$config = $config || {};

		this._super();

		this.id = MAL.View.prototype.id++;

		this.controller = $config.controller;

		this.name = this.name || MAL.toHyphenDelimited(this.objectClassName);

		this._images = [];
		this.prefix = 'mal-' + this.name + '-';

		if (typeof MAL.rmv !== 'undefined') {
			this._imagePath = MAL.rmv.paths.images;
			this._filePathProxy = MAL.rmv.filePathProxy;
		} else {
			MAL.error('Missing MAL.rmv');
		}

		this.element = this._createContainer(this.name, this.html);
		this.element.MALController = this.controller;

		this.layer = MAL.stage[$config.layerType ? 'add' + $config.layerType + 'Layer' : 'getLayer'](this.element);

		this._styles = new MAL.Set();
		this._views = new MAL.Set();

		this.style = this._createStyle(this.styles());
		this._styles.add(this.style);
	},

	/**
	 * the view instance ID
	 * @type {Number}
	 */
	id: 0,

	/**
	 * node used for converting HTML into DOM
	 *
	 * @private
	 */
	_DOMContainer: document.createElement('div'),

	/**
	 * add subview references to the view
	 * @param  {Array} subviews
	 */
	addSubviews: function () {
		var subviews = MAL.argumentsToArray(arguments);
		subviews.forEach(this.addSubview, this);
	},

	/**
	 * adds a subview reference to the view
	 * @param  {Object} subview
	 */
	addSubview: function (subview) {
		if (subview instanceof MAL.ViewController) subview = subview.view;
		this._views.add(subview);
	},

	/**
	 * adds a subview reference to the view that doesnt exist yet
	 *
	 * HACK: this is a hack, and does generate an additional unused view. In the future we can fix this though.
	 *
	 * @param  {Object} subview
	 */
	willHaveSubview: function (Subview) {
		var subview = new Subview();
		this.addSubview(subview);
	},

	/**
	 * creates the container element and injects it with html
	 *
	 * @private
	 * @param  {String} name
	 * @param  {String} html
	 * @return {Object} element
	 */
	_createContainer: function (name, html) {
		var classes = ['view', this.name];
		if (MAL.isArray(this.className)) {
			classes = classes.concat(this.className);
		} else if (MAL.isString(this.className)) {
			classes.push(this.className);
		}
		classes = classes.map(function (className) {
			return 'mal-' + className;
		});

		var element = document.createElement('div');
		element.className = classes.join(' ');
		element.id = 'mal-view' + this.id;

		var processed = MAL.processHTML(html, this._imagePath, this.prefix);

		element.innerHTML = processed.html;
		this._images = processed.images;

		return element;
	},

	/**
	 * creates a Style with the provided rules
	 *
	 * @private
	 * @param  {Object} rules
	 * @return {MAL.Style} style
	 */
	_createStyle: function (rules) {
		//
		var style = new MAL.ViewStyle(rules, {view: this}),
			images = style.getImages();

		this._images.add(images);

		return style;
	},

	/**
	 * adds view prefix to selector string
	 *
	 * @param {String} selector
	 * @return {String} selector
	 */
	addPrefixToSelector: function (selector) {
		return selector.replace(new RegExp('([#\\.])(?!mal-)([A-Z0-9\\-_]*)', 'ig'), '$1' + this.prefix + '$2');
	},

	/**
	 * adds view prefix to className string
	 *
	 * @param {String} selector
	 * @return {String} selector
	 */
	addPrefixToClassName: function (className) {
		return this.prefix + className;
	},

	/**
	 * adds view prefix to html string
	 *
	 * @param {String} selector
	 * @return {String} selector
	 */
	addPrefixToHTML: function (html) {
		return MAL.processHTML(html, this._imagePath, this.prefix).html;
	},

	/**
	 * add additional images that are used in the view
	 *
	 * @return {Array} images
	 */
	addImages: function () {
		this._images.add(MAL.argumentsToArray(arguments));
	},

	/**
	 * returns images that were used in the HTML and CSS for the view
	 *
	 * @return {Array} images
	 */
	getImages: function () {
		var images = this._images.clone();

		this._views.forEach(function (view) {
			images = images.concat(view.getImages());
		});

		return images;
	},

	/**
	 * returns a flat array of styles that are related to the view
	 *
	 * @return {Array} styles
	 */
	getStyles: function () {
		var styles = this._styles.clone();

		this._views.forEach(function (view) {
			styles = styles.concat(view.getStyles());
		});

		return styles;
	},

	/**
	 * works just like element.querySelector but operates directly on the views container
	 *
	 * @param  {String} selector
	 * @param  {Object} [target]
	 * @return {Object} element
	 */
	query: function (selector, target) {
		target = target || this.element;
		return target.querySelector(this.addPrefixToSelector(selector));
	},

	/**
	 * returns a layer object
	 *
	 * @param  {String} selector
	 * @param  {Object} [target]
	 * @return {Object} layer or layerProxy
	 */
	queryLayer: function (selector, target) {
		return MAL.stage.getLayer(this.query(selector, target));
	},

	/**
	 * works just like element.querySelectorAll but operates directly on the views container
	 *
	 * @param  {String} selector
	 * @param  {Object} [target]
	 * @return {Array} elements
	 */
	queryAll: function (selector, target) {
		target = target || this.element;
		return MAL.toArray(target.querySelectorAll(this.addPrefixToSelector(selector)));
	},

	/**
	 * gets a layer proxy for the selectors results
	 *
	 * @param  {String} selector
	 * @param  {Object} [target]
	 * @return {Object} layer or layerProxy
	 */
	queryLayerAll: function (selector, target) {
		return MAL.stage.getLayers.apply(MAL.stage, this.queryAll(selector, target));
	},

	/**
	 * appends the elements container to the provided element
	 *
	 * @param  {Object} target target element
	 */
	appendTo: function (target) {
		this._getElement(target).appendChild(this.element);
	},

	/**
	 * appends the elements to target element
	 *
	 * @param  {Object} target target element
	 * @param  {Array} views array of elements
	 */
	appendElementsTo: function (elements, target) {
		target = this._getElement(target) || this.element;

		elements.forEach(function (element) {
			if (element instanceof MAL.ViewController || element instanceof MAL.View) {
				this.addSubview(element);
			}

			target.appendChild(this._getElement(element));
		}, this);
	},

	/**
	 * gets the element from a View, Controller, or queryString
	 *
	 * @param  {Object} target target element
	 */
	_getElement: function (item) {
		if (item instanceof MAL.View) {
			item = item.element;
		} else if (item instanceof MAL.ViewController) {
			item = item.view.element;
		} else if (MAL.isString(item)) {
			item = this.query(item);
		}

		return item;
	},

	/**
	 * checks if element has a class
	 *
	 * @param  {String} className
	 * @param  {Object/String} element element or query
	 */
	hasClass: function (className, element) {
		if (MAL.isUndefined(element)) {
			element = this.element;
		} else if (MAL.isString(element)) {
			element = this.query(element);
		}

		return MAL.stage.getLayer(element).hasClass(className, className.match(/^mal-/) ? '' : this.prefix);
	},

	/**
	 * adds a class to the elements
	 *
	 * @param  {String} className
	 * @param  {Object/String/Array} elements elements, query, or layer
	 */
	addClass: function (className, elements) {
		if (MAL.isUndefined(elements)) {
			elements = [this.element];
		} else if (MAL.isString(elements)) {
			elements = this.queryAll(elements);
		} else if (!MAL.isArray(elements)) {
			elements = [elements];
		}

		MAL.stage.getLayers(elements).addClass(className, className.match(/^mal-/) ? '' : this.prefix);
	},

	/**
	 * removes a class from the elements
	 *
	 * @param  {String} className
	 * @param  {Object/String/Array} elements elements, query, or layer
	 */
	removeClass: function (className, elements) {
		if (MAL.isUndefined(elements)) {
			elements = [this.element];
		} else if (MAL.isString(elements)) {
			elements = MAL.toArray(this.queryAll(elements));
		} else if (!MAL.isArray(elements)) {
			elements = [elements];
		}

		MAL.stage.getLayers(elements).removeClass(className, className.match(/^mal-/) ? '' : this.prefix);
	},

	/**
	 * converts HTML string into property prefixed DOM
	 *
	 * @param  {String} html
	 * @return {Object} element
	 */
	createElement: function (html) {
		var processed = MAL.processHTML(html, this._imagePath, this.prefix);

		this._DOMContainer.innerHTML = processed.html;

		var elements = MAL.toArray(this._DOMContainer.childNodes);

		this._DOMContainer.removeChild(elements[0]);

		this._images.add(processed.images.clone());

		return elements[0];
	}
}));