/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * @class MAL.tween.CSSLayer
 * @extends MAL.tween.Layer
 *
 * **You should not have to instantiate this class directly, it is managed by {@link MAL.tween.Stage}.**
 *
 * # Layers
 *
 * we have a standard of aliasing the layer getter with $
 *     var $ = stage.getLayers;
 *
 * obtaining a layer (this actually creates a Layer object if one does not already exist)
 *     $('box');
 *
 * obtaining multiple layers (this will create a layer proxy object)
 *     $('box', 'box2');
 *
 * setting a property on a layer
 *     $('box').setProperty('opacity', 0);
 *
 * setting a property on multiple layers
 *     $('box', 'box2').setProperty('opacity', 0);
 *
 * logging out the selected layers out of the proxy
 *
 *     $('box', 'box2').each(function (layer) {
 *      console.log(layer);
 *     });
 *
 * if for some reason you cant to find out if a layer has went through a proxy you can do this
 *     document.getElementById('mal-ad-box').MALTweenLayer; // this property would reference the layer instance
 *
 * # Helpers
 *
 * you can also string things
 *
 *     $('box', 'box2').clearAll().addTween({duration: 1, finish: {x: 100}});
 *
 * # Animating on a layer
 *
 * we only want to do one animation with this layer so we use {@link #addTween}
 *     layer.addTween({duration: 1, finish: {x: 100}}); // animate layer to {x:100} in a second
 *
 * but if we want to do a set of animations in a specific order we would use {@link #addTweenSet}
 *
 *     layer.addTweenSet(
 *      {duration: 1, finish: {x: 100}},
 *      {duration: 1, finish: {x: 0}},
 *      {duration: 1, finish: {y: 100}},
 *      {duration: 1, finish: {y: 0}}
 *     );
 *
 * here is a runnable example of adding a tween set
 *
 *     @example
 *     // add a layer (there is an element already in the dom with an ID of mal-ad-box)
 *     // we are also setting some initial css properties
 *     var layer = stage.addLayer('box', {width: 25, height: 25, 'background-color': '#085083'});
 *
 *     // you can also add multiple tweens like this
 *     layer.addTweenSet(
 *
 *         // wait 500 ms then animate to {x:500} with an ease of 'ease-in-out'
 *         {delay: .5, duration: .5, finish: {x: 500}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {x: 0, y: 100}
 *         {duration: .5, finish: {x: 0, y: 100}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {rotate: 45}
 *         {duration: .5, finish: {rotate: 45}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {x: 500, y: 0}
 *         {duration: .5, finish: {x: 500, y: 0}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {x: 0} then disappear
 *         {duration: .5, finish: {x: 0, y: 0}, easing: 'ease-in-out', callback: function (layer) {
 *          layer.setProperty('visibility', 'hidden');
 *         }}
 *
 *     );
 *
 * # Dealing with bugs and features
 *
 * all layers will inherit the bugs/features settings that are set through the {@link MAL.Environment}, but you can override them on a per layer basis with {@link #setFeatures} and {@link #setBugs}.
 *
 * because a lot of our animations depend on the bleeding edge
 */

/**
 * dispatched when an animation has been added
 *
 * @event animation-add
 * @param {Event} this
 */

/**
 * dispatched when an animation has started
 *
 * @event animation-start
 * @param {Event} this
 */

/**
 * dispatched when an animation has ended
 *
 * @event animation-end
 * @param {Event} this
 */
MAL.define('tween.CSSLayer', MAL.tween.Layer.extend({
	/**
	 * initializes MALTweenCSSLayer
	 *
	 * @method constructor
	 */

	/**
	 * @cfg {MAL.tween.Stage} stage that the layer will be added to
	 */

	/**
	 * @cfg {HTMLElement} element direct element reference
	 */

	/**
	 * @cfg {Object} begin properties that the layer should begin with
	 */

	/**
	 * @cfg {HTMLElement} positionAnchorPointElement (optional)
	 */
	initialize: function ($config) {
		this._super($config);

		// type of layer
		this.layerType = 'CSS';

		// establish element association
		this.element = $config.element;
		this.element.MALTweenLayer = this;

		// layer name / ref
		this.name = this.element.id ? this.element.id : this.element;

		// animation iteration count
		this.iteration = 0;

		// positioning element
		this._positionAnchorPointElement = $config.positionAnchorPointElement;

		/**
		 * @property {Boolean} useAnchorPointOffset
		 * If set to true the layer will always add the positionAnchorPointOffset to x/y movement
		 */
		this.useAnchorPointOffset = false;

		/**
		 * @property {Object} positionAnchorPointOffset
		 * Manages the positionAnchorPointOffset, you an get the x offset by using layer.positionAnchorPointOffset.x / layer.positionAnchorPointOffset.y
		 */
		this.positionAnchorPointOffset = {x: 0, y: 0};

		if (this._positionAnchorPointElement) {
			var anchorPointElementPosition = MAL.findElementPosition(this._positionAnchorPointElement),
				elementPosition = MAL.findElementPosition(this.element);

			this.positionAnchorPointOffset.x = anchorPointElementPosition.x - elementPosition.x;
			this.positionAnchorPointOffset.y = anchorPointElementPosition.y - elementPosition.y;
		}

		this._transformationsDisabled = false;

		// features and bugs
		this._initializedFixes = false;
		this._features = {};
		this._bugs = {};
		this._initializeBugsAndFeatures();

		// sets a reference to the parent stage
		this._stage = $config.stage;

		// animation state
		this._animating = false;

		// signature for override verification
		this._signature = null;

		// persistant callback
		this._persistantCallback = null;

		// holds all tweens
		this._tweens = [];

		// started
		this._started = false;

		// cached properties
		this._cachedProperties = {};

		// css property buffer
		this._bufferedCSSProperties = {};

		// current applied transformations
		this._transformations = {};

		// properties that have been animated on
		this._animatedProperties = {};

		/** @private empty transition callback */
		this._transitionCallback = function () {};

		// get initial transformations and copy them into the layer
		// TODO: look into removing this.. as it doesn't do anything currently
		//this._initialTransformations = this._getComputedTransformations();

		// cached properties
		this._initialProperties = {
			left: this.cssGetComputedProperty('left'),
			top: this.cssGetComputedProperty('top'),
			width: this.cssGetComputedProperty('width'),
			height: this.cssGetComputedProperty('height')
		};

		// startWithTimeout
		this._startWithTimeout = null;

		// default anchor point (used for manual position calculations)
		this._transformOrigin = {x: '50%', y: '50%', z: 0};

		// should we be using backface on the layer?
		this._backface = false;

		// establish event listener (webkit is still using a browser prefix)
		MAL.addEvent(this.element, MAL.Environment.transitionEnd, this._onTransitionEnd);

		// adds a small amount of rotation to all elements to smooth animations
//		if (MAL.Environment.features.rotationPerformanceFix) this._transformations.rotate = '.01deg';

		// set begin properties
		if ($config.begin) this.setProperties($config.begin);
	},

	// ensures that the function operations happen in this order
	_transformationFunctionOrder: [
		'perspective',

		'preRotate',
		'preRotateX',
		'preRotateY',
		'preRotateZ',

		'translateX',
		'translateY',
		'translateZ',

		'skew',
		'skewX',
		'skewY',

		'scale',
		'scaleX',
		'scaleY',
		'scaleZ',

		'rotate',
		'rotateX',
		'rotateY',
		'rotateZ',

		'postX',
		'postY',
		'postZ'
	],

	// custom transformation function aliases
	_transformationFunctionAlias: {
		preRotate: 'rotate',
		preRotateX: 'rotateX',
		preRotateY: 'rotateY',
		preRotateZ: 'rotateZ',

		postX: 'translateX',
		postY: 'translateY',
		postZ: 'translateZ'
	},

	// property suffix list
	_suffixes: {
		perspective: 'px',
		width: 'px',
		height: 'px',
		left: 'px',
		top: 'px',
		skew: 'deg',
		skewX: 'deg',
		skewY: 'deg',
		x: 'px',
		y: 'px',
		z: 'px',
		preRotate: 'deg',
		preRotateX: 'deg',
		preRotateY: 'deg',
		preRotateZ: 'deg',
		rotate: 'deg',
		rotateX: 'deg',
		rotateY: 'deg',
		rotateZ: 'deg',
		postX: 'px',
		postY: 'px',
		postZ: 'px'
	},

	// property dictionary (used for ensuring default values)
	_transformationPropertyDictionary: {
		'x': {name: 'translateX', value: 0},
		'y': {name: 'translateY', value: 0},
		'z': {name: 'translateZ', value: 0},
		'transformOrigin': {name: 'transformOrigin', value: {x: '50%', y: '50%', z: 0}},
		'transformPerspective': {name: 'transformPerspective', value: 0},
		'skew': {name: 'skew', value: 0},
		'skewX': {name: 'skewX', value: 0},
		'skewY': {name: 'skewY', value: 0},
		'scale': {name: 'scale', value: 1},
		'scaleX': {name: 'scaleX', value: 1},
		'scaleY': {name: 'scaleY', value: 1},
		'scaleZ': {name: 'scaleZ', value: 1},
		'preRotate': {name: 'preRotate', value: 0},
		'preRotateX': {name: 'preRotateX', value: 0},
		'preRotateY': {name: 'preRotateY', value: 0},
		'preRotateZ': {name: 'preRotateZ', value: 0},
		'rotate': {name: 'rotate', value: 0},
		'rotateX': {name: 'rotateX', value: 0},
		'rotateY': {name: 'rotateY', value: 0},
		'rotateZ': {name: 'rotateZ', value: 0},
		'postX': {name: 'postX', value: 0},
		'postY': {name: 'postY', value: 0},
		'postZ': {name: 'postZ', value: 0}
	},

	/**
	 * copies the enviorments default bugs and features
	 *
	 * @private
	 */
	_initializeBugsAndFeatures: function () {
		for (var feature in MAL.Environment.features) this._features[feature] = MAL.Environment.features[feature];
		for (var bug in MAL.Environment.bugs) this._bugs[bug] = MAL.Environment.bugs[bug];
	},

	/**
	 * this should be called the first time we try to interface with the object
	 *
	 * @private
	 */
	_initializeFixes: function () {
		if (MAL.Environment.has3d && this._features.translateZPerformanceFix && MAL.isUndefined(this._transformations.translateZ)) this._transformations.translateZ = '0px';
		this._initializedFixes = true;
	},

	/**
	 * enables or disables bugs for the layer
	 *
	 *     layer.setBugs({
	 *      transform3d: true,
	 *      scale: true
	 *     });
	 *
	 * @param {Object} bugs
	 */
	setBugs: function (bugs) {
		for (var bug in bugs) this._bugs[bug] = bugs[bug];
		return this;
	},

	/**
	 * enables or disables features for the layer
	 *
	 *     layer.setFeatures({
	 *      backfacePerformanceFix: false,
	 *      translateZPerformanceFix: false
	 *     });
	 *
	 * @param {Object} features
	 */
	setFeatures: function (features) {
		for (var feature in features) this._features[feature] = features[feature];
		return this;
	},

	/**
	 * add a tween to a layer
	 *
	 *     layer.addTween({duration: 1, finish: {x: 100}});
	 *
	 * @param {MAL.tween.TweenObject} tween
	 */
	addTween: function (options) {
		if (this._transformationsDisabled) MAL.error('BadCommand: layer is currently disabled "' + this.name + '"');

		this._tweens.push(options);

		if (this._stage.started && !this._started) {
			this._started = true;
			this._startNextAnimation();
		}

		this.dispatchEvent('animation-add');
	},

	/**
	 * add tweens to a layer
	 *
	 *     layer.addTweenSet(
	 *      {duration: 1, finish: {x: 100}},
	 *      {duration: 1, finish: {x: 0}},
	 *      {duration: 1, finish: {y: 100}},
	 *      {duration: 1, finish: {y: 0}}
	 *     );
	 *
	 * @param {MAL.tween.TweenObject...} tweens
	 */
	addTweenSet: function (tweens) {
		// convert arguments into an array
		tweens = MAL.argumentsToArray(arguments);

		// add tweens
		tweens.forEach(function (tween) {
			this.addTween(tween);
		}, this);
	},

	/**
	 * add a persistant callback, usually used to dynamic animations like this
	 *
	 *     layer.addTweenPersistantCallback(function (layer) {
	 *      // this get fired after every tween
	 *      if (layer.iteration <= 5) {
	 *       // add a tween with an alternating finish animation based on the next iteration
	 *       layer.addTween({
	 *        duration: .2,
	 *        finish: {
	 *         x: this.iteration % 2 ? 50 : 0
	 *        }
	 *       });
	 *      }
	 *     });
	 *
	 * @param {Function} callback
	 */
	addTweenPersistantCallback: function (callback) {
		callback(this);
		this._persistantCallback = callback;
	},

	/**
	 * add a tween that overrides all other tweens
	 *
	 * @param {MAL.tween.TweenObject} tween
	 */
	addTweenOverride: function (options) {		
		this.clearTweens();
		this.addTween(options);
		this._startNextAnimation();
	},

	/**
	 * add a tween set overrides all other tweens
	 *
	 * @param {MAL.tween.TweenObject...} tweens
	 */
	addTweenSetOverride: function (tweens) {
		// convert arguments into an array
		tweens = MAL.argumentsToArray(arguments);

		this._clearTransformations();		
		this.clearTweens();
		this.addTweenSet(tweens);
		this._startNextAnimation();
	},

	/**
	 * add a persistant callback that overrides all other tweens
	 *
	 * @param {Function} callback
	 */
	addTweenPersistantCallbackOverride: function (callback) {
		this.clearTweens();
		this.addTweenPersistantCallback(callback);
	},

	/**
	 * remove all remaining animations from the layer
	 *
	 *     layer.clearTweens();
	 *
	 */
	clearTweens: function () {
		this._started = false;
		this._animating = false;
		this._signature = null;
		this._tweens = [];
		this._cachedProperties = {};
	},

	/**
	 * remove all animations from the layer, and returns current layer
	 *
	 * @return {MAL.tween.CSSLayer} layer
	 */
	clearAll: function () {
		this.clearTweens();
		this._persistantCallback = null;
		this._clearTransformations();
		this._setCSSTextFromBuffer();

		// This fixes issues when adding another tween onto a layer thats been cleared of transformations.
		this._transformations = {};

		return this;
	},

	/**
	 * set layer transform origin
	 *
	 * @private
	 * @param {Object} origin X and Y origin
	 */
	_setLayerTransformOrigin: function (origin) {
		if (origin.x) origin.x = origin.x + (typeof origin.x === 'number' ? 'px' : '');
		if (origin.y) origin.y = origin.y + (typeof origin.y === 'number' ? 'px' : '');

		origin = MAL.mergeOptions(this._transformOrigin, origin);

		// set it to the CSS
		this._css('transform-origin', origin.x + ' ' + origin.y + (MAL.Environment.has3d ? ' ' + origin.z + 'px' : ''));

		// set it in the layer, so that we can do manual transforms
		this._transformOrigin = MAL.mergeOptions(this._transformOrigin, origin);
	},

	/**
	 * get layer transform origin
	 *
	 * @private
	 * @return {Object} origin
	 */
	_getLayerTransformOrigin: function () {
		return this._transformOrigin;
	},

	/**
	 * transform positioning
	 *
	 * @private
	 * @param {Number} value
	 */
	_setLayerX: function (value) {
		if (MAL.isString(value)) return this._transformations.translateX = value;

		var offset = this.useAnchorPointOffset ? this.positionAnchorPointOffset.x : 0;

		this._transformations.translateX = (value - this._initialProperties.left + offset) + this._suffixes.x;
	},
	_setLayerY: function (value) {
		if (MAL.isString(value)) return this._transformations.translateY = value;

		var offset = this.useAnchorPointOffset ? this.positionAnchorPointOffset.y : 0;

		this._transformations.translateY = (value - this._initialProperties.top + offset) + this._suffixes.y;
	},

	_getLayerX: function () {
		var offset = this.useAnchorPointOffset ? this.positionAnchorPointOffset.x : 0;
		return this._parsePropertyValue(this._transformations.translateX) + this._initialProperties.left - offset;
	},
	_getLayerY: function () {
		var offset = this.useAnchorPointOffset ? this.positionAnchorPointOffset.y : 0;
		return this._parsePropertyValue(this._transformations.translateY) + this._initialProperties.top - offset;
	},

	/**
	 * transform scaling
	 *
	 * @private
	 * @param {Number} value
	 */
	_setLayerScale: function (value) {
		// check for native support
		if (this._bugs.scale) {
			// cache value
			this._transformations._scale = value;

			// apply scales
			this._setLayerScaleX(value, true);
			this._setLayerScaleY(value, true);
		} else {
			this._transformations.scale = value;
		}
	},
	_setLayerScaleX: function (value) {
		// check for native support
		if (this._bugs.scale) {
			// not that awesome but its a middleground
			var width = this._initialProperties.width * value,
				left = this._initialProperties.left - (MAL.delta(this._initialProperties.width, width) * this._transformOrigin.x);

			// apply changed properties to cache
			this.setProperties({
				left: left,
				width: width
			}, true);

			// cache value
			this._transformations._scaleX = value;
		} else {
			this._transformations.scaleX = value;
		}
	},
	_setLayerScaleY: function (value) {
		// check for native support
		if (this._bugs.scale) {
			// not that awesome but its a middleground
			var height = this._initialProperties.height * value,
				top = this._initialProperties.top - (MAL.delta(this._initialProperties.height, height) * this._transformOrigin.y);

			// apply changed properties to cache
			this.setProperties({
				top: top,
				height: height
			}, true);

			// cache value
			this._transformations._scaleY = value;
		} else {
			this._transformations.scaleY = value;
		}
	},

	_getLayerScale: function () {
		// check for native support
		var value = this._bugs.scale ? this._transformations._scale : this._transformations.scale;

		return MAL.defaultValue(1, value);
	},
	_getLayerScaleX: function () {
		// check for native support
		var value = this._bugs.scale ? this._transformations._scaleX : this._transformations.scaleX;

		return MAL.defaultValue(1, value);
	},
	_getLayerScaleY: function () {
		// check for native support
		var value = this._bugs.scale ? this._transformations._scaleY : this._transformations.scaleY;

		return MAL.defaultValue(1, value);
	},

	_setLayerTransformPerspective: function (value) {
		this._transformations.perspective = value;
	},

	_getLayerTransformPerspective: function (value) {
		return this._transformations.perspective;
	},

	/**
	 * starts the next animation
	 *
	 * @private
	 */
	_startNextAnimation: function () {
		var tween = this._tweens.shift();
		if (tween) {
			for (var property in tween.finish) {
				var propertyInfo = this._transformationPropertyDictionary[property];

				if (this._transformationPropertyDictionary[property] && MAL.isUndefined(this._transformations[propertyInfo.name])) {
					this._transformations[propertyInfo.name] = propertyInfo.value;
				}
			}

			this._css('transform', this._constructTransformationString());
			this._setCSSTextFromBuffer();

			// we have to do this incase someone runs an addTween right after an addLayer with start transforms
			this._stage.timeouts.addTimeout((function (self, tween) {
				return function () {
					self._signature = new Date().getTime() * Math.random();
					self._applyAnimation.apply(self, [self._signature, tween]);
				};
			}(this, tween)), 0);

			this._started = true;
		} else {
			this._started = false;
		}
	},

	/**
	 * apply an animation to the layer
	 *
	 * @private
	 * @param {Object} options
	 */
	_applyAnimation: function (signature, options) {
		// if signature missmatch, animation has been overridden
		if (signature !== this._signature) return;

		// set start properties
		if (options.delay) {
			// complex timeout closure...
			this._stage.timeouts.addTimeout((function (self, signature, options) {
				return function () {
					// remove delay
					options.delay = 0;

					// apply animation with start values
					self._applyAnimation.apply(self, [signature, options]);
				};
			}(this, signature, options)), options.delay * 1000);

			// break chain
			return;
		} else if (options.start) {
			// no delay
			this._setStartProperties(options.start);

			this._stage.timeouts.addTimeout((function (self, signature, options) {
				return function () {
					options.start = false;
					self._applyAnimation.apply(self, [signature, options]);
				};

				// TODO: look into the delay bug for FF (http://jsfiddle.net/uaukc/2/)
			}(this, signature, options)), 25);

			return;
		}

		// reestablish options object (prevents undefined errors)
		if (!options) options = {finish:false};

		// set transitionCallback if passed
		if (options.callback) this._transitionCallback = options.callback;

		// set finish properties
		var finishProperties = options.finish || {};

		// set layers animation state
		this._animating = true;

		// increment iteration count
		this.iteration++;

		// clear
		this._clearTransformations();

		// record actual property changes
		var changes = 0;

		for (var name in finishProperties) {
			var property = finishProperties[name],
				value;

			// check for advanced options
			if (MAL.isObject(property) && property.value) {
				value = property.value;
			} else {
				value = property;
			}

			// break out of set command, nothing has changed
			if (this._cachedProperties[name] === value) continue;

			// set property

			this.setProperty(name, value);

			// record changes (we do this to allow us to actually account for an empty animation)
			changes++;
		}

		// reset
		this._setTransformations(finishProperties, options, true);

		// fire animation start
		this.dispatchEvent('animation-start');

		// throw error if a tween was ran that actually does nothing...
		//if (options.finish && !changes) MAL.error('BadTween: being applyed to "' + this.name + '"');

		// apply changes
		this._setCSSTextFromBuffer();

		// instantly fire animation end (this should prolly be fixed)
		if (MAL.isUndefined(options.finish)) {
			//this.dispatchEvent('animation-end');

			// if we are only doing a "start" and there are no more tweens left
			if (!this._tweens.length) this._started = false;
		}
	},


	/**
	 * clears css transitions and transform
	 *
	 * @private
	 */
	_clearTransformations: function () {
		// clear
		this._css({
			'transform': null,
			'transition': 'none',
			'transition-property': null,
			'transition-delay': null,
			'transition-duration': null,
			'transition-timing-function': null
		});

		// clear webkit specific properties
		if (MAL.Environment.has3d && !this._bugs.transform3d) {
			if (this._features.backfacePerformanceFix && !this._backface) this._css('backface-visibility', null);
			if (this._features.perspectivePerformanceFix) this._css('perspective', null);
		}
	},

	/**
	 * sets css transitions and transform
	 *
	 * @private
	 * @param {Object} properties
	 * @param {Object} options
	 */
	_setTransformations: function (properties, options, transition) {
		if (transition) {
			// construct transition string
			transition = this._constructTransition(properties, options);

			// reset
			this._css({
				'transform': this._constructTransformationString(),
				'transition-property': transition.property,
				'transition-delay': '0s',
				'transition-duration': transition.duration,
				'transition-timing-function': transition.easing
			});

			// reset webkit specific properties
			if (MAL.Environment.has3d && !this._bugs.transform3d) {
				if (this._features.backfacePerformanceFix && !this._backface) this._css('backface-visibility', 'hidden'); // fixes flickers for elements that do not utilize backfaces
				if (this._features.perspectivePerformanceFix) this._css('perspective', 1000); // fixes aliasing performance issues
			}

		} else {
			// reset
			this._css({
				'transform': this._constructTransformationString()
			});
		}
	},

	/**
	 * removes transformations (this is used to prevent flickering in some cases)
	 */
	disableTransformations: function () {
		if (this._animating) MAL.error('BadCommand: layer is currently animating "' + this.name + '"');
		this._transformationsDisabled = true;
		this._css({'transform': this._constructTransformationString()});
		this._clearTransformations();
		this._setCSSTextFromBuffer();
		return this;
	},

	/**
	 * restore transformations
	 */
	enableTransformations: function () {
		this._transformationsDisabled = false;
		this._css({'transform': this._constructTransformationString()});
		this._setCSSTextFromBuffer();
		return this;
	},

	/**
	 * sets start properties
	 *
	 * @private
	 * @param {Object} properties
	 */
	_setStartProperties: function (properties) {
		this._clearTransformations();

		for (var name in properties) {
			this.setProperty(name, properties[name]);
		}

		this._setTransformations(properties, {});
	},

	/**
	 * set multiple properties on a layer
	 *
	 *     layer.setProperties({
	 *      opacity: .5,
	 *      x: 100,
	 *      y: 100,
	 *      'background-color': '#000000'
	 *     });
	 *
	 * refer to {@link MAL.tween.TransformObject} for possible properties
	 *
	 * @param {Object} properties
	 */
	setProperties: function (properties, dontApplyToDOM) {
		for (var property in properties)
			this.setProperty(property, properties[property], true);

		// regenerate the transform string
		if (!dontApplyToDOM) {
			this._css('transform', this._constructTransformationString());
			this._setCSSTextFromBuffer();
		}

		return this;
	},

	/**
	 * set a layers property
	 *
	 *     layer.setProperty('rotate', 45);
	 *
	 * refer to {@link MAL.tween.TransformObject} for possible properties
	 *
	 * @param {String} name
	 * @param {String/Number} value
	 */
	setProperty: function (name, value, dontApplyToDOM) {
		if (!this._initializedFixes) this._initializeFixes();

		// method is used to detect if we need any additional setProperty overrides
		var method = MAL.toCamelCaseWithPrefix('_setLayer', name);

		// cache new property value
		this._cachedProperties[name] = value;

		if (this._transformationPropertyDictionary[name]) {
			// set advanced property
			if (this[method]) {
				this[method](value);
			} else {
				var transform = this._transformationPropertyDictionary[name];
				this._transformations[transform.name] = MAL.defaultValue(transform.value, value) + (this._suffixes[name] || '');
			}
		} else {
			// set basic styling property
			this._css(name, value);
		}

		// regenerate the transform string
		if (!dontApplyToDOM) {
			this._css('transform', this._constructTransformationString());
			this._setCSSTextFromBuffer();
		}

		return this;
	},

	/**
	 * get a layers property value
	 *
	 *     layer.getProperty('rotate');
	 *
	 * @param {String} name
	 */
	getProperty: function (name) {
		var method = MAL.toCamelCaseWithPrefix('_getLayer', name),
			value;

		if (this._transformationPropertyDictionary[name]) {
			// get transform property
			value = this[method] ? this[method]() : this._transformations[this._transformationPropertyDictionary[name].name];
		} else {
			// get basic styling property
			value = this._css(name);
		}

		return this._parsePropertyValue(value);
	},

	/**
	 * get layers current position after transformations and positioning
	 *
	 *     layer.getPosition();
	 *
	 * @return {Object} {x: Float, y: Float}
	 */
	getPosition: function () {
		return {
			x: this.cssGetComputedProperty('left') + parseFloat(this._getLayerX()),
			y: this.cssGetComputedProperty('top') + parseFloat(this._getLayerY())
		};
	},

	/**
	 * produces a transformation property string from the _transformations object
	 *
	 * @private
	 */
	_constructTransformationString: function () {
		if (this._transformationsDisabled) return null;

		var string = '';

		this._transformationFunctionOrder.forEach(function (transformation) {
			if (transformation.substr(0,1) === '_' || MAL.isUndefined(this._transformations[transformation])) return;

			// look up alias name
			var name = this._transformationFunctionAlias[transformation] ? this._transformationFunctionAlias[transformation] : transformation;

			string += ' ' + name + '(' + this._transformations[transformation] + ')';
		}, this);

		return string.substr(1);
	},

	/**
	 * produces a transformation property object from a transform string
	 *
	 * @private
	 */
	_constructTransformationObject: function (string) {
		var match,
			transformation = {},
			pattern = new RegExp(/([a-z0-9]+)\((.+?)\)/gi);

		while ((match = pattern.exec(string))) transformation[match[1]] = match[2];

		return transformation;
	},

	/**
	 * produces a transition object
	 *
	 * @private
	 * @param {Object} properties
	 * @param {Object} options
	 */
	_constructTransition: function (properties, options) {
		// option defaults
		var defaultDuration = options.duration || 0,
			defaultDelay = options.delay || 0,
			defaultTiming = options.easing || [0,0,1,1],
			transformSettings = null;

		// transition strings
		var transition = {
			property: '',
			delay: '',
			duration: '',
			easing: ''
		};

		for (var property in properties) {
			// local vars
			var duration,
				delay,
				easing,
				callback,
				value,
				info = properties[property],
				propertyName = MAL.Environment.getCSSPropertyName(property, true);

			if (MAL.isObject(info)) {
				// property has a custom transition
				duration = info.duration || defaultDuration;
				delay = info.delay || defaultDelay;
				easing = info.easing || defaultTiming;
				callback = info.callback || false;
			} else {
				// property has a default transition
				duration = defaultDuration;
				delay = defaultDelay;
				easing = defaultTiming;
				callback = false;
			}

			// externalize property ref
			if (this._transformationPropertyDictionary[property]) {
				// keep ref for abnormal transform operations
				transformSettings = {
					delay: delay,
					duration: duration,
					easing: easing,
					callback: callback
				};

				// set for normal transform operations
				if (!(/^scale/.test(property) && this._bugs.scale))
					this._animatedProperties[MAL.Environment.getCSSPropertyName('transform', true)] = transformSettings;

			} else if (!options.start || MAL.isUndefined(options.start[property])) {
				// add to elements transition list if they are not start options
				this._animatedProperties[propertyName] = {
					delay: delay,
					duration: duration,
					easing: easing,
					callback: callback
				};
			} else if (options.start && options.start[property]) {
				// remove element from _animatedProperties list
				delete this._animatedProperties[propertyName];
			}

			if (property === 'transformOrigin') {
				this.setProperty('transformOrigin', info, false);
			}

		}

		// determine if we overrode scale, and if we did we should be using w,h,x,y to simulate scale
		if (!MAL.isUndefined(this._transformations._scaleX)) this._animatedProperties.left = this._animatedProperties.width = transformSettings;
		if (!MAL.isUndefined(this._transformations._scaleY)) this._animatedProperties.top = this._animatedProperties.height = transformSettings;


		// compute elements transition list
		for (var name in this._animatedProperties) {
			transition.property += name + ',';
			transition.duration += this._animatedProperties[name].duration + 's,';
			transition.delay += this._animatedProperties[name].delay + 's,';
			transition.easing += (MAL.isArray(this._animatedProperties[name].easing) ? 'cubic-bezier(' + this._animatedProperties[name].easing.join(',') + ')' : this._animatedProperties[name].easing) + ',';
		}

		return {
			property: transition.property.slice(0, -1),
			delay: transition.delay.slice(0, -1),
			duration: transition.duration.slice(0, -1),
			easing: transition.easing.slice(0, -1)
		};
	},

	/**
	 * fired everytime a property transition ends
	 *
	 * @private
	 * @param {Event} e
	 */
	_onTransitionEnd: function (e) {
		// prevent events from bubbling
		e.stopPropagation();

		// we dont care about start animations
		if (!this._animating) return false;

		// fire property callback
		if (this._animatedProperties[e.propertyName] && this._animatedProperties[e.propertyName].callback) this._animatedProperties[e.propertyName].callback(this);

		// record that we are no longer animating on the property
		delete this._animatedProperties[e.propertyName];

		// cycle through uncompleted transitions
		var animations = 0;
		for (var name in this._animatedProperties) if (this._animatedProperties[name]) animations++;

		// fire transition completion event
		if (!animations) this._onTransitionCompleted();
	},

	/**
	 * fired when the first transition ends
	 *
	 * @private
	 */
	_onTransitionCompleted:  function () {
		if (!this._animating) return;

		// clear
		this._css({
			'transition-delay': null,
			'transition-duration': null,
			'transition-timing-function': null
		});

		// apply changes
		this._setCSSTextFromBuffer();

		// set layers animation state
		this._animating = false;

		// fire transition callback
		if (this._transitionCallback) {
			this._transitionCallback(this);
			this._transitionCallback = false;
		}

		// fire persistant callback
		if (this._persistantCallback) this._persistantCallback(this);

		// dispatch transition completion event
		this.dispatchEvent('animation-end');

		// start next animation
		this._startNextAnimation();
	},



	/**
	 * sets and gets css property values
	 *
	 * @private
	 * @param {String} property name of the property
	 * @param {String/Number} value value of property
	 * @param {Object} properties set of properties
	 */
	_css: function () {
		if (!MAL.isUndefined(arguments[0]) && !MAL.isUndefined(arguments[1])) {
			// set specific value
			this._cssSetProperty(arguments[0], arguments[1]);
		} else if (MAL.isString(arguments[0])) {
			// return specific value
			return this._cssGetProperty(arguments[0]);
		} else if (MAL.isObject(arguments[0])) {
			// set multiple properties
			this._cssSetProperties(arguments[0]);
		}
	},

	/**
	 * sets the cssText of the layer (this is very IMPORTANT)
	 *
	 * @private
	 */
	_setCSSTextFromBuffer: function () {
		var string = '';

		for (var property in this._bufferedCSSProperties) {
			// do not add things that are strictly used for computing (filters)
			if (property.substr(0,1) !== '_') {
				string += property + ': ' + this._bufferedCSSProperties[property] + '; ';
			}
		}

		this.element.style.cssText = string;
	},

	/**
	 * sets a single css property
	 *
	 * @private
	 * @param {String} property name of the property
	 * @param {String/Number} value value of property
	 */
	_cssSetProperty: function (property, value) {
		// method is used to detect if we need any additional css overrides
		var method = MAL.toCamelCaseWithPrefix('_cssSet', property);

		// add browser prefix to properties that start with a dash
		var propertyName = MAL.Environment.getCSSPropertyName(property, true);

		if (this[method]) {
			// use override
			var override = this[method](value);

			// do we need to save it
			if (override) this._bufferedCSSProperties[override.name] = override.value;
		} else if (MAL.isNull(value)) {
			delete this._bufferedCSSProperties[propertyName];
		} else {
			// add suffix
			if (MAL.isNumber(value) && this._suffixes[property]) value = value + this._suffixes[property];

			this._bufferedCSSProperties[propertyName] = value;
		}
	},

	/**
	 * sets a the _backface boolean if set to visible, otherwise the layer may use it for performance gains
	 *
	 * @private
	 * @param {Object} properties set of properties
	 */
	_cssSetBackfaceVisibility: function (value) {
		if (value === 'visible') this._backface = true;

		return {
			name: MAL.Environment.prefix + 'backface-visibility',
			value: value
		};
	},

	/**
	 * sets a set of css properties
	 *
	 * @private
	 * @param {Object} properties set of properties
	 */
	_cssSetProperties: function (properties) {
		for (var property in properties) {
			this._cssSetProperty(property, properties[property]);
		}
	},

	/**
	 * gets the value of a css property
	 *
	 * @private
	 * @param {String} property name of css property
	 */
	_cssGetProperty: function (property, raw) {
		// method is used to detect if we need any additional css overrides
		var method = MAL.toCamelCaseWithPrefix('_cssGet', property);

		// add browser prefix to properties that start with a dash
		var propertyName = MAL.Environment.getCSSPropertyName(property, true);

		if (this[method]) {
			// use override
			return this[method](property);
		} else {
			// get basic style property
			return this.element.style[propertyName] || this.cssGetComputedProperty(propertyName, raw);
		}
	},

	/**
	 * attempts to retrieve the current transformations from the elements transformation matrix
	 *
	 * @private
	 */
	_getComputedTransformations: function () {
		var x = 0,
			y = 0,
			z = 0,
			matrix = new window.WebKitCSSMatrix(window.getComputedStyle(this.element).webkitTransform);

		if ('m33' in matrix) {
			x = matrix.e;
			y = matrix.f;
		} else {
			x = matrix.m41;
			y = matrix.m42;
			z = matrix.m43;
		}

		return {x: x, y: y};
	}
}));
MAL.tween.CSSLayerProxy = new MAL.tween.LayerProxy(MAL.tween.CSSLayer);