/**
 * @author Chad Scira <chad@mediaartslab.com>
 * @docauthor Chad Scira <chad@mediaartslab.com>
 *
 * @class MAL.tween.JSLayer
 * @extends MAL.tween.Layer
 *
 * **You should not have to instantiate this class directly, it is managed by {@link MAL.tween.Stage}.**
 *
 * # Layers
 *
 * we have a standard of aliasing the layer getter with $
 *     var $ = stage.getLayers;
 *
 * obtaining a layer (this actually creates a Layer object if one does not already exist)
 *     $('box');
 *
 * obtaining multiple layers (this will create a layer proxy object)
 *     $('box', 'box2');
 *
 * setting a property on a layer
 *     $('box').setProperty('opacity', 0);
 *
 * setting a property on multiple layers
 *     $('box', 'box2').setProperty('opacity', 0);
 *
 * if for some reason you cant to find out if a layer has went through a proxy you can do this
 *     document.getElementById('mal-ad-box').MALTweenLayer; // this property would reference the layer instance
 *
 * # Helpers
 *
 * you can also string things
 *
 *     $('box', 'box2').clearAll().addTween({duration: 1, finish: {x: 100}});
 *
 * # Animating on a layer
 *
 * we only want to do one animation with this layer so we use {@link #addTween}
 *     layer.addTween({duration: 1, finish: {x: 100}}); // animate layer to {x:100} in a second
 *
 * but if we want to do a set of animations in a specific order we would use {@link #addTweenSet}
 *
 *     layer.addTweenSet(
 *      {duration: 1, finish: {x: 100}},
 *      {duration: 1, finish: {x: 0}},
 *      {duration: 1, finish: {y: 100}},
 *      {duration: 1, finish: {y: 0}}
 *     );
 *
 * here is a runnable example of adding a tween set
 *
 *     @example
 *     // add a layer (there is an element already in the dom with an ID of mal-ad-box)
 *     // we are also setting some initial css properties
 *     var layer = stage.addLayer('box', {width: 25, height: 25, 'background-color': '#085083'});
 *
 *     // you can also add multiple tweens like this
 *     layer.addTweenSet(
 *
 *         // wait 500 ms then animate to {x:500} with an ease of 'ease-in-out'
 *         {delay: .5, duration: .5, finish: {x: 500}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {x: 0, y: 100}
 *         {duration: .5, finish: {x: 0, y: 100}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {rotate: 45}
 *         {duration: .5, finish: {rotate: 45}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {x: 500, y: 0}
 *         {duration: .5, finish: {x: 500, y: 0}, easing: 'ease-in-out'},
 *
 *         // wait 0 ms then animate to {x: 0} then disappear
 *         {duration: .5, finish: {x: 0, y: 0}, easing: 'ease-in-out', callback: function (layer) {
 *          layer.setProperty('visibility', 'hidden');
 *         }}
 *
 *     );
 *
 * # Dealing with bugs and features
 *
 * all layers will inherit the bugs/features settings that are set through the {@link MAL.Environment}, but you can override them on a per layer basis with {@link #setFeatures} and {@link #setBugs}.
 *
 * because a lot of our animations depend on the bleeding edge
 */

/**
 * dispatched when an animation has been added
 *
 * @event animation-add
 * @param {Event} this
 */

/**
 * dispatched when an animation has started
 *
 * @event animation-start
 * @param {Event} this
 */

/**
 * dispatched when an animation has ended
 *
 * @event animation-end
 * @param {Event} this
 */
MAL.define('tween.JSLayer', MAL.tween.Layer.extend({
	/**
	 * initializes MALTweenJSLayer
	 *
	 * @method constructor
	 * @param {Object} config
	 */

	/**
	 * @cfg {MAL.tween.Stage} stage that the layer will be added to
	 */

	/**
	 * @cfg {HTMLElement} element direct element reference
	 */

	/**
	 * @cfg {Object} begin properties that the layer should begin with
	 */
	initialize: function ($config) {
		this._super($config);

		// type of layer
		this.layerType = 'JS';

		// establish element association
		this.element = $config.element;
		this.element.MALTweenLayer = this;

		// layer name / ref
		this.name = this.element.id ? this.element.id : this.element;

		// features and bugs
		this._features = {};
		this._bugs = {};
		this._initializeBugsAndFeatures();

		// checks if we have transformation support
		this._transformationSupport = !!MAL.Environment.getCSSPropertyName('transform') && !this._bugs.transformations;

		// sets a reference to the parent stage
		this._stage = $config.stage;

		// animation state
		this._animating = false;

		// signature for override verification
		this._signature = null;

		// animation iteration count
		this.iteration = 0;

		// persistant callback
		this._persistantCallback = null;

		// holds all tweens
		this._tweens = [];

		// started
		this._started = false;

		// current tween
		this._currentTween = null;

		// css property buffer
		this._bufferedCSSProperties = {};

		// current applied transformations
		this._transformations = {};

		// current applied filters
		this._filters = {};
		this._filtersCache = {};

		// properties that have been animated on
		this._animatedProperties = {};

		/** @private empty transition callback */
		this._transitionCallback = function () {};

		// cached properties
		this._initialProperties = {
			left: this.cssGetComputedProperty('left'),
			top: this.cssGetComputedProperty('top'),
			width: this.cssGetComputedProperty('width'),
			height: this.cssGetComputedProperty('height')
		};

		// startWithTimeout
		this._startWithTimeout = null;

		// default anchor point (used for manual position calculations)
		this._transformOrigin = {};
		this._setLayerTransformOrigin({x: '50%', y: '50%'});

		this._transparency = true;

		// adds a small amount of rotation to all elements to smooth animations

		//if (this._features.rotationPerformanceFix) this._transformations.rotate = '.01deg';

		// set begin properties
		if ($config.begin) this.setProperties($config.begin);

	},

	// ensures that the filters happen in the specified order
	_filterOrder: [
		'DXImageTransform.Microsoft.Matrix',
		'DXImageTransform.Microsoft.Alpha'
	],

	// ensures that the function operations happen in this order
	_transformationFunctionOrder: [
		'preRotate',

		'translateX',
		'translateY',

		'skew',
		'skewX',
		'skewY',

		'scale',
		'scaleX',
		'scaleY',

		'rotate',

		'postX',
		'postY'
	],

	// custom transformation function aliases
	_transformationFunctionAlias: {
		preRotate: 'rotate',

		postX: 'translateX',
		postY: 'translateY'
	},

	// property suffix dictionary list
	_suffixes: {
		perspective: 'px',
		width: 'px',
		height: 'px',
		left: 'px',
		top: 'px',
		skew: 'deg',
		preRotate: 'deg',
		rotate: 'deg',
		x: 'px',
		y: 'px',
		postX: 'px',
		postY: 'px'
	},

	// property dictionary (used for ensuring default values)
	_transformationPropertyDictionary: {
		'x': {name: 'translateX', value: 0},
		'y': {name: 'translateY', value: 0},
		'transformOrigin': {name: 'transformOrigin', value: {x: '50%', y: '50%', z: 0}},
		'transformPerspective': {name: 'transformPerspective', value: 0},
		'skew': {name: 'skew', value: 0},
		'skewX': {name: 'skewX', value: 0},
		'skewY': {name: 'skewY', value: 0},
		'scale': {name: 'scale', value: 1},
		'scaleX': {name: 'scaleX', value: 1},
		'scaleY': {name: 'scaleY', value: 1},
		'preRotate': {name: 'preRotate', value: 0},
		'rotate': {name: 'rotate', value: 0},
		'postX': {name: 'postX', value: 0},
		'postY': {name: 'postY', value: 0}
	},

	// standard easing dictionary list
	_eases: {
		ease: [0.25, 0.1, 0.25, 1],
		linear: [0, 0, 1, 1],
		'ease-in': [0.42, 0, 1, 1],
		'ease-out': [0, 0, 0.58, 1],
		'ease-in-out': [0.42, 0, 0.58, 1]
	},

	/**
	 * copies the enviorments default bugs and features
	 *
	 * @private
	 */
	_initializeBugsAndFeatures: function () {
		for (var feature in MAL.Environment.features) this._features[feature] = MAL.Environment.features[feature];
		for (var bug in MAL.Environment.bugs) this._bugs[bug] = MAL.Environment.bugs[bug];
	},

	/**
	 * enables or disables bugs for the layer
	 *
	 *     layer.setBugs({
	 *      transform3d: true,
	 *      scale: true
	 *     });
	 *
	 * @param {Object} bugs
	 */
	setBugs: function (bugs) {
		for (var bug in bugs) this._bugs[bug] = bugs[bug];
		return this;
	},

	/**
	 * enables or disables features for the layer
	 *
	 *     layer.setFeatures({
	 *      backfacePerformanceFix: false,
	 *      translateZPerformanceFix: false
	 *     });
	 *
	 * @param {Object} features
	 */
	setFeatures: function (features) {
		for (var feature in features) this._features[feature] = features[feature];
		return this;
	},

	/**
	 * add a tween to a layer
	 *
	 *     layer.addTween({duration: 1, finish: {x: 100}});
	 *
	 * @param {MAL.tween.TweenObject} tween
	 */
	addTween: function (options) {
		this._tweens.push(options);

		if (this._stage.started && !this._started) {
			this._startNextAnimation();
		}

		this.dispatchEvent('animation-add');
	},

	/**
	 * add tweens to a layer
	 *
	 *     layer.addTweenSet(
	 *      {duration: 1, finish: {x: 100}},
	 *      {duration: 1, finish: {x: 0}},
	 *      {duration: 1, finish: {y: 100}},
	 *      {duration: 1, finish: {y: 0}}
	 *     );
	 *
	 * @param {MAL.tween.TweenObject...} tweens
	 */
	addTweenSet: function (tweens) {
		// convert arguments into an array
		tweens = MAL.argumentsToArray(arguments);

		// add tweens
		tweens.forEach(function(tween) {
			this.addTween(tween);
		}, this);
	},

	/**
	 * add a persistant callback, usually used to dynamic animations like this
	 *
	 *     layer.addTweenPersistantCallback(function (layer) {
	 *      // this get fired after every tween
	 *      if (layer.iteration <= 5) {
	 *
	 *       // add a tween with an alternating finish animation based on the next iteration
	 *       layer.addTween({
	 *        duration: .2,
	 *        finish: {
	 *         x: this.iteration % 2 ? 50 : 0
	 *        }
	 *       });
	 *
	 *      }
	 *     });
	 *
	 * @param {Function} callback
	 */
	addTweenPersistantCallback: function (callback) {
		callback(this);
		this._persistantCallback = callback;
	},

	/**
	 * add a tween that overrides all other tweens
	 *
	 * @param {MAL.tween.TweenObject} tween
	 */
	addTweenOverride: function (options) {
		this.clearTweens();
		this.addTween(options);
		this._startNextAnimation();
	},

	/**
	 * add a tween set overrides all other tweens
	 *
	 * @param {MAL.tween.TweenObject...} tweens
	 */
	addTweenSetOverride: function (tweens) {
		// convert arguments into an array
		tweens = MAL.argumentsToArray(arguments);

		this.clearTweens();
		this.addTweenSet(tweens);
		this._startNextAnimation();
	},

	/**
	 * add a persistant callback that overrides all other tweens
	 *
	 * @param {Function} callback
	 */
	addTweenPersistantCallbackOverride: function (callback) {
		this.clearTweens();
		this.addTweenPersistantCallback(callback);
	},

	/**
	 * remove all remaining animations from the layer
	 *
	 *     layer.clearTweens();
	 *
	 */
	clearTweens: function () {
		this._clearTemporaryCachedProperties();
		this._started = true;
		this._animating = false;
		this._signature = null;
		this._tweens = [];
		this._currentTween = null;
	},

	/**
	 * remove all animations from the layer, and returns current layer
	 *
	 * @return {MAL.tween.JSLayer} layer
	 */
	clearAll: function () {
		this._persistantCallback = null;
		this.clearTweens();
		return this;
	},

	/**
	 * flag layer as transparent (this is mainly used for some of our hacks in IE >9)
	 *
	 * @private
	 * @param {Boolean} transparent
	 */
	_setLayerTransparency: function (transparent) {
		this._transparency = transparent;
	},

	/**
	 * set layer transform origin
	 *
	 * @private
	 * @param {Object} origin X and Y origin
	 */
	_setLayerTransformOrigin: function (origin) {
		if (!this._transformationSupport || this._bugs.scale) {
			// turn pixels into percents
			if (!MAL.isUndefined(origin.x)) origin.x = typeof origin.x === 'number' ? (origin.x / this._initialProperties.width) : parseFloat(origin.x) / 100;
			if (!MAL.isUndefined(origin.y)) origin.y = typeof origin.y === 'number' ? (origin.y / this._initialProperties.height) : parseFloat(origin.y) / 100;
		} else {
			if (!MAL.isUndefined(origin.x)) origin.x = origin.x + (typeof origin.x === 'number' ? 'px' : '');
			if (!MAL.isUndefined(origin.y)) origin.y = origin.y + (typeof origin.y === 'number' ? 'px' : '');

			// set it to the CSS
			this._css('transform-origin', origin.x + ' ' + origin.y);
		}

		// set it in the layer, so that we can do manual transforms
		this._transformOrigin = MAL.mergeOptions(this._transformOrigin, origin);
	},

	/**
	 * get layer transform origin
	 *
	 * @private
	 * @return {Object} origin
	 */
	_getLayerTransformOrigin: function () {
		return this._transformOrigin;
	},

	/**
	 * transform positioning
	 *
	 * @private
	 * @param {Number} value
	 */
	_setLayerX: function (value) {
		if (this._transformationSupport) {
			// browsers that can handle translate

			// TODO: add support for % values
			this._transformations.translateX = (value - this._initialProperties.left) + this._suffixes.x;
		} else {
			// browsers that cant handle translate
			this._bufferedCSSProperties.left = value + this._suffixes.x;
		}
	},
	_setLayerY: function (value) {
		if (this._transformationSupport) {
			// browsers that can handle translate

			// TODO: add support for % values
			this._transformations.translateY = (value - this._initialProperties.top) + this._suffixes.y;
		} else {
			// browsers that cant handle translate
			this._bufferedCSSProperties.top = value + this._suffixes.y;
		}
	},

	_getLayerX: function () {
		if (this._transformationSupport) {
			// browsers that can handle translate
			return this._transformations.translateX ? this._initialProperties.left + MAL.parseNumber(this._transformations.translateX) : this._initialProperties.left;
		} else {
			// browsers that cant handle translate
			return this._bufferedCSSProperties.left ? MAL.parseNumber(this._bufferedCSSProperties.left) : this._initialProperties.left;
		}
	},
	_getLayerY: function () {
		if (this._transformationSupport) {
			// browsers that can handle translate
			return this._transformations.translateY ? this._initialProperties.top + MAL.parseNumber(this._transformations.translateY) : this._initialProperties.top;
		} else {
			// browsers that cant handle translate
			return this._bufferedCSSProperties.top ? MAL.parseNumber(this._bufferedCSSProperties.top) : this._initialProperties.top;
		}
	},

	/**
	 * transform scaling
	 *
	 * @private
	 * @param {Number} value
	 */
	_setLayerScale: function (value) {
		// check if browser doesn't support scale or has a scale bug
		if (!this._transparency && MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			this._setFilter('DXImageTransform.Microsoft.Matrix', 'M11=' + value + ', M21=0, M12=0, M22=' + value + ', SizingMethod="auto expand"', [value,value]);

			// apply everything to the DOM so that our offsets are correct
			this._setCSSTextFromCache();

			// buffer actual inital width/height
			if (!this._bufferedCSSProperties._initialProperties) this._bufferedCSSProperties._initialProperties = {width: this._parsePropertyValue(this.element.style.width), height: this._parsePropertyValue(this.element.style.height)};

			// set left according to scale
			this._bufferedCSSProperties.left = (this._initialProperties.left - (this.element.offsetWidth - this._bufferedCSSProperties._initialProperties.width) * this._transformOrigin.x) + 'px';

			// set top according to scale
			this._bufferedCSSProperties.top = (this._initialProperties.top - (this.element.offsetHeight - this._bufferedCSSProperties._initialProperties.height) * this._transformOrigin.y) + 'px';

		} else if (!this._transformationSupport || this._bugs.scale) {
			this._setLayerScaleX(value, true);
			this._setLayerScaleY(value, true);

			// cache value
			this._transformations._scale = value;
		} else {
			this._transformations.scale = value;
		}
	},
	_setLayerScaleX: function (value, doNotSetTransform) {
		if (!this._transparency && MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			// browser can use filters

			// matrix scaling
			var matrix = this._getFilter('DXImageTransform.Microsoft.Matrix'),
				scaleY = matrix ? matrix[1] : 1;

			this._setFilter('DXImageTransform.Microsoft.Matrix', 'M11=' + value + ', M21=0, M12=0, M22=' + scaleY + ', SizingMethod="auto expand"', [value,scaleY]);

			// apply everything to the DOM so that our offsets are correct
			this._setCSSTextFromCache();

			// buffer actual inital width/height
			if (!this._bufferedCSSProperties._initialProperties) this._bufferedCSSProperties._initialProperties = {width: this._parsePropertyValue(this.element.style.width), height: this._parsePropertyValue(this.element.style.height)};

			// set left according to scale
			this._bufferedCSSProperties.left = (this._initialProperties.left - (this.element.offsetWidth - this._bufferedCSSProperties._initialProperties.width) * this._transformOrigin.x) + 'px';
		} else if (!this._transformationSupport || this._bugs.scale) {
			// browser doesn't support scale or has a scale bug

			// not that awesome but its a middleground
			var width = this._initialProperties.width * value,
				left = this._initialProperties.left - (MAL.delta(this._initialProperties.width, width) * this._transformOrigin.x);

			// apply changed properties to cache
			this._bufferedCSSProperties.left = left + 'px';
			this._bufferedCSSProperties.width = width + 'px';

			// cache value
			this._transformations._scaleX = value;
		} else {
			this._transformations.scaleX = value;
		}
	},
	_setLayerScaleY: function (value, doNotSetTransform) {
		if (!this._transparency && MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			// browser can use filters

			// matrix scaling
			var matrix = this._getFilter('DXImageTransform.Microsoft.Matrix'),
				scaleX = matrix ? matrix[0] : 1;

			this._setFilter('DXImageTransform.Microsoft.Matrix', 'M11=' + scaleX + ', M21=0, M12=0, M22=' + value + ', SizingMethod="auto expand"', [scaleX,value]);

			// apply everything to the DOM so that our offsets are correct
			this._setCSSTextFromCache();

			// buffer actual inital width/height
			if (!this._bufferedCSSProperties._initialProperties) this._bufferedCSSProperties._initialProperties = {width: this._parsePropertyValue(this.element.style.width), height: this._parsePropertyValue(this.element.style.height)};

			// set top according to scale
			this._bufferedCSSProperties.top = (this._initialProperties.top - (this.element.offsetHeight - this._bufferedCSSProperties._initialProperties.height) * this._transformOrigin.y) + 'px';
		} else if (!this._transformationSupport || this._bugs.scale) {
			// browser doesn't support scale or has a scale bug

			// not that awesome but its a middleground
			var height = this._initialProperties.height * value,
				top = this._initialProperties.top - (MAL.delta(this._initialProperties.height, height) * this._transformOrigin.y);

			// apply changed properties to cache
			this._bufferedCSSProperties.top = top + 'px';
			this._bufferedCSSProperties.height = height + 'px';

			// cache value
			this._transformations._scaleY = value;
		} else {
			this._transformations.scaleY = value;
		}
	},

	_getLayerScale: function () {
		if (!this._transparency && MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			var matrix = this._getFilter('DXImageTransform.Microsoft.Matrix');
			return matrix ? matrix[0] : 1;
		} else {
			// check for native support
			var value = !this._transformationSupport || this._bugs.scale ? this._transformations._scale : this._transformations.scale;

			return MAL.defaultValue(1, value);
		}
	},
	_getLayerScaleX: function () {
		if (!this._transparency && MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			var matrix = this._getFilter('DXImageTransform.Microsoft.Matrix');
			return matrix ? matrix[0] : 1;
		} else {
			// check for native support
			var value = !this._transformationSupport || this._bugs.scale ? this._transformations._scaleX : this._transformations.scaleX;

			return MAL.defaultValue(1, value);
		}
	},
	_getLayerScaleY: function () {
		if (!this._transparency && MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			var matrix = this._getFilter('DXImageTransform.Microsoft.Matrix');
			return matrix ? matrix[1] : 1;
		} else {
			// check for native support
			var value = !this._transformationSupport || this._bugs.scale ? this._transformations._scaleY : this._transformations.scaleY;

			return MAL.defaultValue(1, value);
		}
	},

	/**
	 * transform rotation setter
	 *
	 * @private
	 * @param {Number} value
	 */
	_setLayerPreRotate: function (value) {
		this._transformations.preRotate = value + this._suffixes.preRotate;
	},

	_setLayerRotate: function (value) {
		if (this._transformationSupport) {
			// browsers that can handle translate
			this._transformations.rotate = value + this._suffixes.rotate;
		} else if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			// store xywh info
			if (!this._bufferedCSSProperties._rotationInfo) this._bufferedCSSProperties._rotationInfo = {
				x: this.element.offsetLeft,
				y: this.element.offsetTop,
				w: this.element.offsetWidth,
				h: this.element.offsetHeight
			};

			var radian = 0.0174532925,
				angle = value * radian,
				costheta = Math.cos(angle),
				sintheta = Math.sin(angle),
				a = costheta,
				b = sintheta,
				c = -sintheta,
				d = costheta,
				e = 0,
				f = 0;

			// set linear transformation via Matrix Filter
			this._setFilter('DXImageTransform.Microsoft.Matrix', 'M11=' + a + ', M21=' + b + ', M12=' + c + ', M22=' + d + ', SizingMethod="auto expand"', value);

			// apply everything to the DOM so that our offsets are correct
			this._setCSSTextFromCache();

			// bounding box dimensions
			// IE has updated these values based on transform set above
			var wb = this.element.offsetWidth;
			var hb = this.element.offsetHeight;

			// determine how far origin has shifted
			var sx = (wb - this._bufferedCSSProperties._rotationInfo.w) / 2;
			var sy = (hb - this._bufferedCSSProperties._rotationInfo.h) / 2;

			// translation, corrected for origin shift
			// rounding helps, but doesn't eliminate, integer jittering
			this._bufferedCSSProperties.left = Math.round(this._bufferedCSSProperties._rotationInfo.x + e - sx) + 'px';
			this._bufferedCSSProperties.top = Math.round(this._bufferedCSSProperties._rotationInfo.y + f - sy) + 'px';
		}
	},

	/**
	 * transform rotation getter
	 *
	 * @private
	 */
	_getLayerRotate: function () {
		if (this._transformationSupport) {
			// browsers that can handle translate
			return this._transformations.rotate;
		} else if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			var value = this._getFilter('DXImageTransform.Microsoft.Matrix');
			return MAL.isUndefined(value) ? 0 : value;
		}
	},

	/**
	 * transform skew
	 *
	 * @private
	 * @param {Number} value
	 */
	_setLayerSkew: function (value) {
		this._transformations.skew = value + this._suffixes.skew;
	},
	_setLayerSkewX: function (value) {
		this._transformations.skewX = value + this._suffixes.skew;
	},
	_setLayerSkewY: function (value) {
		this._transformations.skewY = value + this._suffixes.skew;
	},

	/**
	 * steps to the next animation in the animation stack
	 *
	 * @private
	 */
	_startNextAnimation: function () {
		var tween = this._tweens.shift();
		if (tween) {
			// we have to do this incase someone runs an addTween right after an addLayer with start transforms
			this._stage.timeouts.addTimeout((function (self, tween) {
				return function () {
					self._signature = new Date().getTime() * Math.random();
					self._applyAnimation.apply(self, [self._signature, tween]);
				};
			}(this, tween)), 0);

			this._started = true;
		} else {
			this._started = false;
		}
	},

	/**
	 * updates the layers cssText based on the current tween thats being applied
	 *
	 * @private
	 * @param {Number} time
	 */
	_update: function (time) {
		if (!this._currentTween) return;

		var tween = this._currentTween,
			finished = time > (tween.startTime + tween.duration),
			percent = (time - tween.startTime) / tween.duration;

		// in case we are late
		if (percent > 1) percent = 1;

		// apply ease if provided
		if (MAL.isArray(tween.easing)) {
			percent = MAL.cubicBezierWithPercent(percent, tween.easing[0], tween.easing[1], tween.easing[2], tween.easing[3]);
		} else {
			percent = tween.easing(percent);
		}

		// apply changes to properties
		for (var property in tween.values.start) {
			var value = MAL.Util.shortenFloat(tween.values.start[property] + (tween.values.delta[property] * percent), 5);
			this.setProperty(property, value, true);
		}

		// apply everything to the DOM
		this._setCSSTextFromCache();

		// check if animation is complete
		if (finished) {
			this._currentTween = null;
			this._clearTemporaryCachedProperties();

			// fire persistant callback
			if (this._persistantCallback) this._persistantCallback(this);

			// fire callback
			if (tween.callback) tween.callback(this);

			this.dispatchEvent('animation-end');
			this._startNextAnimation();
		}

	},

	/**
	 * constructs and adds an animation to the tween stack, also handles the delay
	 *
	 * @private
	 * @param {Object} options
	 */
	_applyAnimation: function (signature, options) {
		// if signature missmatch, animation has been overridden
		if (signature !== this._signature) return;

		// we don't really need to do it this way but to keep the core logic consistant...
		if (options.delay) {
			this._stage.timeouts.addTimeout((function (self, signature, options) {
				return function () {
					// remove delay
					options.delay = 0;

					// apply animation with start values
					self._applyAnimation.apply(self, [signature, options]);
				};
			}(this, signature, options)), options.delay * 1000);

			return;
		}

		// checks if we have transformation support
		this._transformationSupport = !!MAL.Environment.getCSSPropertyName('transform') && !this._bugs.transformations;

		// force hardware acceleration (webkit is the only browser that supports a Z axis)
		if (MAL.Environment.has3d && !this._bugs.transform3d && this._features.backfacePerformanceFix) this._transformations.translateZ = '0px';

		// set start with props
		if (options.start) this._setStartProperties(options.start);

		// generate deltas
		var tween = {};
		tween.duration = options.duration * 1000 || 1000;
		tween.startTime = new Date().getTime();
		if (options.easing) {
			tween.easing = MAL.isString(options.easing) && this._eases[options.easing] ? this._eases[options.easing] : options.easing;
		} else {
			tween.easing = this._eases.linear;
		}

		tween.values = {
			start: {},
			delta: {},
			finish: options.finish
		};
		tween.callback = options.callback;

		for (var property in options.finish) {
			tween.values.start[property] = MAL.parseNumber(this.getProperty(property));
			tween.values.delta[property] = MAL.delta(tween.values.start[property], options.finish[property]);
		}

		this._currentTween = tween;
		this._animating = true;

		// increment iteration count
		this.iteration++;

		this.dispatchEvent('animation-start');
	},

	/**
	 * sets the layers start properties and updates the cssText
	 *
	 * @private
	 */
	_setStartProperties: function (properties) {
		this.setProperties(properties);
		this._setCSSTextFromCache();
	},

	/**
	 * set multiple properties on a layer
	 *
	 *     layer.setProperties({
	 *      opacity: .5,
	 *      x: 100,
	 *      y: 100,
	 *      'background-color': '#000000'
	 *     });
	 *
	 * refer to {@link MAL.tween.TransformObject} for possible properties
	 *
	 * @param {Object} properties
	 */
	setProperties: function (properties, dontApplyToDOM) {
		for (var property in properties)
			this.setProperty(property, properties[property], true);

		// regenerate the transform string
		if (!dontApplyToDOM) {
			if (this._transformationSupport) this._css('transform', this._constructTransformationString());
			this._setCSSTextFromCache();
		}

		return this;
	},

	/**
	 * set a layers property
	 *
	 *     layer.setProperty('rotate', 45);
	 *
	 * refer to {@link MAL.tween.TransformObject} for possible properties
	 *
	 * @param {String} name
	 * @param {String/Number} value
	 */
	setProperty: function (name, value, _dontApplyToDOM) {
		// method is used to detect if we need any additional setProperty overrides
		var method = MAL.toCamelCaseWithPrefix('_setLayer', name);

		if (this._transformationPropertyDictionary[name]) {
			// set advanced property
			if (this[method]) {
				this[method](value);
			} else {
				var transform = this._transformationPropertyDictionary[name];
				this._transformations[transform.name] = MAL.defaultValue(transform.value, value) + (this._suffixes[name] || '');
			}
		} else {
			// set basic styling property
			this._css(name, value);
		}

		// regenerate the transform string
		if (!_dontApplyToDOM) {
			if (this._transformationSupport) this._css('transform', this._constructTransformationString());
			this._setCSSTextFromCache();
		}

		return this;
	},

	/**
	 * get a layers property value
	 *
	 *     layer.getProperty('rotate');
	 *
	 * @param {String} name
	 */
	getProperty: function (name) {
		var method = MAL.toCamelCaseWithPrefix('_getLayer', name),
			value;

		if (this._transformationPropertyDictionary[name]) {
			// get transform property
			value = this[method] ? this[method]() : this._transformations[this._transformationPropertyDictionary[name].name];
		} else {
			// get basic styling property
			value = this._css(name);
		}

		return this._parsePropertyValue(value);
	},

	/**
	 * get layers current position after transformations and positioning
	 *
	 *     layer.getPosition();
	 *
	 * @return {Object} {x: Float, y: Float}
	 */
	getPosition: function () {
		return {
			x: this.cssGetComputedProperty('left') + (this._transformationSupport ? parseFloat(this._getLayerX()) : 0),
			y: this.cssGetComputedProperty('top') + (this._transformationSupport ? parseFloat(this._getLayerY()) : 0)
		};
	},

	/**
	 * sets and gets css property values
	 *
	 * @private
	 * @param {String} property name of the property
	 * @param {String/Number} value value of property
	 * @param {Object} properties set of properties
	 */
	_css: function () {
		if (!MAL.isUndefined(arguments[0]) && !MAL.isUndefined(arguments[1])) {
			// set specific value
			this._cssSetProperty(arguments[0], arguments[1]);
		} else if (MAL.isString(arguments[0])) {
			// return specific value
			return this._cssGetProperty(arguments[0]);
		} else if (MAL.isObject(arguments[0])) {
			// set multiple properties
			this._cssSetProperties(arguments[0]);
		}
	},

	/**
	 * clears temporary cached properties (tween buffer properties)
	 *
	 * @private
	 */
	_clearTemporaryCachedProperties: function () {
		for (var property in this._bufferedCSSProperties) {
			if (property.substr(0,1) === '_') delete this._bufferedCSSProperties[property];
		}
	},

	/**
	 * sets the cssText of the layer (this is very IMPORTANT)
	 *
	 * @private
	 */
	_setCSSTextFromCache: function () {
		var string = '';

		// apply filters or transforms
		if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			this._setFilterString();
		} else if (this._transformationSupport) {
			this._css('transform', this._constructTransformationString());
		}

		for (var property in this._bufferedCSSProperties) {
			// do not add things that are strictly used for computing (filters)
			if (property.substr(0,1) !== '_') {
				string += property + ': ' + this._bufferedCSSProperties[property] + '; ';
			}
		}

		this.element.style.cssText = string;
	},

	/**
	 * produces a transformation property string from the _transformations object
	 *
	 * @private
	 */
	_constructTransformationString: function () {
		var string = '';

		this._transformationFunctionOrder.forEach(function (transformation) {
			if (transformation.substr(0,1) === '_' || MAL.isUndefined(this._transformations[transformation])) return;

			// look up alias name
			var name = this._transformationFunctionAlias[transformation] ? this._transformationFunctionAlias[transformation] : transformation;

			string += ' ' + name + '(' + this._transformations[transformation] + ')';
		}, this);

		return string.substr(1);
	},

	/**
	 * produces a transformation property object from a transform string
	 *
	 * @private
	 */
	_constructTransformationObject: function (string) {
		var match,
			transformation = {},
			pattern = new RegExp(/([a-z0-9]+)\((.+?)\)/gi);

		while ((match = pattern.exec(string))) transformation[match[1]] = match[2];

		return transformation;
	},

	/**
	 * sets a filter (IE <9 only method)
	 *
	 * @private
	 * @param {String} name filter name
	 * @param {String} value filter value string
	 * @param {Number} raw filter value float
	 */
	_setFilter: function (name, value, raw) {
		// we cache the raw filter so we don't have to keep reconverting it
		this._filtersCache[name] = raw;

		// destination value string
		this._filters[name] = value;
	},

	/**
	 * returns a filters value float (IE <9 only method)
	 *
	 * @private
	 * @param {String} name filter name
	 */
	_getFilter: function (name) {
		// returns actual integer that you would put into cssSet...
		return this._filtersCache[name];
	},

	/**
	 * constructs and sets the css filter property string out of the applied filters (IE <9 only method)
	 *
	 * @private
	 */
	_setFilterString: function () {
		var string = '';

		// for performance sake use a normal for loop
		for (var i = 0; i < this._filterOrder.length; i++)
			if (!MAL.isUndefined(this._filters[this._filterOrder[i]])) string += 'progid:' + this._filterOrder[i] + '(' + this._filters[this._filterOrder[i]] + ') ';

		if (string) this._bufferedCSSProperties.filter = string.slice(0,-1);
	},

	/**
	 * sets a single css property
	 *
	 * @private
	 * @param {String} property name of the property
	 * @param {String/Number} value value of property
	 */
	_cssSetProperty: function (property, value) {
		// method is used to detect if we need any additional css overrides
		var method = MAL.toCamelCaseWithPrefix('_cssSet', property);

		// add browser prefix to properties that start with a dash
		var propertyName = MAL.Environment.getCSSPropertyName(property, true);

		if (this[method]) {
			// use override
			var override = this[method](value);

			// do we need to save it
			if (override) this._bufferedCSSProperties[override.name] = override.value;
		} else if (MAL.isNull(value)) {
			delete this._bufferedCSSProperties[propertyName];
		} else {
			// add suffix
			if (MAL.isNumber(value) && this._suffixes[property]) value = value + this._suffixes[property];

			this._bufferedCSSProperties[propertyName] = value;
		}
	},

	/**
	 * sets a set of css properties
	 *
	 * @private
	 * @param {Object} properties set of properties
	 */
	_cssSetProperties: function (properties) {
		for (var property in properties) {
			this._cssSetProperty(property, properties[property]);
		}
	},

	/**
	 * gets the value of a css property
	 *
	 * @private
	 * @param {String} property name of css property
	 */
	_cssGetProperty: function (property, raw) {
		// method is used to detect if we need any additional css overrides
		var method = MAL.toCamelCaseWithPrefix('_cssGet', property);

		// add browser prefix to properties that start with a dash
		var propertyName = MAL.Environment.getCSSPropertyName(property, true);

		if (this[method]) {
			// use override
			return this[method](property);
		} else {
			// get basic style property
			return this.element.style[propertyName] || this.cssGetComputedProperty(propertyName, raw);
		}
	},

	/**
	 * overridden opacity setter
	 *
	 * @private
	 * @param {Number} value
	 */
	_cssSetOpacity: function (value) {
		if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			// custom alpha filter for older IE versions
			this._setFilter('DXImageTransform.Microsoft.Alpha', 'opacity=' + (value * 100), value);
		} else {
			// browser supports standard alpha
			return {name: 'opacity', value: value};
		}
	},

	/**
	 * overridden opacity getter
	 *
	 * @private
	 */
	_cssGetOpacity: function () {
		if (MAL.Environment.browser === 'msie' && MAL.Environment.documentMode < 9) {
			// custom alpha filter for older IE versions
			var value = this._getFilter('DXImageTransform.Microsoft.Alpha');
			return MAL.isUndefined(value) ? 1 : value;
		} else {
			// browser supports standard alpha
			return this._bufferedCSSProperties.opacity || this.element.style.opacity || 1;
		}
	},

	/**
	 * helper for some of our string values
	 *
	 * @private
	 */
	_scalePercentOrAdd: function (current, value) {
		if (value.indexOf('%') !== -1) {
			current *= parseFloat(value)/100;
		} else {
			current += parseFloat(value);
		}
		return current;
	}
}));
MAL.tween.JSLayerProxy = new MAL.tween.LayerProxy(MAL.tween.JSLayer);